/*****************************************************************************
 * divx_ui.c: DivX Parameters editor
 *****************************************************************************
 * $Id: divx_ui.c,v 1.66 2004/12/05 18:27:59 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <signal.h>
#include <unistd.h>
#include <string.h>
#include <errno.h>
#include <math.h>
#include <sys/time.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/Xaw/XawInit.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Scrollbar.h>
#include <X11/Xaw/Paned.h>
#include <X11/Xaw/Command.h>
#include <X11/Xaw/AsciiText.h>
#include <pthread.h>
#include <sys/wait.h>

#ifdef HAVE_FFMPEG
# include <ffmpeg/avcodec.h>
#endif

#include "config.h"

#include "divx.h"
#include "strtab.h"
#include "toolbox.h"
#include "xdtv.h"
#include "grab.h"
#include "x11.h"
#include "popup_ui.h"
#include "file_sel.h"

#ifdef HAVE_XPM
# include <X11/xpm.h>
#endif

#ifdef HAVE_XPM
# include "xdtv.xpm"
#ifdef HAVE_PIXMAPS
# include "../pixmaps/stream_streamingoptions.xpm"
# include "../pixmaps/stream_choosehttpport.xpm"
# include "../pixmaps/stream_openvideooptions.xpm"
# include "../pixmaps/stream_openaudiooptions.xpm"
# include "../pixmaps/stream_openvideooptions_off.xpm"
# include "../pixmaps/stream_openaudiooptions_off.xpm"
# include "../pixmaps/stream_openmainoptions.xpm"

# include "../pixmaps/audio_audiooptions.xpm"
# include "../pixmaps/audio_openvideooptions.xpm"
# include "../pixmaps/audio_openstreamingoptions.xpm"
# include "../pixmaps/audio_openmainoptions.xpm"

# include "../pixmaps/video_choosecodec.xpm"
# include "../pixmaps/video_videooptions.xpm"
# include "../pixmaps/video_otheroptions.xpm"
# include "../pixmaps/video_pathavi.xpm"
# include "../pixmaps/video_openaudiooptions.xpm"
# include "../pixmaps/video_openstreamingoptions.xpm"
# include "../pixmaps/video_openmainoptions.xpm"

# include "../pixmaps/video_applysave.xpm"
# include "../pixmaps/video_browse.xpm"
# include "../pixmaps/video_schedule.xpm"
# include "../pixmaps/video_preview.xpm"

# include "../pixmaps/audio_applysave.xpm"

# include "../pixmaps/stream_startstreaming.xpm"
# include "../pixmaps/stream_startstreaming_off.xpm"
# include "../pixmaps/stream_applysave.xpm"
# include "../pixmaps/stream_close.xpm"
#endif
#endif

/*-------------------------------------------------------------------------*/
extern void save_cb (Widget widget, XtPointer clientdata, XtPointer call_data);
extern void SetBgAction (Widget, XEvent *, String *, Cardinal *);
extern void RecordMovieAction (Widget , XEvent * , String * , Cardinal *);
extern void action_cb (Widget widget, XtPointer clientdata, XtPointer call_data);
extern void OptionsAction (Widget, XEvent *, String *, Cardinal *);
extern Widget app_shell, divx_shell, divxaudio_shell, streamparams_shell, c_record;
extern Display *dpy;
extern Atom wm_protocols[2];
extern int debug;
extern int bttvfd,vbifd;
extern int xawpopup;
extern char lastfilename[256];

extern divx_t divx;

static int w_modul = WRECPAD;
static int h_modul = HRECPAD;

#ifdef HAVE_XPM
  Pixmap icon_pixmap;
  Pixmap icon_shapemask;
#endif

static Widget divx_shell, divx_paned;
static Widget divxaudio_shell, divxaudio_paned;
static Widget streamparams_shell, streamparams_paned;

static Widget c_divxcodec, c_wmodul, c_hmodul;
static Widget l_divxwidth,        s_divxwidth;
static Widget l_divxheight,       s_divxheight;
static Widget l_divxbitrate,      s_divxbitrate;
static Widget l_divxfps, s_divxfps;
static Widget l_divxmaxgap, s_divxmaxgap;
static Widget c_divxaudiofrgmts, s_divxaudiofrgmts;
static Widget c_divxaudiosizefrgmt, s_divxaudiosizefrgmt;
static Widget l_divxdelay, s_divxdelay;
static Widget c_divxquality;
static Widget l_divxminquantizer, s_divxminquantizer;
static Widget l_divxmaxquantizer, s_divxmaxquantizer;
static Widget c_divxaudiobufsize, s_divxaudiobufsize;
static Widget t_divxfilepath;
#ifdef HAVE_LAME
static Widget c_compressaudio;
static Widget c_stereo;
static Widget c_mp3_vbr;
static Widget p_divxmp3vbrquality,   l_divxmp3vbrquality,   s_divxmp3vbrquality;
static Widget p_divxmp3bitrate,   l_divxmp3bitrate,   s_divxmp3bitrate;
static Widget p_divxmp3quality,   l_divxmp3quality,   s_divxmp3quality;
#endif // HAVE_LAME
static Widget c_displayframes;
static Widget c_sub;
static Widget c_chg;
static Widget c_schedule;
static Widget c_player;
static Widget c_path;
static Widget c_path_selector;
static Widget c_openmain;
static Widget c_stropts;
static Widget c_divxaudio;
static Widget c_streamopen;
static Widget c_divxopen;
static Widget c_divxvideocodec;
static Widget c_divxsize;
static Widget c_divxdisplay;
static Widget c_divxaudioopen;
static Widget c_save;
static Widget c_audiosave;
static Widget c_streamparamssave;
static Widget c_streamparamsclose;

Widget c_streaming;
static Widget c_stream_onoff;
static Widget t_http_port_str;
static Widget c_http_port;
static Widget c_divxaudioopen;
static Widget c_divxopen;
char * http_port_str;
char http_port[256];

enum {
  DIVX_CODEC = 10,
  DIVX_HMODUL,
  DIVX_WMODUL,
  DIVX_QUALITY,
  DIVX_COMPRESS_AUDIO,
  DIVX_STEREO,
  DIVX_MP3_VBR,
  DIVX_DISPLAY_FRAMES,
  DIVX_SUB,
  DIVX_CHG,
  DIVX_STREAM_ONOFF
};

static struct STRTAB on_off_list[] = {
  {1, "On"},
  {0, "Off"},
  {-1, NULL}
};

static struct STRTAB stereo_list[] = {
  {1, "Stereo"},
  {0, "Mono"},
  {-1, NULL}
};

static struct STRTAB mp3_vbr_list[] = {
  {1, "VBR (Variable)"},
  {0, "CBR (Constant)"},
  {-1, NULL}
};

static struct STRTAB codec_list[] = {
#ifdef HAVE_DIVX4LINUX
  {DIVX4LINUX, "DivX"},
#endif
#ifdef HAVE_FFMPEG
  {FFMPEG_MPEG4, "FFMPEG - Mpeg4"},
  {FFMPEG_MPEG1, "FFMPEG - Mpeg1Video"},
#if LIBAVCODEC_BUILD >= 4734
  {FFMPEG_FFVHUFF, "FFMPEG - Ffvhuff"},
#endif
  {FFMPEG_HUFFYUV, "FFMPEG - Huffyuv"},
#endif
#ifdef HAVE_XVID
  {XVID, "XviD"},
#endif
  {UNCOMPRESSED, "UnCompressed"},
  {-1, NULL}
};

static struct STRTAB modulo_list[] = {
  {1,  " 1"},
  {2,  " 2"},
  {4,  " 4"},
  {8,  " 8"},
  {16, "16"},
  {32, "32"},
  {-1, NULL}
};

#ifdef HAVE_DIVX4LINUX
static struct STRTAB divx4linux_quality_list[] = {
  {1,  "Fastest"},
  {2,  "Fast"},
  {3,  "Medium"},
  {4,  "Slow"},
  {5,  "Slowest"},
  {-1, NULL}
};
#endif

#ifdef HAVE_FFMPEG
static struct STRTAB ffmpeg_quality_list[] = {
  {0,  "none (very low quality)"},
  {2,  "log (low quality)"},
  {3,  "phods (low quality)"},
  {4,  "EPZS (medium quality)"},
  {1,  "full (slow)"},
  {5,  "X1 (experimental)"},
  {-1, NULL}
};
#endif

#ifdef HAVE_XVID
static struct STRTAB xvid_quality_list[] = {
  {0,  "H623, No ME"},
  {1,  "MPEG, Low"},
  {2,  "H623, Low"},
  {3,  "H623, Medium"},
  {4,  "H623, High"},
  {5,  "H623, Very High"},
  {6,  "H623, Ultra High"},
  {-1, NULL}
};
#endif

static struct STRTAB uncompressed_quality_list[] = {
  {0, "default"},
  {-1, NULL}
};

static struct STRTAB *quality_list = NULL;

/*-------------------------------------------------------------------------*/
#define FIX_RIGHT_TOP       \
    XtNleft,XawChainRight,  \
    XtNright,XawChainRight, \
    XtNtop,XawChainTop,     \
    XtNbottom,XawChainTop

#define DIVX_WIDTH_RANGE                768.0
#define DIVX_HEIGHT_RANGE               576.0
#define DIVX_BITRATE_RANGE            10000.0
#define DIVX_FPS_RANGE                   30.0
#define DIVX_MAXGAP_RANGE               500.0
#define DIVX_AUDIOFRGMTS_RANGE          100.0
#define DIVX_AUDIOSIZEFRGMT_RANGE     65535.0
#define DIVX_DELAY_RANGE                500.0
#define DIVX_QUALITY_RANGE                5.0
#define DIVX_MINQUANTIZER_RANGE          64.0
#define DIVX_MAXQUANTIZER_RANGE          64.0
#define AUDIO_BUFFERSIZE_RANGE        65535.0
#ifdef HAVE_LAME
#  define MP3_BITRATE_RANGE             512.0
#  define MP3_QUALITY_RANGE               9.0
#  define MP3_VBR_QUALITY_RANGE           9.0
#endif // HAVE_LAME

/*-------------------------------------------------------------------------*/

struct CALL_ACTION call_options_5 = { 0, "Options", {NULL} };

/*--------------------------------------------------------------------------*/

/* for some libc (SunOS 5.8) printf("%s",NULL) segfaults.... */
static char* nn(char * s) {
  return s ?s : "(null)";
}


struct STRTAB *
set_quality_list(int codec)
{
  switch (codec)
    {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      return divx4linux_quality_list;
      break;
#endif
#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
    case FFMPEG_MPEG1:
#if LIBAVCODEC_BUILD >= 4734
    case FFMPEG_FFVHUFF:     
#endif
    case FFMPEG_HUFFYUV:
      return ffmpeg_quality_list;
      break;
#endif
#ifdef HAVE_XVID
    case XVID:
      return xvid_quality_list;
      break;
#endif
    case UNCOMPRESSED:
      return uncompressed_quality_list;
      break;
    default:
      return NULL;
      break;
    }
}

/*-------------------------------------------------------------------------*/

struct CALL_ACTION call_recordmovie = { 0, "RecordMovie", {NULL} };

/*-------------------------------------------------------------------------*/

// Code from mwxcd http://member.nifty.ne.jp/nmaruichi/
// the purpose is to have in Xaw a file selector to choose the path easily

static void srcSelectFinish(const char *dirName, const char *fileName)
{
	char Path[8192];
	if (fileName[0] != '\0')
	  sprintf(Path, "%s/%s", dirName, fileName);
	else
	  sprintf(Path, "%s", dirName);
	if(t_divxfilepath)
	  XtVaSetValues (t_divxfilepath, XtNstring, Path, NULL);
}


static void srcFileSelector(Widget w, XtPointer client_data, XtPointer call_data)
{
	char dirName[4096];
	char fileName[1024];
	extern selectorPanes filePickupPaneInfo[];
	extern selectorPanes fileSortPaneInfo[];
	static selectorPanes *defPickup = &filePickupPaneInfo[1];
	static selectorPanes *defSort = &fileSortPaneInfo[1];
	dirName[0] = '\0';
	fileName[0] = '\0';
	fileCheck(divx.filename, sizeof(dirName), dirName, sizeof(fileName), fileName);
	fileSelectorCall(w, dirName, defPickup, defSort, srcSelectFinish);
}

/*-------------------------------------------------------------------------*/

void wfc(int i)
{
        waitpid(-1, NULL, WNOHANG);
        signal(SIGCHLD, wfc);
}

void execute(Widget w, XtPointer client_data, XtPointer call_data)
{
  char *cmd = (char *)client_data;
  pthread_t thread;
  pthread_create (&thread, NULL, (void *(*)(void *)) system, cmd);
}

static void
close_divx_win (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  XtCallActionProc (divx_shell, "DivX", NULL, NULL, 0);
}

static void
close_divxaudio_win (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  XtCallActionProc (divxaudio_shell, "DivXAudio", NULL, NULL, 0);
}

static void
close_streamparamswin (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  XtCallActionProc (streamparams_shell, "StreamParams", NULL, NULL, 0);
}

void
DivXAction (Widget widget, XEvent * event,
              String * params, Cardinal * num_params)
{
  static int mapped = 0, first = 1;

  if (event && event->type == ClientMessage)
    {
      if (event->xclient.data.l[0] == wm_protocols[1])
        {
          if (debug)
            fprintf (stderr, "Config: wm_save_yourself\n");
          XSetCommand (XtDisplay (divx_shell), XtWindow (divx_shell), NULL,
                       0);
          return;
        }
    }

  if (divx_shell == NULL)
    {
      fprintf(stderr, "DivXAction: ooops: divx_shell is NULL\n");
      if (xawpopup)
        PopupMessage("DivXAction:\nooops: divx_shell is NULL");
      return;
    }

  if (is_divx_in_progress())
    {
      fprintf(stderr, "DivXAction: During a record, you cann't reach the Video Recorder Options window\n");
      if (xawpopup)
        PopupMessage("The Video Recorder Options Window\ncan't be reached during a record\n");
      return;
    }

  if (mapped)
    {
      XtPopdown (divx_shell);
      mapped = 0;
      if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     :     (Off)          R", NULL);             
    }
  else
    {
      XtPopup (divx_shell, XtGrabNone);
      mapped = 1;
      if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     : (Options)      Alt+R", NULL);
      if (first)
        {
          XSetWMProtocols (XtDisplay (divx_shell), XtWindow (divx_shell),
                           wm_protocols, 2);
          first = 0;
        }
    }
}

void
DivXAudioAction (Widget widget, XEvent * event,
              String * params, Cardinal * num_params)
{
  static int mapped = 0, first = 1;

  if (event && event->type == ClientMessage)
    {
      if (event->xclient.data.l[0] == wm_protocols[1])
        {
          if (debug)
            fprintf (stderr, "Config: wm_save_yourself\n");
          XSetCommand (XtDisplay (divxaudio_shell), XtWindow (divxaudio_shell), NULL,
                       0);
          return;
        }
    }

  if (divxaudio_shell == NULL)
    {
      fprintf(stderr, "DivXAudioAction: ooops: divxaudio_shell is NULL\n");
      if (xawpopup)
        PopupMessage("DivXAudioAction:\nooops: divxaudio_shell is NULL");
      return;
    }

  if (is_divx_in_progress())
    {
      fprintf(stderr, "DivXAudioAction: During a record, you cann't reach the Audio Recorder Options window\n");
      if (xawpopup)
        PopupMessage("The Audio Recorder Options Window\ncan't be reached during a record\n");
      return;
    }

  if (mapped)
    {
      XtPopdown (divxaudio_shell);
      mapped = 0;
      if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     :     (Off)          R", NULL);      
    }
  else
    {
      XtPopup (divxaudio_shell, XtGrabNone);
      mapped = 1;
      if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     : (Options)     Ctrl+R", NULL);
      if (first)
        {
          XSetWMProtocols (XtDisplay (divxaudio_shell), XtWindow (divxaudio_shell),
                           wm_protocols, 2);
          first = 0;
        }
    }
}

void
StreamParamsAction (Widget widget, XEvent * event, String * params, Cardinal * num_params)
{
  static int mapped = 0, first = 1;

  if (event && event->type == ClientMessage)
    {
      if (event->xclient.data.l[0] == wm_protocols[1])
        {
          if (debug)
            fprintf (stderr, "Config: wm_save_yourself\n");
          XSetCommand (XtDisplay (streamparams_shell), XtWindow (streamparams_shell), NULL, 0);
          return;
        }
    }

  if (streamparams_shell == NULL)
    {
      fprintf(stderr, "StreamParamsAction: ooops: streamparams_shell is NULL\n");
            if (xawpopup)
              PopupMessage("StreamParamsAction:\nooops: streamparams_shell is NULL\n");
      return;
    }

  if (mapped)
    {
      XtPopdown (streamparams_shell);
      mapped = 0;
      if (is_divx_in_progress() && c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     :      (On)          R", NULL); 
      else if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     :     (Off)          R", NULL);      
    }
  else
    {
      XtPopup (streamparams_shell, XtGrabNone);
      mapped = 1;
      if (c_record)
        XtVaSetValues (c_record, XtNlabel, "Record Movie     : (Options)    Shift+R", NULL);
      if (first)
        {
          XSetWMProtocols (XtDisplay (streamparams_shell), XtWindow (streamparams_shell), wm_protocols, 2);
          first = 0;
        }
    }
}

static void
divx_menu_cb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  int cd = (int) clientdata;
  int j;
  char label[64];
  int value;
  char *tok;
  int argc;

  switch (cd)
    {
      /* DivX Parameters window entries */
    case DIVX_CODEC:
      if (-1 != (j = popup_menu (widget, "Codec", codec_list)))
        {
          divx.codec = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Codec",
                   int_to_str (divx.codec, codec_list));
          XtVaSetValues (c_divxcodec, XtNlabel, label, NULL);
          quality_list = set_quality_list(divx.codec);
        }
      break;

    case DIVX_WMODUL:
      if (-1 != (j = popup_menu (widget, "W-Modul", modulo_list)))
        {
          w_modul = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s: %5d", "W-Modul", w_modul);
          XtVaSetValues (c_wmodul, XtNlabel, label, NULL);

          // Recompute new width value
          value = divx.width / w_modul;
          value = value * w_modul;
          divx.width = value;
          if (s_divxwidth)
            set_float (s_divxwidth, XtNtopOfThumb, (float)value/DIVX_WIDTH_RANGE);

          sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Width", divx.width);
          if (l_divxwidth)
            XtVaSetValues(l_divxwidth, XtNlabel, label, NULL);
        }
      break;

    case DIVX_HMODUL:
      if (-1 != (j = popup_menu (widget, "H-Modul", modulo_list)))
        {
          h_modul = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s: %5d", "H-Modul", h_modul);
          XtVaSetValues (c_hmodul, XtNlabel, label, NULL);

          // Recompute new height value
          value = divx.height / h_modul;
          value = divx.height * h_modul;
          divx.height = value;
          if (s_divxheight)
            set_float (s_divxheight, XtNtopOfThumb, (float)value/DIVX_HEIGHT_RANGE);

          sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Height", divx.height);
          if (l_divxheight)
            XtVaSetValues(l_divxheight, XtNlabel, label, NULL);
        }
      break;

    case DIVX_QUALITY:
      if (-1 != (j = popup_menu (widget, "Quality", quality_list)))
        {
          divx.quality = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Quality",
                   int_to_str (divx.quality, quality_list));
          XtVaSetValues (c_divxquality, XtNlabel, label, NULL);
        }
      break;

#ifdef HAVE_LAME
    case DIVX_COMPRESS_AUDIO:
      if (-1 != (j = popup_menu (widget, "Compress Audio", on_off_list)))
        {
          divx.compress_audio = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Compress Audio",
                   int_to_str (divx.compress_audio, on_off_list));
          XtVaSetValues (c_compressaudio, XtNlabel, label, NULL);
          if (j == 0)
            {
              XtVaSetValues(l_divxmp3bitrate, XtNsensitive, False, NULL);
              XtVaSetValues(s_divxmp3bitrate, XtNsensitive, False, NULL);
              XtVaSetValues(l_divxmp3quality, XtNsensitive, False, NULL);
              XtVaSetValues(s_divxmp3quality, XtNsensitive, False, NULL);
            }
          else
            {
              XtVaSetValues(l_divxmp3bitrate, XtNsensitive, True, NULL);
              XtVaSetValues(s_divxmp3bitrate, XtNsensitive, True, NULL);
              XtVaSetValues(l_divxmp3quality, XtNsensitive, True, NULL);
              XtVaSetValues(s_divxmp3quality, XtNsensitive, True, NULL);
            }
        }
      break;

    case DIVX_STEREO:
      if (-1 != (j = popup_menu (widget, "Stereo Mode", stereo_list)))
        {
          divx.stereo = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Stereo Mode",
                   int_to_str (divx.stereo, stereo_list));
          XtVaSetValues (c_stereo, XtNlabel, label, NULL);
        }
      break;

   case DIVX_MP3_VBR:
      if (-1 != (j = popup_menu (widget, "Bitrate Mode", mp3_vbr_list)))
        {
          divx.mp3_vbr = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Bitrate Mode",
                   int_to_str (divx.mp3_vbr, mp3_vbr_list));
          XtVaSetValues (c_mp3_vbr, XtNlabel, label, NULL);
	  if (int_to_str (divx.compress_audio, on_off_list) == "On")
	  {
          	if (j == 1)
            		{
              		XtVaSetValues(l_divxmp3bitrate, XtNsensitive, False, NULL);
              		XtVaSetValues(s_divxmp3bitrate, XtNsensitive, False, NULL);
              		XtVaSetValues(l_divxmp3vbrquality, XtNsensitive, True, NULL);
              		XtVaSetValues(s_divxmp3vbrquality, XtNsensitive, True, NULL);
            		}
          	else
            		{
              		XtVaSetValues(l_divxmp3bitrate, XtNsensitive, True, NULL);
              		XtVaSetValues(s_divxmp3bitrate, XtNsensitive, True, NULL);
              		XtVaSetValues(l_divxmp3vbrquality, XtNsensitive, False, NULL);
              		XtVaSetValues(s_divxmp3vbrquality, XtNsensitive, False, NULL);
            		}
	  }
	  else
	  {
              XtVaSetValues(l_divxmp3bitrate, XtNsensitive, False, NULL);
              XtVaSetValues(s_divxmp3bitrate, XtNsensitive, False, NULL);
              XtVaSetValues(l_divxmp3vbrquality, XtNsensitive, False, NULL);
              XtVaSetValues(s_divxmp3vbrquality, XtNsensitive, False, NULL);
	  }

        }
      break;

#endif

    case DIVX_DISPLAY_FRAMES:
      if (-1 != (j = popup_menu (widget, "Display Frames", on_off_list)))
        {
          divx.display_frames = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Display Frames",
                   int_to_str (divx.display_frames, on_off_list));
          XtVaSetValues (c_displayframes, XtNlabel, label, NULL);
        }
      break;

    case DIVX_SUB:
      if (-1 != (j = popup_menu (widget, "Record Subtitles", on_off_list)))
        {
          divx.sub = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Record Subtitles",
                   int_to_str (divx.sub, on_off_list));
          XtVaSetValues (c_sub, XtNlabel, label, NULL);
        }
      break;

    case DIVX_CHG:
      if (-1 != (j = popup_menu (widget, "Change Channel", on_off_list)))
        {
          divx.chg = j;
          sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Change Channel",
                   int_to_str (divx.chg, on_off_list));
          XtVaSetValues (c_chg, XtNlabel, label, NULL);
        }
      break;

     case DIVX_STREAM_ONOFF:
      if (-1 != (j = popup_menu (widget, "Streaming Server Activation", on_off_list)))
        {
          if (!is_divx_in_progress())
	  {
	    divx.stream = j;
            sprintf (label, "%-" LABEL_WIDTH3 "s:     %s  ", "Streaming Server Activation",
                   int_to_str (divx.stream, on_off_list));
            XtVaSetValues (c_stream_onoff, XtNlabel, label, NULL);
          }

	  if (debug)
	    fprintf(stderr,"Streaming Server Activation: %d\n",divx.stream);

          if (!divx.stream)
          {
	    XtVaSetValues(t_http_port_str, XtNsensitive, False, NULL);
	    XtVaSetValues(c_streaming, XtNsensitive, False, NULL);
	    XtVaSetValues(c_divxaudioopen, XtNsensitive, False, NULL);
	    XtVaSetValues(c_divxopen, XtNsensitive, False, NULL);
            tok = "gray95";
	    argc = 1;
	    SetBgAction (t_http_port_str, NULL, &tok, &argc);
#ifdef HAVE_XPM
#ifdef HAVE_PIXMAPS
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_startstreaming_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_streaming, XtNbitmap, icon_pixmap, NULL);
  
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openvideooptions_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_divxopen, XtNbitmap, icon_pixmap, NULL); 
    
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openaudiooptions_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_divxaudioopen, XtNbitmap, icon_pixmap, NULL); 
#endif
#endif
          }
          else
          {
            XtVaSetValues(t_http_port_str, XtNsensitive, True, NULL);
	    XtVaSetValues(c_streaming, XtNsensitive, True, NULL);
	    XtVaSetValues(c_divxaudioopen, XtNsensitive, True, NULL);
	    XtVaSetValues(c_divxopen, XtNsensitive, True, NULL);
            tok = "white";
	    argc = 1;
	    SetBgAction (t_http_port_str, NULL, &tok, &argc);
#ifdef HAVE_XPM
#ifdef HAVE_PIXMAPS
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_startstreaming_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_streaming, XtNbitmap, icon_pixmap, NULL);
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openvideooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_divxopen, XtNbitmap, icon_pixmap, NULL); 
    
              XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openaudiooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
              XtVaSetValues(c_divxaudioopen, XtNbitmap, icon_pixmap, NULL);
#endif
#endif
          }
        }
      break;

    default:
      break;
    }
}

static void
divx_jump_scb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  char *name;
  int value = 0;
  float data;
  char label[64];

  data = *(float *) call_data;
  name = XtName (XtParent (widget));
#if 0
  fprintf (stderr, "jump to %f (%s/%d)\n", *(float *) call_data, name, value);
#endif

  if (0 == strcmp(name, "divxwidth"))
    {
      value = (int) rint(data * DIVX_WIDTH_RANGE);
      if (value == 0)
        value = w_modul; // minimun acceptable value

      // make sure width is divisible by w_modul
      value /= w_modul;
      value *= w_modul;
      divx.width = value;

      if (s_divxwidth)
        set_float (s_divxwidth, XtNtopOfThumb, (float)value/DIVX_WIDTH_RANGE);

      sprintf (label, "%-" LABEL_WIDTH3 "s: %5d", "Width", divx.width);
      //sprintf(label, "Width: %3d", divx.width);
      if (l_divxwidth)
        XtVaSetValues(l_divxwidth, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxheight"))
    {
      value = (int) rint(data * DIVX_HEIGHT_RANGE);
      if (value == 0)
        value = h_modul; // minimun acceptable value

      // make sure height is divisible by h_modul
      value /= h_modul;
      value *= h_modul;
      divx.height = value;

      if (s_divxheight)
        set_float (s_divxheight, XtNtopOfThumb, (float)value/DIVX_HEIGHT_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Height", divx.height);
      if (l_divxheight)
        XtVaSetValues(l_divxheight, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxbitrate"))
    {
      value = (int) rint(data * DIVX_BITRATE_RANGE);
      if (value == 0)
        value = 1;
      divx.bitrate = value*1000;

      if (s_divxbitrate)
        set_float (s_divxbitrate, XtNtopOfThumb, (float)value/DIVX_BITRATE_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Bitrate (kbit/s)", value);
      if (l_divxbitrate)
        XtVaSetValues(l_divxbitrate, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxfps"))
    {
      value = (int) rint(data * DIVX_FPS_RANGE);
      if (value == 0)
        value = 1;
      divx.fps = value*1000;

      if (s_divxfps)
        set_float (s_divxfps, XtNtopOfThumb, (float)value/DIVX_FPS_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Fps", value);
      if (l_divxfps)
        XtVaSetValues(l_divxfps, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxmaxgap"))
    {
      value = (int) rint(data * DIVX_MAXGAP_RANGE);
      if (value == 0)
        value = 1;
      divx.maxgap = (float) value/1000;

      if (s_divxmaxgap)
        set_float (s_divxmaxgap, XtNtopOfThumb, (float)value/DIVX_MAXGAP_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Max. Gap A/V (ms)", value);
      if (l_divxmaxgap)
        XtVaSetValues(l_divxmaxgap, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxdelay"))
    {
      value = (int) rint(data * 2 * DIVX_DELAY_RANGE - DIVX_DELAY_RANGE);
      divx.delay = value/1000.0;

      if (s_divxdelay)
        set_float (s_divxdelay, XtNtopOfThumb, (1+(float)value/DIVX_DELAY_RANGE)/2);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Delay A/V (ms)", value);

      if (debug) {
      fprintf(stderr,"\n");
      fprintf(stderr,"Value: %d\n", (int)value);
      fprintf(stderr,"Delay: %f\n", (float)divx.delay);
      }
      if (l_divxdelay)
        XtVaSetValues(l_divxdelay, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxaudiofrgmts"))
    {
      value = (int) rint(data * DIVX_AUDIOFRGMTS_RANGE);
      if (value == 0)
        value = 1;
      divx.audio_fragments = (float) value;

      if (s_divxaudiofrgmts)
        set_float (s_divxaudiofrgmts, XtNtopOfThumb, (float)value/DIVX_AUDIOFRGMTS_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Fragments", value);
      if (c_divxaudiofrgmts)
        XtVaSetValues(c_divxaudiofrgmts, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxaudiosizefrgmt"))
    {
      value = (int) rint(data * DIVX_AUDIOSIZEFRGMT_RANGE);
      if (value == 0)
        value = 1024;
      divx.audio_sizefragment = (float) value;

      if (s_divxaudiosizefrgmt)
        set_float (s_divxaudiosizefrgmt, XtNtopOfThumb, (float)value/DIVX_AUDIOSIZEFRGMT_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Size Fragment", value);
      if (c_divxaudiosizefrgmt)
        XtVaSetValues(c_divxaudiosizefrgmt, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxminquantizer"))
    {
      value = (int) (data * DIVX_MINQUANTIZER_RANGE);
      if (value == 0)
        value = 1;
      divx.min_quantizer = value;

      if (s_divxminquantizer)
        set_float (s_divxminquantizer, XtNtopOfThumb, (float)value/DIVX_MINQUANTIZER_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Min Quantizer", divx.min_quantizer);
      if (l_divxminquantizer)
        XtVaSetValues(l_divxminquantizer, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxmaxquantizer"))
    {
      value = (int) (data * DIVX_MAXQUANTIZER_RANGE);
      if (value == 0)
        value = 1;
      divx.max_quantizer = value;

      if (s_divxmaxquantizer)
        set_float (s_divxmaxquantizer, XtNtopOfThumb, (float)value/DIVX_MINQUANTIZER_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Max Quantizer", divx.max_quantizer);
      if (l_divxmaxquantizer)
        XtVaSetValues(l_divxmaxquantizer, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxaudiobufsize"))
    {
      value = (int) (data * AUDIO_BUFFERSIZE_RANGE);
      if (value == 0)
        value = 441;
      divx.audio_buffer_size = value;

      if (s_divxaudiobufsize)
        set_float (s_divxaudiobufsize, XtNtopOfThumb, (float)value/AUDIO_BUFFERSIZE_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Buffer Size", divx.audio_buffer_size);
      if (c_divxaudiobufsize)
        XtVaSetValues(c_divxaudiobufsize, XtNlabel, label, NULL);
    }
#ifdef HAVE_LAME
  else if (0 == strcmp(name, "divxmp3bitrate"))
    {
      value = (int) (data * MP3_BITRATE_RANGE);
      if (value == 0)
        value = 16;
      divx.mp3_bitrate = value;

      if (s_divxmp3bitrate)
        set_float (s_divxmp3bitrate, XtNtopOfThumb, (float)value/MP3_BITRATE_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 Bitrate", divx.mp3_bitrate);
      if (l_divxmp3bitrate)
        XtVaSetValues(l_divxmp3bitrate, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxmp3vbrquality"))
    {
      value = (int) (data * MP3_VBR_QUALITY_RANGE);
      if (value == 0)
        value = 1 ;
      divx.mp3_vbr_quality = value;

      if (s_divxmp3vbrquality)
        set_float (s_divxmp3vbrquality, XtNtopOfThumb, (float)value/MP3_VBR_QUALITY_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 VBR Quality", divx.mp3_vbr_quality);
      if (l_divxmp3vbrquality)
        XtVaSetValues(l_divxmp3vbrquality, XtNlabel, label, NULL);
    }
  else if (0 == strcmp(name, "divxmp3quality"))
    {
      value = (int) (data * MP3_QUALITY_RANGE);
      if (value == 0)
        value = 1 ;
      divx.mp3_quality = value;

      if (s_divxmp3quality)
        set_float (s_divxmp3quality, XtNtopOfThumb, (float)value/MP3_QUALITY_RANGE);

      sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 Quality", divx.mp3_quality);
      if (l_divxmp3quality)
        XtVaSetValues(l_divxmp3quality, XtNlabel, label, NULL);
    }
#endif // HAVE_LAME

#if 0
  printf("%s: value = %d\n", name, value);
#endif
}

static void
divx_scroll_scb (Widget widget, XtPointer clientdata, XtPointer call_data)
{
  int move = (int) call_data;
  Dimension length;
  float shown, top1, top2;
  char *name;
  float step;
  float range;
  name = XtName (XtParent (widget));

  XtVaGetValues (widget,
                 XtNlength, &length,
                 XtNshown, &shown, XtNtopOfThumb, &top1, NULL);

  if (strcmp(name, "divxwidth") == 0)
    {
      range = DIVX_WIDTH_RANGE;
      step = w_modul;
    }
  else if (strcmp(name, "divxheight") == 0)
    {
      range = DIVX_HEIGHT_RANGE;
      step = h_modul;
    }
  else if (strcmp(name, "divxbitrate") == 0)
    {
      range = DIVX_BITRATE_RANGE;
      step = 1.0;
    }
  else if (strcmp(name, "divxfps") == 0)
    {
      range = DIVX_FPS_RANGE;
      step = 1.0;
    }
   else if (strcmp(name, "divxmaxgap") == 0)
    {
      range = DIVX_MAXGAP_RANGE;
      step = 1.0;
    }
   else if (strcmp(name, "divxdelay") == 0)
    {
      range = DIVX_DELAY_RANGE;
      step = 1.0;
    }
   else if (strcmp(name, "divxaudiofrgmts") == 0)
    {
      range = DIVX_AUDIOFRGMTS_RANGE;
      step = 1.0;
    }
   else if (strcmp(name, "divxaudiosizefrgmt") == 0)
    {
      range = DIVX_AUDIOSIZEFRGMT_RANGE;
      step = 1024.0;
    }
  else if (
           (strcmp(name, "divxminquantizer") == 0)
           ||(strcmp(name, "divxmaxquantizer") == 0)
           )
    {
      range = DIVX_MINQUANTIZER_RANGE;
      step = 1.0;
    }
  else if (strcmp(name, "divxaudiobufsize") == 0)
    {
      range = AUDIO_BUFFERSIZE_RANGE;
      step = 441.0;
    }
#ifdef HAVE_LAME
  else if (strcmp(name, "divxmp3bitrate") == 0)
    {
      range = MP3_BITRATE_RANGE;
      step = 16.0;
    }
  else if (strcmp(name, "divxmp3vbrquality") == 0)
    {
      range = MP3_VBR_QUALITY_RANGE;
      step = 1.0;
    }
  else if (strcmp(name, "divxmp3quality") == 0)
    {
      range = MP3_QUALITY_RANGE;
      step = 1.0;
    }
#endif // HAVE_LAME
  else
    {
      range = 65535.0;
      step = 5.0;
    }

  //top2 = top1 + (float) move / length / 5;
  if (move > 0)
    top2 = top1 + step/range;
  else
    top2 = top1 - step/range;
  if (top2 < 0.0)
    top2 = 0.0;
  if (top2 > 1.0)
    top2 = 1.0;

#if 0
  fprintf (stderr, "name: %s, range = %f, step = %f\n", name, range, step);
  fprintf (stderr, "top1 = %f, move = %d, step/range = %f, top2 = %f\n",top1,move,step/range,top2);
  fprintf (stderr, "scroll by %d\tlength %d\tshown %f\ttop %f => %f\n",
           move, length, shown, top1, top2);
#endif

  divx_jump_scb (widget, clientdata, &top2);
}

void
create_divxwin(void)
{
  Widget c,p;
  char label[64];

  divx_shell = XtVaAppCreateShell ("DivX", "XdTV",
                                   topLevelShellWidgetClass,
                                   dpy, XtNclientLeader, app_shell, XtNvisual,vinfo->visual,
				   XtNcolormap,colormap, XtNdepth,vinfo->depth,
				   NULL);

  divx_paned =
    XtVaCreateManagedWidget ("paned", panedWidgetClass, divx_shell, NULL);

  /* DivX Codec Label */
  c_divxvideocodec = XtVaCreateManagedWidget ("ldivxcodec", labelWidgetClass, divx_paned,
                           PANED_FIX, NULL);

  /* Divx Codec Menu */
  c_divxcodec = c = XtVaCreateManagedWidget ("divxcodec", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_CODEC);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Codec",
           int_to_str (divx.codec, codec_list));
  quality_list = set_quality_list(divx.codec);
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* DivX Size Label */
  c_divxsize = XtVaCreateManagedWidget ("ldivxsize", labelWidgetClass, divx_paned,
                           PANED_FIX, NULL);

  /* DivX W-Modul Slider */
  c_wmodul = c = XtVaCreateManagedWidget ("wmodul", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_WMODUL);
  sprintf (label, "%-" LABEL_WIDTH3 "s: %5d", "W-Modul", w_modul);
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* DivX Width Slider */
  p =
    XtVaCreateManagedWidget ("divxwidth", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxwidth = XtVaCreateManagedWidget ("ldivxwidth", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxwidth = XtVaCreateManagedWidget ("sdivxwidth", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxwidth, XtNtopOfThumb, (float)divx.width/DIVX_WIDTH_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Width", divx.width);
  XtVaSetValues(l_divxwidth, XtNlabel, label, NULL);

  XtAddCallback (s_divxwidth, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxwidth, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX H-Modul Slider */
  c_hmodul = c = XtVaCreateManagedWidget ("hmodul", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_HMODUL);
  sprintf (label, "%-" LABEL_WIDTH3 "s: %5d", "H-Modul", h_modul);
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* DivX Height Slider */
  p =
    XtVaCreateManagedWidget ("divxheight", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxheight = XtVaCreateManagedWidget ("ldivxheight", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxheight = XtVaCreateManagedWidget ("sdivxheight", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxheight, XtNtopOfThumb, (float)divx.height/DIVX_HEIGHT_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Height", divx.height);
  XtVaSetValues(l_divxheight, XtNlabel, label, NULL);

  XtAddCallback (s_divxheight, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxheight, XtNscrollProc, divx_scroll_scb, NULL);

   /* DivX Bitrate Slider */
  p =
    XtVaCreateManagedWidget ("divxbitrate", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxbitrate = XtVaCreateManagedWidget ("ldivxbitrate", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxbitrate = XtVaCreateManagedWidget ("sdivxbitrate", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxbitrate, XtNtopOfThumb, (float)divx.bitrate/1000/DIVX_BITRATE_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5ld", "Bitrate (kbit/s)", divx.bitrate/1000);
  XtVaSetValues(l_divxbitrate, XtNlabel, label, NULL);

  XtAddCallback (s_divxbitrate, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxbitrate, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX FPS Slider */
  p =
    XtVaCreateManagedWidget ("divxfps", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxfps = XtVaCreateManagedWidget ("ldivxfps", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxfps = XtVaCreateManagedWidget ("sdivxfps", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxfps, XtNtopOfThumb, (float)divx.fps/1000/DIVX_FPS_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5ld", "Fps", (unsigned long)divx.fps/1000);
  XtVaSetValues(l_divxfps, XtNlabel, label, NULL);

  XtAddCallback (s_divxfps, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxfps, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX Quality Menu */
  c_divxquality = c = XtVaCreateManagedWidget ("divxquality", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_QUALITY);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Quality",
           nn(int_to_str (divx.quality, quality_list)));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* DivX min Quantizer Slider */
  p =
    XtVaCreateManagedWidget ("divxminquantizer", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxminquantizer = XtVaCreateManagedWidget ("ldivxminquantizer", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxminquantizer = XtVaCreateManagedWidget ("sdivxminquantizer", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxminquantizer, XtNtopOfThumb, (float)divx.min_quantizer/DIVX_MINQUANTIZER_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Min Quantizer", divx.min_quantizer);
  XtVaSetValues(l_divxminquantizer, XtNlabel, label, NULL);

  XtAddCallback (s_divxminquantizer, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxminquantizer, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX max Quantizer Slider */
  p =
    XtVaCreateManagedWidget ("divxmaxquantizer", panedWidgetClass, divx_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxmaxquantizer = XtVaCreateManagedWidget ("ldivxmaxquantizer", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxmaxquantizer = XtVaCreateManagedWidget ("sdivxmaxquantizer", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxmaxquantizer, XtNtopOfThumb, (float)divx.max_quantizer/DIVX_MAXQUANTIZER_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Max Quantizer", divx.max_quantizer);
  XtVaSetValues(l_divxmaxquantizer, XtNlabel, label, NULL);

  XtAddCallback (s_divxmaxquantizer, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxmaxquantizer, XtNscrollProc, divx_scroll_scb, NULL);

  /* Other Options Label */
  c_divxdisplay = XtVaCreateManagedWidget ("ldivxdisplay", labelWidgetClass, divx_paned,
                           PANED_FIX, NULL);

  /* DivX Display Frames Menu */
  c_displayframes = c = XtVaCreateManagedWidget ("divxdisplay", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_DISPLAY_FRAMES);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Display Frames",
           int_to_str (divx.display_frames, on_off_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* Record Subtitles Menu */
  c_sub = c = XtVaCreateManagedWidget ("divxsub", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_SUB);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Record Subtitles",
           int_to_str (divx.sub, on_off_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* Change channel during record Menu */
  c_chg = c = XtVaCreateManagedWidget ("divxchg", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_CHG);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Change Channel",
           int_to_str (divx.chg, on_off_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  c_schedule = c = XtVaCreateManagedWidget ("ldivxschedule", commandWidgetClass, divx_paned, PANED_FIX, NULL);
  signal(SIGCHLD, wfc);
  XtAddCallback (c, XtNcallback, execute, "xterm -geometry 80x35+0+0 -title 'Schedule a Record: (Need AT & CRON)' -e sh xdtv_record.sh");
  
  c_player = c = XtVaCreateManagedWidget ("ldivxplayer", commandWidgetClass, divx_paned, PANED_FIX, NULL);
  signal(SIGCHLD, wfc);
  XtAddCallback (c, XtNcallback, execute, lastfilename);
  
  /* select the divx.filename through a GUI */

  c_path= c = XtVaCreateManagedWidget ("ldivxfilepath", labelWidgetClass, divx_paned,
                           PANED_FIX, NULL);  
  c_path_selector = c = XtVaCreateManagedWidget ("ldivxfilepathselector", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback(c, XtNcallback, srcFileSelector, NULL);  
    
  /* DivX file path */

  t_divxfilepath = c = XtVaCreateManagedWidget ("divxfilepath", asciiTextWidgetClass, divx_paned,
                                                PANED_FIX, NULL);
  if (divx.filename == NULL) divx.filename = getcwd(NULL,0);

  XtVaSetValues (t_divxfilepath, XtNstring, divx.filename, NULL);
  

/* Open DivX Audio Menu */

  c_divxaudioopen = XtVaCreateManagedWidget ("ldivxaudioopen", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c_divxaudioopen, XtNcallback, close_divxaudio_win, (XtPointer) NULL);

/* Open Streaming Menu */

  c_streamopen = XtVaCreateManagedWidget ("lstreamopen", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c_streamopen, XtNcallback, close_streamparamswin, (XtPointer) NULL);
  
/* Open Main Options Menu */
  c_openmain = XtVaCreateManagedWidget ("lopenmain", commandWidgetClass, divx_paned, PANED_FIX, NULL);
    XtAddCallback (c_openmain, XtNcallback, action_cb, (XtPointer) & call_options_5);  

/* Button to close DivX Menu */

  c_save = XtVaCreateManagedWidget ("ldivxsave", commandWidgetClass, divx_paned,
                               PANED_FIX, NULL);
  XtAddCallback (c_save, XtNcallback, save_cb, (XtPointer) NULL);
  XtAddCallback (c_save, XtNcallback, close_divx_win, (XtPointer) NULL);

#ifdef HAVE_XPM
  XpmCreatePixmapFromData(XtDisplay(divx_shell),
                          RootWindowOfScreen(XtScreen(divx_shell)),
                          icon_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(divx_shell, XtNiconPixmap, icon_pixmap, NULL);
#ifdef HAVE_PIXMAPS
  XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_choosecodec_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxvideocodec, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_videooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxsize, XtNbitmap, icon_pixmap, NULL);  
    
  XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_otheroptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxdisplay, XtNbitmap, icon_pixmap, NULL); 
 
  XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_pathavi_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_path, XtNbitmap, icon_pixmap, NULL);  
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_openaudiooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxaudioopen, XtNbitmap, icon_pixmap, NULL);  
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_openstreamingoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streamopen, XtNbitmap, icon_pixmap, NULL);  
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_openmainoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_openmain, XtNbitmap, icon_pixmap, NULL);         
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_applysave_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_save, XtNbitmap, icon_pixmap, NULL);  
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_browse_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_path_selector, XtNbitmap, icon_pixmap, NULL);
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_schedule_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_schedule, XtNbitmap, icon_pixmap, NULL);
  
    XpmCreatePixmapFromData(XtDisplay(divx_paned),
                          RootWindowOfScreen(XtScreen(divx_paned)),
                          video_preview_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_player, XtNbitmap, icon_pixmap, NULL);        
#endif
#endif

}

void
create_divxaudiowin(void)
{
  Widget c,p;
  char label[64];
#ifdef HAVE_XPM  
  Pixmap icon_pixmap;
  Pixmap icon_shapemask;
#endif

  divxaudio_shell = XtVaAppCreateShell ("DivXAudio", "XdTV",
                                   topLevelShellWidgetClass,
                                   dpy, XtNclientLeader, app_shell, XtNvisual,vinfo->visual,
				   XtNcolormap,colormap, XtNdepth,vinfo->depth,
				   NULL);

  divxaudio_paned =
    XtVaCreateManagedWidget ("paned", panedWidgetClass, divxaudio_shell, NULL);


  /* AVI Audio Label */
  c_divxaudio = XtVaCreateManagedWidget ("ldivxaudio", labelWidgetClass, divxaudio_paned,
                           PANED_FIX, NULL);

  /* Audio Buffer Size Slider */
  p =
    XtVaCreateManagedWidget ("divxaudiobufsize", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  c_divxaudiobufsize = XtVaCreateManagedWidget ("ldivxaudiobufsize", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxaudiobufsize = XtVaCreateManagedWidget ("sdivxaudiobufsize", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxaudiobufsize, XtNtopOfThumb, (float)divx.audio_buffer_size/AUDIO_BUFFERSIZE_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Buffer Size", divx.audio_buffer_size);
  XtVaSetValues(c_divxaudiobufsize, XtNlabel, label, NULL);

  XtAddCallback (s_divxaudiobufsize, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxaudiobufsize, XtNscrollProc, divx_scroll_scb, NULL);

#ifdef HAVE_LAME
  /* Select compressed audio */
  c_compressaudio = c =
    XtVaCreateManagedWidget ("divxcompressedaudio", commandWidgetClass,
                             divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_COMPRESS_AUDIO);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Compress Audio",
           int_to_str (divx.compress_audio, on_off_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* Select bitrate mode */
  c_mp3_vbr = c =
    XtVaCreateManagedWidget ("divxmp3vbr", commandWidgetClass,
                             divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_MP3_VBR);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Bitrate Mode",
           int_to_str (divx.mp3_vbr, mp3_vbr_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* MP3 Constant Bitrate Slider */
  p_divxmp3bitrate = p =
    XtVaCreateManagedWidget ("divxmp3bitrate", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxmp3bitrate = XtVaCreateManagedWidget ("ldivxmp3bitrate", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxmp3bitrate = XtVaCreateManagedWidget ("sdivxmp3bitrate", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxmp3bitrate, XtNtopOfThumb, (float)divx.mp3_bitrate/MP3_BITRATE_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 Bitrate", divx.mp3_bitrate);
  XtVaSetValues(l_divxmp3bitrate, XtNlabel, label, NULL);
  if (int_to_str (divx.mp3_vbr, mp3_vbr_list) == "VBR (Variable)" || int_to_str (divx.compress_audio, on_off_list) == "Off")
    {
	  XtVaSetValues(l_divxmp3bitrate, XtNsensitive, False, NULL);
    }
  XtAddCallback (s_divxmp3bitrate, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxmp3bitrate, XtNscrollProc, divx_scroll_scb, NULL);

  /* MP3 VBR Quality Slider */
  p_divxmp3vbrquality = p =
    XtVaCreateManagedWidget ("divxmp3vbrquality", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxmp3vbrquality = XtVaCreateManagedWidget ("ldivxmp3vbrquality", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxmp3vbrquality = XtVaCreateManagedWidget ("sdivxmp3vbrquality", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxmp3vbrquality, XtNtopOfThumb, (float)divx.mp3_vbr_quality/MP3_VBR_QUALITY_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 VBR Quality", divx.mp3_vbr_quality);
  XtVaSetValues(l_divxmp3vbrquality, XtNlabel, label, NULL);
  if (int_to_str (divx.mp3_vbr, mp3_vbr_list) == "CBR (Constant)" || int_to_str (divx.compress_audio, on_off_list) == "Off")
    {
	  XtVaSetValues(l_divxmp3vbrquality, XtNsensitive, False, NULL);
    }
  XtAddCallback (s_divxmp3vbrquality, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxmp3vbrquality, XtNscrollProc, divx_scroll_scb, NULL);

  /* Select stereo mode */
  c_stereo = c =
    XtVaCreateManagedWidget ("divxstereo", commandWidgetClass,
                             divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_STEREO);
  sprintf (label, "%-" LABEL_WIDTH3 "s:       %s", "Stereo Mode",
           int_to_str (divx.stereo, stereo_list));
  XtVaSetValues (c, XtNlabel, label, NULL);

  /* MP3 Quality Slider */
  p_divxmp3quality = p =
    XtVaCreateManagedWidget ("divxmp3quality", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxmp3quality = XtVaCreateManagedWidget ("ldivxmp3quality", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxmp3quality = XtVaCreateManagedWidget ("sdivxmp3quality", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxmp3quality, XtNtopOfThumb, (float)divx.mp3_quality/MP3_QUALITY_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "MP3 Quality", divx.mp3_quality);
  XtVaSetValues(l_divxmp3quality, XtNlabel, label, NULL);
    if (int_to_str (divx.compress_audio, on_off_list) == "Off")
    {
	  XtVaSetValues(l_divxmp3quality, XtNsensitive, False, NULL);
    }
  XtAddCallback (s_divxmp3quality, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxmp3quality, XtNscrollProc, divx_scroll_scb, NULL);

#endif // HAVE_LAME

  /* DivX MaxGap Slider */
  p =
    XtVaCreateManagedWidget ("divxmaxgap", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxmaxgap = XtVaCreateManagedWidget ("ldivxmaxgap", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxmaxgap = XtVaCreateManagedWidget ("sdivxmaxgap", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxmaxgap, XtNtopOfThumb, (float)divx.maxgap*1000/DIVX_MAXGAP_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5ld", "Max. Gap A/V (ms)", (unsigned long)((float)(divx.maxgap*1000)));
  XtVaSetValues(l_divxmaxgap, XtNlabel, label, NULL);

  XtAddCallback (s_divxmaxgap, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxmaxgap, XtNscrollProc, divx_scroll_scb, NULL);

/* DivX delay */
  p =
    XtVaCreateManagedWidget ("divxdelay", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  l_divxdelay = XtVaCreateManagedWidget ("ldivxdelay", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxdelay = XtVaCreateManagedWidget ("sdivxdelay", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxdelay, XtNtopOfThumb, (1+divx.delay*1000/DIVX_DELAY_RANGE)/2);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5ld", "Delay A/V (ms)", (unsigned long)((float)(divx.delay*1000)));
  if (debug) {
  fprintf(stderr,"Delay: %f\n", (float)divx.delay);
  }
  XtVaSetValues(l_divxdelay, XtNlabel, label, NULL);

  XtAddCallback (s_divxdelay, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxdelay, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX Audio Fragments Slider */
  p =
    XtVaCreateManagedWidget ("divxaudiofrgmts", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  c_divxaudiofrgmts = XtVaCreateManagedWidget ("ldivxaudiofrgmts", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxaudiofrgmts = XtVaCreateManagedWidget ("sdivxaudiofrgmts", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxaudiofrgmts, XtNtopOfThumb, (float)divx.audio_fragments/DIVX_AUDIOFRGMTS_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Fragments", divx.audio_fragments);
  XtVaSetValues(c_divxaudiofrgmts, XtNlabel, label, NULL);

  XtAddCallback (s_divxaudiofrgmts, XtNjumpProc, divx_jump_scb, NULL);

  XtAddCallback (s_divxaudiofrgmts, XtNscrollProc, divx_scroll_scb, NULL);

  /* DivX Audio Size Fragment Slider */
  p =
    XtVaCreateManagedWidget ("divxaudiosizefrgmt", panedWidgetClass, divxaudio_paned,
                             XtNorientation, XtEvertical, PANED_FIX, NULL);

  c_divxaudiosizefrgmt = XtVaCreateManagedWidget ("ldivxaudiosizefrgmt", labelWidgetClass, p,
                               XtNshowGrip, False, NULL);

  s_divxaudiosizefrgmt = XtVaCreateManagedWidget ("sdivxaudiosizefrgmt", scrollbarWidgetClass, p,
                                          PANED_FIX, NULL);

  set_float(s_divxaudiosizefrgmt, XtNtopOfThumb, (float)divx.audio_sizefragment/DIVX_AUDIOSIZEFRGMT_RANGE);
  sprintf(label, "%-" LABEL_WIDTH3 "s: %5d", "Audio Size Fragment", divx.audio_sizefragment);
  XtVaSetValues(c_divxaudiosizefrgmt, XtNlabel, label, NULL);
  XtAddCallback (s_divxaudiosizefrgmt, XtNjumpProc, divx_jump_scb, NULL);
  XtAddCallback (s_divxaudiosizefrgmt, XtNscrollProc, divx_scroll_scb, NULL);

/* Open DivX Video Menu */

  c_divxopen = XtVaCreateManagedWidget ("ldivxopen", commandWidgetClass, divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c_divxopen, XtNcallback, close_divx_win, (XtPointer) NULL);

/* Open Streaming Menu */

  c_streamopen = XtVaCreateManagedWidget ("lstreamopen", commandWidgetClass, divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c_streamopen, XtNcallback, close_streamparamswin, (XtPointer) NULL);
  
/* Open Main Options Menu */
  c_openmain = XtVaCreateManagedWidget ("lopenmain", commandWidgetClass, divxaudio_paned, PANED_FIX, NULL);
    XtAddCallback (c_openmain, XtNcallback, action_cb, (XtPointer) & call_options_5);   

/* Button to close DivXAudio Menu */

  c_audiosave = XtVaCreateManagedWidget ("ldivxsave", commandWidgetClass, divxaudio_paned, PANED_FIX, NULL);
  XtAddCallback (c_audiosave, XtNcallback, save_cb, (XtPointer) NULL);
  XtAddCallback (c_audiosave, XtNcallback, close_divxaudio_win, (XtPointer) NULL);

#ifdef HAVE_XPM
  XpmCreatePixmapFromData(XtDisplay(divxaudio_shell),
                          RootWindowOfScreen(XtScreen(divxaudio_shell)),
                          icon_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(divxaudio_shell, XtNiconPixmap, icon_pixmap, NULL);
#ifdef HAVE_PIXMAPS
  XpmCreatePixmapFromData(XtDisplay(divxaudio_paned),
                          RootWindowOfScreen(XtScreen(divxaudio_paned)),
                          audio_audiooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxaudio, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(divxaudio_paned),
                          RootWindowOfScreen(XtScreen(divxaudio_paned)),
                          audio_openvideooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxopen, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(divxaudio_paned),
                          RootWindowOfScreen(XtScreen(divxaudio_paned)),
                          audio_openstreamingoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streamopen, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(divxaudio_paned),
                          RootWindowOfScreen(XtScreen(divxaudio_paned)),
                          audio_openmainoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_openmain, XtNbitmap, icon_pixmap, NULL);  
  
  XpmCreatePixmapFromData(XtDisplay(divxaudio_paned),
                          RootWindowOfScreen(XtScreen(divxaudio_paned)),
                          audio_applysave_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_audiosave, XtNbitmap, icon_pixmap, NULL);      
#endif
#endif

}

void
create_streamparamswin()
{
  Widget c;
  char label[64];
#ifdef HAVE_XPM  
  Pixmap icon_pixmap;
  Pixmap icon_shapemask;
#endif

  streamparams_shell = XtVaAppCreateShell ("StreamParams", "XdTV",
                                   topLevelShellWidgetClass,
                                   dpy, XtNclientLeader, app_shell, XtNvisual,vinfo->visual,
				   XtNcolormap,colormap, XtNdepth,vinfo->depth,
				   NULL);
  streamparams_paned =
    XtVaCreateManagedWidget ("paned", panedWidgetClass, streamparams_shell, NULL);

/* Streaming Options Label */
  c_stropts = XtVaCreateManagedWidget ("lstropts", labelWidgetClass, streamparams_paned,
                           PANED_FIX, NULL);

  /* Streaming activation Menu */
  c_stream_onoff = c = XtVaCreateManagedWidget ("Streaming_Server", commandWidgetClass, streamparams_paned,
                               PANED_FIX, NULL);
    XtAddCallback (c, XtNcallback, divx_menu_cb, (XtPointer) DIVX_STREAM_ONOFF);
      sprintf (label, "%-" LABEL_WIDTH3 "s:     %s  ", "Streaming Server Activation",
           int_to_str (divx.stream, on_off_list));
      XtVaSetValues (c, XtNlabel, label, NULL);


/* Http Port Label */
  c_http_port = c = XtVaCreateManagedWidget ("lhttp_port", labelWidgetClass, streamparams_paned,
                           PANED_FIX, NULL);
  t_http_port_str = c = XtVaCreateManagedWidget ("http_port", asciiTextWidgetClass, streamparams_paned,
                                                PANED_FIX, NULL);
  if (!divx.stream)
    XtVaSetValues(t_http_port_str, XtNsensitive, False, NULL);

  sprintf (http_port, "%d", divx.http_port);
  XtVaSetValues (t_http_port_str, XtNstring, http_port, NULL);


/* launch streaming Action */
  c_streaming = c = XtVaCreateManagedWidget ("lstreaming", commandWidgetClass, streamparams_paned, PANED_FIX, NULL);
  //XtAddCallback (c, XtNcallback, execute, "xterm");
  XtAddCallback (c, XtNcallback, action_cb, (XtPointer) & call_recordmovie);
    if (!divx.stream)
      XtVaSetValues(c_streaming, XtNsensitive, False, NULL);

/* Open DivX Video Menu */
  c_divxopen = XtVaCreateManagedWidget ("ldivxopen", commandWidgetClass, streamparams_paned,
                               PANED_FIX, NULL);
    XtAddCallback (c_divxopen, XtNcallback, close_divx_win, (XtPointer) NULL);
    if (!divx.stream)
      XtVaSetValues(c_divxopen, XtNsensitive, False, NULL);


/* Open DivX Audio Menu */
  c_divxaudioopen = XtVaCreateManagedWidget ("ldivxaudioopen", commandWidgetClass, streamparams_paned,
                               PANED_FIX, NULL);
    XtAddCallback (c_divxaudioopen, XtNcallback, close_divxaudio_win, (XtPointer) NULL);
    if (!divx.stream)
      XtVaSetValues(c_divxaudioopen, XtNsensitive, False, NULL);

    
/* Open Main Options Menu */
  c_openmain = XtVaCreateManagedWidget ("lopenmain", commandWidgetClass, streamparams_paned, PANED_FIX, NULL);
    XtAddCallback (c_openmain, XtNcallback, action_cb, (XtPointer) & call_options_5);  
        

/* Button to apply & save Stream Parameters GUI */
  c_streamparamssave = XtVaCreateManagedWidget ("lstreamparamssave", commandWidgetClass, streamparams_paned, PANED_FIX, NULL);
  XtAddCallback (c_streamparamssave, XtNcallback, save_cb, (XtPointer) NULL);


/* Button to close Stream Parameters GUI */
  c_streamparamsclose = XtVaCreateManagedWidget ("lstreamparamsclose", commandWidgetClass, streamparams_paned, PANED_FIX, NULL);
  XtAddCallback (c_streamparamsclose, XtNcallback, close_streamparamswin, (XtPointer) NULL);

#ifdef HAVE_XPM
  XpmCreatePixmapFromData(XtDisplay(streamparams_shell),
                          RootWindowOfScreen(XtScreen(streamparams_shell)),
                          icon_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(streamparams_shell, XtNiconPixmap, icon_pixmap, NULL);
#ifdef HAVE_PIXMAPS
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_streamingoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_stropts, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_choosehttpport_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_http_port, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openmainoptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_openmain, XtNbitmap, icon_pixmap, NULL);       
if (!divx.stream)
  {
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_startstreaming_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streaming, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openvideooptions_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxopen, XtNbitmap, icon_pixmap, NULL); 
    
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openaudiooptions_off_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxaudioopen, XtNbitmap, icon_pixmap, NULL); 
  }
  else
  {
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_startstreaming_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streaming, XtNbitmap, icon_pixmap, NULL);
  
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openvideooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxopen, XtNbitmap, icon_pixmap, NULL); 
    
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_openaudiooptions_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_divxaudioopen, XtNbitmap, icon_pixmap, NULL); 
  }
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_applysave_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streamparamssave, XtNbitmap, icon_pixmap, NULL);  
  
  XpmCreatePixmapFromData(XtDisplay(streamparams_paned),
                          RootWindowOfScreen(XtScreen(streamparams_paned)),
                          stream_close_xpm,
                          &icon_pixmap,
                          &icon_shapemask, NULL);
  XtVaSetValues(c_streamparamsclose, XtNbitmap, icon_pixmap, NULL);      
#endif
#endif

}



void update_divxfilename(void)
{
  if(t_divxfilepath)
    XtVaGetValues (t_divxfilepath, XtNstring, &divx.filename, NULL);
}

void update_http_port(void)
{
  if(!t_http_port_str) return;
  XtVaGetValues (t_http_port_str, XtNstring, &http_port_str, NULL);
  sscanf (http_port_str, "%d", &divx.http_port);
  if (debug)
    fprintf(stderr,"divx.http_port : %d\n",divx.http_port);
}
