/*****************************************************************************
 * $Id: mixer-oss.c,v 1.3 2004/09/19 15:55:49 alainjj Exp $
 * Program under GNU General Public License (see ../COPYING)
 *****************************************************************************/

#include "config.h"
#ifdef HAVE_OSS
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <string.h>
#include <unistd.h>

#if defined(HAVE_LINUX_SOUNDCARD_H)
#include <linux/soundcard.h>
#elif defined(HAVE_SOUNDCARD_H)
#include <soundcard.h>
#elif defined(HAVE_SYS_SOUNDCARD_H)
#include <sys/soundcard.h>
#elif defined(HAVE_MACHINE_SOUNDCARD_H)
#include <machine/soundcard.h>
#endif

#include "mixer.h"
extern int debug;
#define NB_DEVICES 5
static char *names[] = SOUND_DEVICE_NAMES;
static char *device_list[NB_DEVICES] = {"line", "pcm", "igain", "dig1", "vol"};
static int mix;
static int dev_id[NB_DEVICES] = { -1, -1, -1, -1, -1};
static int devmask, recmask, old_recsrc, recsrc;
static int volume[NB_DEVICES];
static int muted[NB_DEVICES];
static char *filename="/dev/mixer";

static int oss_open(void) {
  int i, j, dix = 0x0a0a;
  if(mixer_tvchan) device_list[MIXER_CHAN_TV]=mixer_tvchan;
  if(mixer_pcmchan) device_list[MIXER_CHAN_PCM]=mixer_pcmchan;
  if(mixer_dev) filename=mixer_dev;
#ifdef __FreeBSD__
  device_list[MIXER_CHAN_CAPTURE]="rec";
#endif
    if (-1 == (mix = open (filename, O_RDWR)))
    {
      perror ("mixer open");
      return -1;
    }

  if (-1 == ioctl (mix, MIXER_READ (SOUND_MIXER_DEVMASK), &devmask))
    {
      perror ("mixer read devmask");
      return -1;
    }
  //printf("devmask = 0x%04X\n", devmask);    

  if (-1 == ioctl (mix, MIXER_READ (SOUND_MIXER_RECMASK), &recmask))
    {
      perror ("mixer read recmask");
      return -1;
    }
  //printf("recmask = 0x%04X\n", recmask);    

  for (j = 0; j < NB_DEVICES; j++)
    {
      if (device_list[j] == (char *)NULL)
        continue;

      for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
        {
          if (debug)
            printf("mixer names [%d] = '%s'\n", i, names[i]);

          if ((1 << i) & devmask
              && strcasecmp (names[i], device_list[j]) == 0)
            {
              if (-1 == ioctl (mix, MIXER_READ (i), &volume[j]))
                {
                  perror ("mixer read volume");
                  return -1;
                }
              else
                {
                  dev_id[j] = i;
                  muted[j] = 0;
                  if (volume[j] == 0)
                    {
                      if (j == MIXER_CHAN_CAPTURE)
                        volume[j] = dix;
                      else
                        volume[j] = 0x4444;
                      if (-1 == ioctl (mix, MIXER_WRITE (i), &volume[j]))
                        {
                          perror ("mixer write volume");
                          return -1;
                        }
                    }

                  if (debug)
                    printf
                      ("mixer: %s (%d) opened with volume = %d = 0x%04X\n",
                       device_list[j], dev_id[j], volume[j], volume[j]);
                }
              break;
            }
        }
    }



  if (-1 == dev_id[0])
    {
      fprintf (stderr, "mixer: hav'nt found device '%s'\nmixer: available: ",
               mixer_tvchan);
      for (i = 0; i < SOUND_MIXER_NRDEVICES; i++)
        if ((1 << i) & devmask)
          fprintf (stderr, " '%s'", names[i]);
      fprintf (stderr, "\n");
    }

  if (-1 == ioctl (mix, MIXER_READ (SOUND_MIXER_RECSRC), &recsrc))
    {
      perror ("mixer read recsrc");
      return -1;
    }
  old_recsrc = recsrc;
  for (i = 0; i < NB_DEVICES; i++)
    {
      int c = 0;
      if ((1 << dev_id[i]) & recmask)
	c= ((1 << dev_id[i]) & recsrc);
      if (debug)
        printf ("mixer: mixer_open: captured (%s) = %d\n", device_list[i],c);
    }

  // Make sure IGAIN isn't muted
  if (-1 == ioctl (mix, MIXER_READ (dev_id[MIXER_CHAN_CAPTURE]), &volume[MIXER_CHAN_CAPTURE]))
    fprintf(stderr, "mixer: mixer_open: couldn't read %s volume\n", device_list[MIXER_CHAN_CAPTURE]);

  if(debug)
    fprintf(stderr, "****************igain volume = 0x%04X\n", volume[MIXER_CHAN_CAPTURE]);

  if ( ((volume[MIXER_CHAN_CAPTURE] & 0xFF) == 0) || ((volume[MIXER_CHAN_CAPTURE] & 0xFF00)>>8 == 0) )
    {
      if (-1 == ioctl (mix, MIXER_WRITE (dev_id[MIXER_CHAN_CAPTURE]), &dix))
        fprintf(stderr, "mixer: mixer_open: couldn't write %s volume\n", device_list[MIXER_CHAN_CAPTURE]);
  if (debug)
    fprintf(stderr, "mixer: mixer_open: OSS setting IGAIN volume to %04X\n", dix);
    }
  return (-1 != dev_id[MIXER_CHAN_TV]) ? 0 : -1;
}

static void oss_close(void) {
  int i;
  close (mix);
  for (i = 0; i < NB_DEVICES; i++) dev_id[i] = -1;
}

static void oss_mute (int chan, int muted2) {
  int val;
  if (-1 == dev_id[chan]) return;
  if(muted2) val=0; else val=volume[chan];
  if (-1 == ioctl (mix, MIXER_WRITE (dev_id[chan]), &val)) {
    perror("mute/unmute");
    return;
  }
  muted[chan] = muted2;
}

static int oss_get_volume (int chan) {
  return (-1 == dev_id[chan]) ? -1 : (volume[chan] & 0x7f);
}

static void oss_set_volume (int chan, int val) {
  if (-1 == dev_id[chan]) return;
  val &= 0x7f;
  volume[chan] = val | (val << 8);;
  if (-1 == ioctl (mix, MIXER_WRITE (dev_id[chan]), &volume[chan]))
    {
      perror ("mixer write volume");
      return;
    }
  muted[chan] = 0;
  return;
}

static void oss_capture(void) {
  if (-1 == dev_id[MIXER_CHAN_PCM]) return;
  if (!((1 << dev_id[MIXER_CHAN_TV]) & recmask)) return;
  recsrc = (1 << dev_id[MIXER_CHAN_TV]);
  if (-1 == ioctl (mix, MIXER_WRITE (SOUND_MIXER_RECSRC), &recsrc)) {
    perror("oss_recsrc");
    return;
  }
}


static void oss_uncapture (void) {
  recsrc = old_recsrc;
  if (-1 == ioctl (mix, MIXER_WRITE (SOUND_MIXER_RECSRC), &recsrc)) {
    perror("oss_recsrc");
    return;
  }
}

struct MIXER mixer_oss = {
  "oss",
  oss_open,
  oss_close,
  oss_mute,
  oss_get_volume,
  oss_set_volume,
  oss_capture,
  oss_uncapture
};
#endif
