/*****************************************************************************
 * xfile.c: for file selector: management of X files
 *****************************************************************************
 * $Id: xfile.c,v 1.1 2004/08/07 23:00:10 pingus77 Exp $
 *****************************************************************************
 *
 * Adapted by Pingus 2004
 *
 *****************************************************************************
 *
 * Copyright (C) by Nobuyuki Maruichi
 * From mwxcd 1.32
 *
 * Version1.0	for Sony-NEWS	(NEWS-OS Release3.3a:UNIX 4.3BSD/X11R3)
 *              Copyright1991	Arakawa Laboratry(Nobuyuki Maruichi)
 * Version1.1	for FM TOWNS II	(TOWNS Linux 0.973:Linux/X11R5)
 *				Copyright1993	Nobuyuki Maruichi
 *
 *				Date 1992/4/15 (Wed.)--1992/4/18(Sat.)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *
 ***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <strings.h>
#include <time.h>
#include <signal.h>
#include <pwd.h>
#include <grp.h>

#include <X11/Intrinsic.h>	/* Intrinsics Definitions */
#include <X11/StringDefs.h>	/* Standard Name-String definitions */
#include <X11/Xaw/Label.h>
#include <X11/Xaw/List.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>
#include <unistd.h>

#include "xfile.h"
#ifndef	ANOTHER_FUNCTION
#include "stringuty.h"
#endif /*ANOTHER_FUNCTION*/

int (*sort_comp)(const void *file1, const void *file2) = ascii_comp;

char *filePickup;
char *wildCard = NULL;

char *modes[] = {
	"---", "--x", "-w-", "-wx",
	"r--", "r-x", "rw-", "rwx",
};

FileLink *XfvLinkAlloc(const char *string_ptr)
{
	FileLink *fileLinkAlloc;
	FileInfo *fileInfoAlloc;

	fileLinkAlloc = XtNew(FileLink);
	fileInfoAlloc = XtNew(FileInfo);
	fileLinkAlloc->pos = fileInfoAlloc;
	fileLinkAlloc->pos->size_kb = NULL;
	fileLinkAlloc->pos->attr = NULL;
	fileLinkAlloc->pos->link = NULL;
	fileLinkAlloc->pos->user = NULL;
	fileLinkAlloc->pos->size = NULL;
	fileLinkAlloc->pos->date = NULL;
	fileLinkAlloc->pos->filename = NULL;
	fileLinkAlloc->pos->filename = XtNewString(string_ptr);
	fileLinkAlloc->next = NULL;
	return (fileLinkAlloc);
}

void XfvLinkFree(FileLink *release_ptr)
{
	if(release_ptr != NULL) {
		if(release_ptr->pos != NULL) {
			XtFree(release_ptr->pos->size_kb);
			XtFree(release_ptr->pos->attr);
			XtFree(release_ptr->pos->link);
			XtFree(release_ptr->pos->user);
			XtFree(release_ptr->pos->size);
			XtFree(release_ptr->pos->date);
			XtFree(release_ptr->pos->filename);
			XtFree((char*)release_ptr->pos);
		}
		XtFree((char*)release_ptr);
	}
}

void XfvLinkedFree(FileLink *release_top_ptr)
{
	FileLink *next_ptr;
	FileLink *release_ptr;

	for (release_ptr = release_top_ptr;
		 release_ptr != NULL; release_ptr = next_ptr) {
		next_ptr = release_ptr->next;
		XfvLinkFree(release_ptr);
	}
}

char *ls_attr(int mode)
{
	int file_type;
	char perms[12], *work;

	switch (mode & S_IFMT) {
	case S_IFREG:
		file_type = '-';
		break;
	case S_IFDIR:
		file_type = 'd';
		break;
	case S_IFCHR:
		file_type = 'c';
		break;
	case S_IFBLK:
		file_type = 'b';
		break;
	default:
		file_type = '?';
		break;
	}
	sprintf(perms, "%c%3.3s%3.3s%3.3s", file_type,
	 modes[(mode >> 6) & 07], modes[(mode >> 3) & 07], modes[mode & 07]);

	if ((mode & S_ISUID) != 0)
		perms[3] = 's';
	if ((mode & S_ISGID) != 0)
		perms[6] = 's';
	if ((mode & S_ISVTX) != 0)
		perms[9] = 't';
	work = (char *) malloc(12);
	if (work != NULL) {
		sprintf(work, "%10.10s", perms);
	}
	return (work);
}

FileLink *Xfvprintout(const char *dir, const char *name)
{
	struct stat sbuf;
	char newname[1024];
	char stringbuf[1024];
	FileLink *xfnp;
	struct passwd *file_user;
	struct group *file_group;

	sprintf(newname, "%s/%s", dir, name);

	/********************************************
	 *	At this point we know the file exists,
	 *	so this won't fail.
	 ********************************************/
	stat(newname, &sbuf);

	xfnp = XfvLinkAlloc(name);
	if (xfnp != NULL) {
		sprintf(stringbuf, "%5lu ", (sbuf.st_size + 1023) / 1024);
		xfnp->pos->size_kb = XtNewString(stringbuf);
		xfnp->pos->attr = ls_attr(sbuf.st_mode);
		sprintf(stringbuf, "%3d ", sbuf.st_nlink);
		xfnp->pos->link = XtNewString(stringbuf);
		file_user = getpwuid(sbuf.st_uid);
		if(file_user == NULL)
			sprintf(stringbuf, "%u/", sbuf.st_uid);
		else
			sprintf(stringbuf, "%s/", file_user->pw_name);
		file_group = getgrgid(sbuf.st_gid);
		if(file_group == NULL)
			sprintf(stringbuf + strlen(stringbuf), "%u", (unsigned)sbuf.st_gid);
		else
			sprintf(stringbuf + strlen(stringbuf), "%s", file_group->gr_name);
		xfnp->pos->user = XtNewString(stringbuf);
		sprintf(stringbuf, "%9lu ", sbuf.st_size);
		xfnp->pos->size = XtNewString(stringbuf);
		sprintf(stringbuf, "%.12s", ctime(&sbuf.st_mtime) + 4);
		xfnp->pos->date = XtNewString(stringbuf);
		memcpy(&xfnp->pos->stat_buf, &sbuf, sizeof(sbuf));
	}
	return (xfnp);
}

FileLink *Xfvlist(const char *name)
{
	DIR *dp;
	FileLink *fileTop_ptr = NULL, **fileLink_ptr;
	struct direct *dir;

	/***********************
	 *	Open the directory.
	 ***********************/
	if ((dp = opendir(name)) == NULL) {
		fprintf(stderr, "%s: cannot open.\n", name);
		return(NULL);
	}
	/************************
	 *	For each entry...
	 ************************/
	dir = readdir(dp);
	if(dir != NULL) {
		fileTop_ptr = Xfvprintout(name, dir->d_name);
		fileLink_ptr = &fileTop_ptr->next;
		while ((dir = readdir(dp)) != NULL) {
			/*************************
			 *	Skip removed files.
			 *************************/
			if (dir->d_ino == 0)
				continue;
			/******************
			 * Print it out,
			 ******************/
			*fileLink_ptr = Xfvprintout(name, dir->d_name);
			fileLink_ptr = &(*fileLink_ptr)->next;
		}
	}
	closedir(dp);
	return (fileTop_ptr);
}

FileInfo *XfvListMake(FileLink *bk, int *n)
{
	FileLink *nx;
	FileInfo *XfvLists, *XfvLMakep, *XfvListsWork;

	XfvLists = (FileInfo *) malloc(sizeof(FileInfo) * *n);
	XfvLMakep = XfvLists;
	for (; bk != 0; bk = nx) {
		XfvListsWork = bk->pos;
		*XfvLMakep = *XfvListsWork;
		++XfvLMakep;
		nx = bk->next;
	};
	return (XfvLists);
}

int linkCount(FileLink *linkTop_ptr)
{
	int counter;
	FileLink *current_ptr;

	counter = 0;
	current_ptr = linkTop_ptr;
	while(current_ptr != NULL) {
		++counter;
		current_ptr = current_ptr->next;
	}
	return(counter);
}

#ifdef	ANOTHER_FUNCTION
int arrayCount(XtPointer *pointerArray)
{
	int counter;
	XtPointer *current_ptr;

	counter = 0;
	while(*(pointerArray+counter) != NULL) {
		++counter;
	}
	return(counter);
}
#endif /*ANOTHER_FUNCTION*/

int ascii_comp(const void *file1, const void *file2)
{
	int status = 0;
	FileInfo *file1_ptr = *(FileInfo **)file1;
	FileInfo *file2_ptr = *(FileInfo **)file2;

	switch(-((file1_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR) 
			+((file2_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR)) {
	case -1:
		status = -1;
		break;
	case 0:
		status = strcmp(file1_ptr->filename, file2_ptr->filename);
		break;
	case +1:
		status = +1;
		break;
	default:
		break;
	}
	return(status);
}

#define numcmp(n1, n2)	(-((n1) < (n2)) + ((n1) > (n2)))

int size_comp(const void *file1, const void *file2)
{
	int status = 0;
	FileInfo *file1_ptr = *(FileInfo **)file1;
	FileInfo *file2_ptr = *(FileInfo **)file2;

	switch(((file1_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR) 
			+ 2 * ((file2_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR)) {
	case 0x00:
		status = numcmp(file1_ptr->stat_buf.st_size, 
							file2_ptr->stat_buf.st_size);
		break;
	case 0x01:
		status = -1;
		break;
	case 0x02:
		status = +1;
		break;
	case 0x03:
		status = numcmp(file1_ptr->stat_buf.st_size, 
							file2_ptr->stat_buf.st_size);
		break;
	default:
		break;
	}
	return(status);
}

int suffix_comp(const void *file1, const void *file2)
{
	int status = 0;
	const char *suffix1, *suffix2;
	FileInfo *file1_ptr = *(FileInfo **)file1;
	FileInfo *file2_ptr = *(FileInfo **)file2;

	switch(((file1_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR) 
			+ 2 * ((file2_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR)) {
	case 0x00:
	case 0x03:
		suffix1 = strrchr(file1_ptr->filename, '.');
		if(suffix1 == file1_ptr->filename) {
			suffix1 = NULL;
		}
		suffix2 = strrchr(file2_ptr->filename, '.');
		if(suffix2 == file2_ptr->filename) {
			suffix2 = NULL;
		}
		switch((suffix1 != NULL) + 2 * (suffix2 != NULL)) {
		case 0x00:
			status = strcmp(file1_ptr->filename, file2_ptr->filename);
			break;
		case 0x01:
			status = +1;
			break;
		case 0x02:
			status = -1;
			break;
		case 0x03:
			status = strcmp(suffix1, suffix2);
			break;
		default:
			break;
		}
		break;
	case 0x01:
		status = -1;
		break;
	case 0x02:
		status = +1;
		break;
	default:
		break;
	}
	return(status);
}

int date_comp(const void *file1, const void *file2)
{
	int status = 0;
	double dstatus;
	FileInfo *file1_ptr = *(FileInfo **)file1;
	FileInfo *file2_ptr = *(FileInfo **)file2;

	switch(((file1_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR) 
			+ 2 * ((file2_ptr->stat_buf.st_mode & S_IFMT) == S_IFDIR)) {
	case 0x00:
	case 0x03:
		dstatus = difftime(file1_ptr->stat_buf.st_mtime, 
							file2_ptr->stat_buf.st_mtime);
		status = +(dstatus < 0.0) - (dstatus > 0.0);
		break;
	case 0x01:
		status = -1;
		break;
	case 0x02:
		status = +1;
		break;
	default:
		break;
	}
	return(status);
}

#ifdef	LINK_NAMELIST_TYPE

char **makeFileList(FileLink *linkTop_ptr)
{
	int counter;
	int limit;
	int st_mode, dirFlag;
	char **fileList;
	char filename[8192];
	const char *dirmark[] = {"", "[Dir] "};
	FileLink *current_ptr;

	limit = linkCount(linkTop_ptr);
	fileList = (char **)XtMalloc(sizeof(char *) * (limit+1));
	for(counter = 0, current_ptr = linkTop_ptr;
					current_ptr != NULL && counter < limit;
					++counter, current_ptr = current_ptr->next) {
		st_mode = current_ptr->pos->stat_buf.st_mode & S_IFMT;
		dirFlag = (st_mode == S_IFDIR);
		sprintf(filename, "%s%s", current_ptr->pos->filename, 
							dirmark[dirFlag]);
		*(fileList + counter) = XtNewString(filename);
	}
	*(fileList + limit) = NULL;
	return(fileList);
}
#endif	/*LINK_NAMELIST_TYPE*/

char **makeFileList(FileInfo **arrayTop_ptr)
{
	int counter;
	int limit;
	int st_mode, dirFlag;
	char **fileList;
	char filename[8192];
	const char *dirmark[] = {"", "[Dir] "};

	limit = arrayCount((XtPointer*)arrayTop_ptr);
	fileList = (char **)XtMalloc(sizeof(char *) * (limit+1));
	for(counter = 0;
			 *(arrayTop_ptr+counter) != NULL && counter < limit;
					++counter) {
		st_mode = 
			(*(arrayTop_ptr+counter))->stat_buf.st_mode & S_IFMT;
		dirFlag = (st_mode == S_IFDIR);
		sprintf(filename, "%s%s",dirmark[dirFlag], (*(arrayTop_ptr+counter))->filename);
		*(fileList + counter) = XtNewString(filename);
	}
	*(fileList + limit) = NULL;
	return(fileList);
}

void fileListFree(char **fileList)
{
	int counter;

	if(fileList != NULL) {
		for(counter = 0;*(fileList+counter) != NULL;++counter) {
			XtFree(*(fileList + counter));
		}
		XtFree((char*)fileList);
	}
}

FileInfo **makeFileInfoList(FileLink *linkTop_ptr)
{
	int counter;
	int lcounter;
	int limit;
	FileInfo **fileList;
	FileLink *current_ptr;

	limit = linkCount(linkTop_ptr);
	fileList = (FileInfo**)XtMalloc(sizeof(FileInfo*) * (limit+1));
	for(counter = 0;counter < limit;++counter) {
		*(fileList + counter) = NULL;
	}
	for(counter = 0, lcounter = 0, current_ptr = linkTop_ptr;
			current_ptr != NULL && lcounter < limit;
			++lcounter, current_ptr = current_ptr->next) {
		if(ismeta(filePickup, current_ptr->pos->filename) == 0 || 
				(current_ptr->pos->stat_buf.st_mode & S_IFMT)
												 == S_IFDIR) {
			*(fileList + counter) = current_ptr->pos;
			++counter;
		}
	}
	*(fileList + limit) = NULL;
	return(fileList);
}

void fileListSelect(Widget w, XtPointer client_data, XtPointer call_data)
{
	int status;
	FileInfo *currentFile;
	FileSelctorInfo *fileInfo = (FileSelctorInfo*)client_data;
	XawListReturnStruct *listPane = (XawListReturnStruct*)call_data;

	currentFile = *(fileInfo->infoList + listPane->list_index);
	switch(currentFile->stat_buf.st_mode & S_IFMT) {
	case S_IFLNK:
	case S_IFDIR:
		status = chdir(currentFile->filename);
		if(status == 0) {
			remakeFileSelector(fileInfo);
		}
		break;
	case S_IFREG:
		XtVaSetValues(fileInfo->fileWidget, XtNstring, 
									currentFile->filename, NULL);
		break;
	}
}

void remakeFilePickup(FileSelctorInfo *infomation)
{
	infomation->infoList = makeFileInfoList(infomation->linkedInfo);
	if(sort_comp != NULL) {
		qsort(infomation->infoList, 
			arrayCount((XtPointer*)infomation->infoList), 
			sizeof(*infomation->infoList), sort_comp);
	}
	infomation->nameList = makeFileList(infomation->infoList);
	XawListChange(infomation->listWidget, infomation->nameList, 
				arrayCount((XtPointer*)infomation->nameList), 1000, True);
}

void remakeFileSelector(FileSelctorInfo *infomation)
{
	if(infomation->nameList != NULL)
		XtFree((char*)infomation->nameList);
	XfvLinkedFree(infomation->linkedInfo);
	free(infomation->searchingPath);
	infomation->searchingPath = getcwd(NULL, 0);
	XtVaSetValues(infomation->dirWidget, 
					XtNlabel, infomation->searchingPath, NULL);
	infomation->linkedInfo = Xfvlist(infomation->searchingPath);

	remakeFilePickup(infomation);
}

#define	eqnull(a, b)	(((a) == '\0') && ((b) == '\0'))
#define	isnull(a, b)	(((a) == '\0') || ((b) == '\0'))
#define	ifinal(a, b)	(-((a) == '\0') + ((b) == '\0'))

static int meta_select(const char **meta, char c)
{
	int f1;
	int b, n;
	
	for(f1 = 0,b = 0;**meta != ']' && **meta != '\0';b = n,++*meta) {
		switch(n = **meta) {
		case '-':
			if(*(*meta + 1) == ']' || *(*meta + 1) == '\0') {
				f1 |= (c >= b);
				break;
			}
			if(*(*meta + 1) >= b) {
				f1 |= (c >= b && c <= *(*meta + 1));
			} else {
				f1 |= (c >= *(*meta + 1) && c <= b);
			}
			break;
		default:
			f1 |= (c == n);
			break;
		}
		b = n;
	}
	if(**meta == ']')
		++*meta;
	return(!f1);
}

int ismeta(const char *meta, const char *name)
{
	int	c;
	int	f1 = 0;
	int	loopFlag = 0;
	
	if(meta == NULL)
		return(0);
	do {
		switch(c = *meta++) {
		case '*':
			do {
				f1 = ismeta(meta, name);
			} while(f1 < 0 && *name++ != '\0');
			loopFlag = 0;
			break;
		case '?':
			f1 = *name++ != '\0';
			loopFlag = f1;
			break;
		case '[':
			f1 = 1;
			loopFlag = *name != '\0';
			if(loopFlag == 0) {
				f1 = 1;
				break;
			}
			f1 = -meta_select(&meta, *name);
			++name;
			loopFlag = !f1;
			break;
		case '\\':
			c = *meta++;
			if(c == '\0') {
				f1 = 1;
				loopFlag = 0;
				break;
			}
		default:
			f1 = (isnull(c, *name)?ifinal(c, *name):-(c != *name));
			loopFlag = (c == *name) && !isnull(c, *name);
			++name;
			break;
		}
	} while(loopFlag);
	return(f1);
}

