/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2001  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2001  Hvard Kvlen
 *  Copyright (C) 2001, Jorn Baayen <jorn@nl.linux.org>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA
 * 02111-1307, USA.
 *
 */

#include "config.h"
#include <math.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <pthread.h>
#include <glib.h>
#include <gtk/gtk.h>
#include <tremor/ogg.h>
#include <tremor/ivorbiscodec.h>
#include <tremor/ivorbisfile.h>

#include "util.h"
#include "i18n.h"

#include "vorbis.h"
#include "vcedit.h"

static struct vte_struct {
	FILE *in;
	gchar *filename;
} vte;

static void fail(gchar *error);
static gint init_files(vcedit_state *state);
static gint close_files(vcedit_state *state);

extern pthread_mutex_t vf_mutex;

/* From mpg123.c, as no standardized Ogg Vorbis genres exists. */
static const gchar *vorbis_genres[] =
{
	N_("Blues"), N_("Classic Rock"), N_("Country"), N_("Dance"),
	N_("Disco"), N_("Funk"), N_("Grunge"), N_("Hip-Hop"),
	N_("Jazz"), N_("Metal"), N_("New Age"), N_("Oldies"),
	N_("Other"), N_("Pop"), N_("R&B"), N_("Rap"), N_("Reggae"),
	N_("Rock"), N_("Techno"), N_("Industrial"), N_("Alternative"),
	N_("Ska"), N_("Death Metal"), N_("Pranks"), N_("Soundtrack"),
	N_("Euro-Techno"), N_("Ambient"), N_("Trip-Hop"), N_("Vocal"),
	N_("Jazz+Funk"), N_("Fusion"), N_("Trance"), N_("Classical"),
	N_("Instrumental"), N_("Acid"), N_("House"), N_("Game"),
	N_("Sound Clip"), N_("Gospel"), N_("Noise"), N_("Alt"),
	N_("Bass"), N_("Soul"), N_("Punk"), N_("Space"),
	N_("Meditative"), N_("Instrumental Pop"),
	N_("Instrumental Rock"), N_("Ethnic"), N_("Gothic"),
	N_("Darkwave"), N_("Techno-Industrial"), N_("Electronic"),
	N_("Pop-Folk"), N_("Eurodance"), N_("Dream"),
	N_("Southern Rock"), N_("Comedy"), N_("Cult"),
	N_("Gangsta Rap"), N_("Top 40"), N_("Christian Rap"),
	N_("Pop/Funk"), N_("Jungle"), N_("Native American"),
	N_("Cabaret"), N_("New Wave"), N_("Psychedelic"), N_("Rave"),
	N_("Showtunes"), N_("Trailer"), N_("Lo-Fi"), N_("Tribal"),
	N_("Acid Punk"), N_("Acid Jazz"), N_("Polka"), N_("Retro"),
	N_("Musical"), N_("Rock & Roll"), N_("Hard Rock"), N_("Folk"),
	N_("Folk/Rock"), N_("National Folk"), N_("Swing"),
	N_("Fast-Fusion"), N_("Bebob"), N_("Latin"), N_("Revival"),
	N_("Celtic"), N_("Bluegrass"), N_("Avantgarde"),
	N_("Gothic Rock"), N_("Progressive Rock"),
	N_("Psychedelic Rock"), N_("Symphonic Rock"), N_("Slow Rock"),
	N_("Big Band"), N_("Chorus"), N_("Easy Listening"),
	N_("Acoustic"), N_("Humour"), N_("Speech"), N_("Chanson"),
	N_("Opera"), N_("Chamber Music"), N_("Sonata"), N_("Symphony"),
	N_("Booty Bass"), N_("Primus"), N_("Porn Groove"),
	N_("Satire"), N_("Slow Jam"), N_("Club"), N_("Tango"),
	N_("Samba"), N_("Folklore"), N_("Ballad"), N_("Power Ballad"),
	N_("Rhythmic Soul"), N_("Freestyle"), N_("Duet"),
	N_("Punk Rock"), N_("Drum Solo"), N_("A Cappella"),
	N_("Euro-House"), N_("Dance Hall"), N_("Goa"),
	N_("Drum & Bass"), N_("Club-House"), N_("Hardcore"),
	N_("Terror"), N_("Indie"), N_("BritPop"), N_("Negerpunk"),
	N_("Polsk Punk"), N_("Beat"), N_("Christian Gangsta Rap"),
	N_("Heavy Metal"), N_("Black Metal"), N_("Crossover"),
	N_("Contemporary Christian"), N_("Christian Rock"),
	N_("Merengue"), N_("Salsa"), N_("Thrash Metal"),
	N_("Anime"), N_("JPop"), N_("Synthpop")
};

static gchar* get_comment(vorbis_comment *vc, gchar *label)
{
	gchar *tag;
	if (vc && (tag = vorbis_comment_query(vc, label, 0)) != NULL)
		return convert_from_utf8(tag);
	else
		return g_strdup("");
}

static char** get_comment_list(vorbis_comment *vc)
{
	int i;
	char **strv;

	strv = g_new0(char*, vc->comments + 1);
	for (i = 0; i < vc->comments; i++)
	{
		g_message(vc->user_comments[i]);
		strv[i] = g_strdup(vc->user_comments[i]);
	}

	return strv;
}

static char** add_tag(char **list, char *label, char *tag)
{
	char **ptr = list, *reallabel = g_strconcat(label, "=", NULL);

	g_strstrip(tag);
	if (strlen(tag) == 0)
		tag = NULL;
	/*
	 * There can be several tags with the same label.  We clear
	 * them all.
	 */
	while (*ptr != NULL)
	{
		if (!g_strncasecmp(reallabel, *ptr, strlen(reallabel)))
		{
			g_free(*ptr);
			if (tag != NULL)
			{
				tag = convert_to_utf8(tag);
				*ptr = g_strconcat(reallabel, tag, NULL);
				g_free(tag);
				tag = NULL;
				ptr++;
			}
			else
			{
				char **str;
				for (str = ptr; *str; str++)
					*str = *(str + 1);
			}
			
		}
		else
			ptr++;
	}
	if (tag)
	{
		int i = 0;
		for (ptr = list; *ptr; ptr++)
			i++;
		list = g_renew(char*, list, i + 2);
		tag = convert_to_utf8(tag);
		list[i] = g_strconcat(reallabel, tag, NULL);
		list[i + 1] = NULL;
		g_free(tag);
	}
	g_free(reallabel);

	return list;
}

static void add_list(vorbis_comment *vc, char **comments)
{
//	while (*comments)
//		vorbis_comment_add(vc, *comments++);
}


static void fail(gchar *error)
{
	gchar *errorstring;
	errorstring = g_strdup_printf(_("An error occured:\n%s"), error);
	
	xmms_show_message(
		_("Error!"),
		errorstring,
		_("Ok"), FALSE, NULL, NULL);

	g_free(errorstring);
	return;
}



static gint init_files(vcedit_state *state)
{
	if ((vte.in = fopen(vte.filename, "rb")) == NULL)
		return -1;

	if (vcedit_open(state, vte.in) < 0)
	{
		fclose(vte.in);
		return -1;
	}
	return 0;
}

static gint close_files(vcedit_state *state)
{
	int retval = 0, ofh;
	char *tmpfn;
	FILE* out;
	
	tmpfn = g_strdup_printf("%s.XXXXXX", vte.filename);

	if ((ofh = mkstemp(tmpfn)) < 0)
	{
		g_free(tmpfn);
		fclose(vte.in);
		return -1;
	}

	if ((out = fdopen(ofh, "wb")) == NULL)
	{
		close(ofh);
		remove(tmpfn);
		g_free(tmpfn);
		fclose(vte.in);
		return -1;
	}

	if (vcedit_write(state, out) < 0)
	{
		g_warning("vcedit_write: %s", state->lasterror);
		retval = -1;
	}

	fclose(vte.in);

	if (fclose(out) != 0)
		retval = -1;

	if (retval < 0 || rename(tmpfn, vte.filename) < 0)
	{
		remove(tmpfn);
		retval = -1;
	}

	g_free(tmpfn);
	return retval;
}


static void label_set_text(GtkWidget * label, char *str, ...)
{
}

/***********************************************************************/

void vorbis_file_info_box(char *fn)
{
}
