/*****************************************************
*
*  nvpanel_gui.c
*
*  Description - Code for the GUI (GTK based) app for
*  audio control panel. This make use of common lib for
*  talking with the driver
*
*  Copyright(c) 2002-2003 NVIDIA Corporation
*
******************************************************
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>
#include <unistd.h>
#include <string.h>
#include <stdio.h>

#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include "../commonlib/commlib.h"

/* Include all the xpms */
#include "nvicon_xpm.h"
#include "Spkr2_xpm.h"
#include "Spkr4_xpm.h"
#include "Spkr6_xpm.h"
/*#include "Headphone_xpm.h"*/
#include "NVBanner_xpm.h"
#include "nforce_xpm.h"


enum spkr_settings {
    HEAD_PHONE  = 0,
    RESERVED_1,
    SPKR_2      = 2,
    RESERVED_2,
    SPKR_4,
    RESERVED_3,
    SPKR_6,
    SPKR_MAXNODE
};

static struct spkr_name{
    const char* name;
    GtkWidget  *Spkrimage;
}spkr_name[SPKR_MAXNODE] = {
    [HEAD_PHONE] = {"Headphone",NULL},
    [RESERVED_1] = {"",0},
    [SPKR_2]     = {"2 Speakers",NULL},
    [RESERVED_2] = {"",0},
    [SPKR_4]     = {"4 Speakers",NULL},
    [RESERVED_3] = {"",0},
    [SPKR_6]     = {"6 Speakers",NULL},
};

enum rec_settings {
    REC_MIC = 0,
    REC_CD,
    REC_VIDEO,
    REC_AUX,
    REC_LINE_IN ,
    REC_STEREOMIX,
    /*  REC_MONOMIX,*/
    REC_MAXNODE
};

static struct recchannels{
    const char* name;
    int   recmask;
}recchannels[REC_MAXNODE] = {
    [REC_MIC]       = {"Microphone",SOUND_MASK_MIC},
    [REC_CD]        = {"CD Audio",SOUND_MASK_CD},
    [REC_VIDEO]     = {"Video",SOUND_MASK_VIDEO},
    [REC_AUX]       = {"Aux",SOUND_MASK_LINE1},
    [REC_LINE_IN]   = {"Line In",SOUND_MASK_LINE},
    [REC_STEREOMIX] = {"Stereo Mix",SOUND_MASK_IGAIN},
    /*[REC_MONOMIX] = {"Mono Mix",SOUND_MASK_PHONEOUT}, */
};

enum vol_settings {
    VOL_MASTER = 0,
    VOL_LINEIN,
    VOL_CD,
    VOL_AUX,
    VOL_PHONE,
    VOL_VIDEO,
    VOL_MIC = 6,
    VOL_IGAIN,
    VOL_PRELEFT,
    VOL_PRERIGHT,
    VOL_PRERLEFT,
    VOL_PRERRIGHT,
    VOL_PRECENTER,
    VOL_PRESUB,
    VOL_MAXNODE
};

static struct volregs{
    const char* name;
    int   regindex;
    GtkWidget *scale;
    int      range;
    GtkWidget *mutebutton;
    int      mutestatus;
}volregs[VOL_MAXNODE] = {
    [VOL_MASTER]     = {"Master",SOUND_MIXER_VOLUME,NULL,0,NULL,0},
    [VOL_LINEIN]     = {"Line In",SOUND_MIXER_LINE,NULL,0,NULL,0},
    [VOL_CD]         = {"CD ",SOUND_MIXER_CD,NULL,0,NULL,0},
    [VOL_AUX]        = {"Aux",SOUND_MIXER_LINE1,NULL,0,NULL,0},
    [VOL_PHONE]      = {"Phone",SOUND_MIXER_PHONEIN,NULL,0,NULL,0},
    [VOL_VIDEO]      = {"Video",SOUND_MIXER_VIDEO,NULL,0,NULL,0},
    [VOL_MIC]        = {"Microphone",SOUND_MIXER_MIC,NULL,0,NULL,0},
    [VOL_IGAIN]      = {" Record ",SOUND_MIXER_IGAIN,NULL,0,NULL,0},
    [VOL_PRELEFT]    = {"Left",NV_MIXER_PRELEFT,NULL,0,NULL,0},
    [VOL_PRERIGHT]   = {"Right",NV_MIXER_PRERIGHT,NULL,0,NULL,0},
    [VOL_PRERLEFT]   = {"Rear Left",NV_MIXER_PRERLEFT,NULL,0,NULL,0},
    [VOL_PRERRIGHT]  = {"Rear Right",NV_MIXER_PRERRIGHT,NULL,0,NULL,0},
    [VOL_PRECENTER]  = {"Center",NV_MIXER_PRECENTER,NULL,0,NULL,0},
    [VOL_PRESUB]     = {"Sub",NV_MIXER_PRESUB,NULL,0,NULL,0},
};

enum infodetails {
    INFO_MAN = 0,
    INFO_LINUX,
    INFO_DRIVER,
    INFO_CODEC,
    INFO_MAXNODE,
};

static struct info_page{
    const char *name;
    char       *value;
    int        key;
    GtkWidget  *valuewidget;
}info_page[INFO_MAXNODE] = {
    [INFO_MAN]    = {"  Manufacturer ", "  NVIDIA Corporation ", 0, NULL},
    [INFO_LINUX]  = {"  Linux Kernel Version ","  2.x.x.x  ", NV_MIXER_LINUXVER, NULL},
    [INFO_DRIVER] = {"  Driver Version ","  x.x.x.x  ", NV_MIXER_DRIVERVER, NULL},
    [INFO_CODEC]  = {"  Codec Model ","  REALTEK/MSI/SIGMATEL ", NV_MIXER_CODECVER, NULL},
};

static GtkWidget *Mainwindow   = NULL;
static int SpkrSelect          = 0;
static GtkWidget *dbbutton     = NULL;
static int dbstate             = 0;
static GtkWidget *Analogbutton = NULL, *Digitalbutton = NULL;
static GtkWidget *Swaplinein   = NULL, *Swapmicin     = NULL;

/*************************************************** 
* Function to create the pixmap from xpm 
*****************************************************/
static GtkWidget* create_nvpixmap(const char ** data)
{
    GdkPixbuf *pixbuf = NULL;
    GtkWidget *image  = NULL;
    pixbuf = gdk_pixbuf_new_from_xpm_data(data);
    if(pixbuf) {
        image = gtk_image_new_from_pixbuf(pixbuf);
        g_object_unref(G_OBJECT(pixbuf));
    }
    return image;
}

/*************************************************** 
* Function to create the dialog for error 
*****************************************************/
static void do_dialog(gchar * str)
{
    GtkWidget *dialog  = NULL;
    dialog = gtk_message_dialog_new((GtkWindow *) Mainwindow,
                                GTK_DIALOG_DESTROY_WITH_PARENT,
                                GTK_MESSAGE_ERROR,
                                GTK_BUTTONS_NONE,
                                str);
    gtk_dialog_run(GTK_DIALOG(dialog));
    gtk_widget_destroy(dialog);

}
/*************************************************** 
*  Callback function used to set the speaker properly
*  and also to show the right speaker setting xpm
****************************************************/
static void
on_spkrmode_change(GtkEditable     *editable,
                  gpointer         user_data)
{
    const gchar *entry_value;
    entry_value = gtk_entry_get_text(GTK_ENTRY(editable));

    if(strcmp(entry_value,"") == 0 ) {
        return;
    }
    switch(SpkrSelect){
        case SPKR_2:
        case SPKR_4:
        case SPKR_6:
            if(spkr_name[SpkrSelect].Spkrimage) {
                gtk_container_remove(GTK_CONTAINER(user_data),spkr_name[SpkrSelect].Spkrimage);
                gtk_widget_unref(spkr_name[SpkrSelect].Spkrimage);
                spkr_name[SpkrSelect].Spkrimage = NULL;
            }
        break;
    }
    SpkrSelect = 0;

    while( SpkrSelect < SPKR_MAXNODE) {
        if(strcmp(entry_value,spkr_name[SpkrSelect].name) == 0) {
          break;
        }
        SpkrSelect += 1;
    }

    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }

    if(set_speaker_select(SpkrSelect) != -1 ) {
    
        switch(SpkrSelect)
        {
            case SPKR_2:
                spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr2_xpm);
                break;
            case SPKR_4:
                spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr4_xpm);
                break;
            case SPKR_6:
                spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr6_xpm);
                break;
        }

        if(spkr_name[SpkrSelect].Spkrimage) {
            gtk_container_add(GTK_CONTAINER(user_data),spkr_name[SpkrSelect].Spkrimage);
            gtk_widget_ref(spkr_name[SpkrSelect].Spkrimage);
            gtk_widget_show(spkr_name[SpkrSelect].Spkrimage);
        }

    }

    free_mixer();
}

/*************************************************** 
* Function to initialize the speaker widget 
***************************************************/
static int Init_Speaker(GtkWidget *combentry ,GtkWidget * frame)
{
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }

    spkr_name[SpkrSelect].Spkrimage = NULL;
    SpkrSelect =  get_speaker_select();

    switch(SpkrSelect){
        case SPKR_2:
            spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr2_xpm);
            break;
        case SPKR_4:
            spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr4_xpm);
            break;
        case SPKR_6:
            spkr_name[SpkrSelect].Spkrimage = create_nvpixmap((const char **)Spkr6_xpm);
            break;
    }
    
    if(spkr_name[SpkrSelect].Spkrimage) {
        gtk_container_add(GTK_CONTAINER(frame),spkr_name[SpkrSelect].Spkrimage);
        gtk_widget_ref(spkr_name[SpkrSelect].Spkrimage);
        gtk_entry_set_text(GTK_ENTRY(combentry),(gpointer)(spkr_name[SpkrSelect].name));
    }

    free_mixer();

    return 0;
}

/******************************************************* 
* Function to set/reset linein swap for S. left & right 
*******************************************************/ 
void
on_swaplinein_clicked ( GtkButton * button,
                      gpointer    user_data)
{
    int state = GTK_TOGGLE_BUTTON(button)->active;
    
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    if(set_swaplinein(state) == -1) {
        do_dialog(" Unable to swap linein for S. left  right \n");
    }
    
    free_mixer();
}

/******************************************************* 
* Function to set/reset micin swap for center & lfe
*******************************************************/ 
void
on_swapmicin_clicked ( GtkButton * button,
                      gpointer    user_data)
{
    int state = GTK_TOGGLE_BUTTON(button)->active;
    
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    if(set_swapmicin(state) == -1) {
        do_dialog(" Unable to swap micin for center & lfe \n");
    }
    
    free_mixer();
}

/*************************************************** 
* Function to set/reset the Analog output 
***************************************************/ 
void
on_analogbutton_clicked ( GtkButton * button,
                      gpointer    user_data)
{
    int state = GTK_TOGGLE_BUTTON(button)->active;
    
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    if(set_analogout(state) == -1) {
        do_dialog(" Unable to set Analog out \n");
    }
    
    free_mixer();
}

/*************************************************** 
* Function to set/reset the Digital output
***************************************************/  
void
on_digitalbutton_clicked ( GtkButton * button,
                      gpointer    user_data)
{
    int state = GTK_TOGGLE_BUTTON(button)->active;
    
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    if(set_digitalout(state) == -1) {
        do_dialog(" Unable to set Digital out \n");
        return;
    }
    
    free_mixer();
}

/*************************************************** 
* Function to verify whether it is realtek codec
***************************************************/ 
static int Verify_Realtekcodec()
{
	int found = 0, retType;
	char infovalue[20] = {0};

	if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }
	memset(infovalue,0,strlen(infovalue));
    get_nvinfo(NV_MIXER_CODECVER,infovalue,&retType);
    if(retType == 1) { 
        found = 1;
    }
	free_mixer();
    return found;
}

/*************************************************** 
* Function to get the output state 
***************************************************/ 
static int Init_outputstate()
{
    int state = 0;
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }
    state = get_analogout();
    if( state) {
        gtk_button_clicked((GtkButton * )Analogbutton);
    }

    state = get_digitalout();
    if( state) {
        gtk_button_clicked((GtkButton * )Digitalbutton);
    }
    
	/* do the following calls only if realtek codec */
    if(Swaplinein) {
	    state = get_swaplinein();
	    if( state) {
            gtk_button_clicked((GtkButton * )Swaplinein);
        }
    }
    if(Swapmicin) {
	    state = get_swapmicin();
	    if( state) {
            gtk_button_clicked((GtkButton * )Swapmicin);
        }
    }

    free_mixer();

    return 0;
}

/*************************************************** 
* Function to mute premix based on master mute 
***************************************************/ 
void
on_mastermute_clicked  (GtkButton       *button,
                       gpointer         user_data)
{
    int index = 0;
    int mutestate = GTK_TOGGLE_BUTTON(button)->active;
    
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }

    if( set_mute(volregs[VOL_MASTER].regindex, mutestate) != -1 ) {
            volregs[index].mutestatus = mutestate;
    }
   
    free_mixer();
}

/***************************************************************** 
* Function to scale the premix volume based on Master Scale 
****************************************************************/
void
on_masterscale_changed( GtkRange        *range,
                    gpointer         user_data)
{
    int regindex = 0;
    gdouble prevalue = 0, newvalue = 0;
    gdouble value  = gtk_range_get_value( range);
    
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }

    /* check the premix volume and change the range as well */
    for(regindex = VOL_PRELEFT; regindex < (VOL_PRESUB + 1); regindex++) {

        prevalue  = gtk_range_get_value((GtkRange *) volregs[regindex].scale);
        newvalue  = (prevalue * value ) / 100;  
        volregs[regindex].range = (int)newvalue;
        if(volregs[regindex].mutestatus) {
           newvalue = 0;
        }
        if(set_common_volregister(volregs[regindex].regindex, (int)newvalue) == -1) {
           do_dialog(" Unable to set volume \n");
        }

    }

    volregs[VOL_MASTER].range = (int) value;
    free_mixer();
    
}

/*************************************************** 
* Function to enable / disable mic boost 
***************************************************/ 
void
on_micboost_clicked ( GtkButton * button,
                      gpointer    user_data)
{
    dbstate = GTK_TOGGLE_BUTTON(button)->active;
    
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    if(set_mic_boost(dbstate) == -1) {
       do_dialog(" Unable to set Mic Boost \n");
    }
    
    free_mixer();
}

/*************************************************** 
* Function to disable/enable mute button 
***************************************************/ 
void
on_mutebutton_clicked  (GtkButton       *button,
                       gpointer         user_data)
{
    int index = -1, value;
    int mutestate = GTK_TOGGLE_BUTTON(button)->active;
    index = (int)((long)user_data);
    
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }
    
    if(mutestate) {
        value = 0;
    }else {
        value = volregs[index].range;
    }
    
    if(set_common_volregister(volregs[index].regindex, (int)value) != -1) {
      volregs[index].mutestatus = mutestate;
    }
    
    if((dbstate) && (!mutestate)) {
        if(set_mic_boost(dbstate) == -1) {
            do_dialog(" Unable to set Boost \n");
        }
    }
    
    free_mixer();
}

/*************************************************** 
* function to change the volume as such 
***************************************************/ 
void
on_volume_changed ( GtkRange        *range,
                    gpointer         user_data)
{
    int index = -1;
    gdouble val  = gtk_range_get_value( range);
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }

    index = (int)((long)user_data);

    /* check for premix volumes - scale it with master */
    if((index >= VOL_PRELEFT) || (index <= VOL_PRESUB)) {
        val *= gtk_range_get_value( (GtkRange *)volregs[VOL_MASTER].scale);
        val /= 100;
    }

    volregs[index].range = (int)val;
    if(volregs[index].mutestatus) {
       val = 0;
    }

    if(set_common_volregister(volregs[index].regindex, (int)val) == -1) {
       do_dialog(" Unable to change Volume \n");
    }

    free_mixer();
}

/***************************************************************** 
* Function to Initialize the mixer - read the register and set 
*****************************************************************/
static int Init_commonvolreg()
{
    int volreg = VOL_LINEIN, value = 0, mute = 0;
    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }
    while(volreg < VOL_MAXNODE)
    {
        value = get_common_volregister(volregs[volreg].regindex, &mute);
        value &= 0xff;
        if((volreg < VOL_IGAIN) && (value == 0)) {
            /* set the default volume */
            volregs[volreg].mutestatus = 1;
            value = 0x43;
            gtk_button_clicked((GtkButton * )volregs[volreg].mutebutton);
        }
        gtk_range_set_value((GtkRange *)volregs[volreg].scale,(gdouble)value);
        volregs[volreg].range = (int)value;
        volreg += 1;
    }
    
    /* check for the Mic boost variable */
    dbstate =   get_common_volregister(NV_MIXER_MICBOOST,&mute);

    if(dbstate) {
        gtk_button_clicked((GtkButton * )dbbutton);
    }

    /* check for master mute */
    mute = get_mute(volregs[VOL_MASTER].regindex);
    if(mute) {
        gtk_button_clicked((GtkButton * )volregs[VOL_MASTER].mutebutton);
    }
    
    free_mixer();

    return 0;
}

/*************************************************** 
* Function to change the recording inputs
***************************************************/ 
void
on_recmode_change(GtkEditable     *editable,
                  gpointer         user_data)
{
    const gchar *entry_value;
    int  rectype = 0;
    entry_value = gtk_entry_get_text(GTK_ENTRY(editable));

    while( rectype < REC_MAXNODE) {
        if(strcmp(entry_value,recchannels[rectype].name) ==  0) {
          break;
        }
        rectype += 1;
    }

    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return;
    }

    switch(rectype){
        case REC_MIC:
        case REC_CD:
        case REC_VIDEO:
        case REC_AUX:
        case REC_LINE_IN:
        case REC_STEREOMIX:
        /*case REC_MONOMIX:*/
            if(set_record_channel(recchannels[rectype].recmask) == -1) {
                  do_dialog(" Unable to change Record input \n");
        
            }
            break;
    }

    free_mixer();
}

/*************************************************** 
* Function to set the rec widget initially 
***************************************************/ 
static int Init_recselect(GtkWidget *combentry)
{
    int rec_select = -1, rectype = 0;

    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }
    rec_select = get_record_channel();
    
    while(rectype < REC_MAXNODE) {
        if(rec_select ==  recchannels[rectype].recmask) {
          break;
        }
        rectype += 1;
    }
    
    if(rectype < REC_MAXNODE) {
        gtk_entry_set_text(GTK_ENTRY(combentry),(gpointer)(recchannels[rectype].name));
    }
    
    free_mixer();

    return 0;
}

/******************************************************** 
* Function to get the information about the hardware 
*******************************************************/ 
static int Get_Info()
{
    int  index = 0, retType = 0;
    char infovalue[20] = {0};

    /*open the mixer now */
    if(open_mixer() == -1 ) {
        do_dialog(" Nvaudio not Loaded \n");
        return -1;
    }
   
    for(index = INFO_LINUX; index < INFO_MAXNODE; index++)   {
    
        /* clear the info array */
        memset(infovalue,0,strlen(infovalue));
        get_nvinfo(info_page[index].key,infovalue,&retType);
        if(0 != strcmp(infovalue,"")) { 
            gtk_label_set_text((GtkLabel*)info_page[index].valuewidget,(const char*)infovalue);
        }
    }

    free_mixer();

    return 0;
}


/*************************************************** 
* Function to destroy the window 
***************************************************/ 
static gboolean delete ( GtkWidget *widget,
                         GtkWidget *event,
                         gpointer  data)
{
    if(Mainwindow)
        Mainwindow = NULL;
    gtk_main_quit();
    return FALSE;
}

/*************************************************** 
* Function to create the window and layout 
***************************************************/ 

GtkWidget*
create_Mainwindow (void)
{
    if(!Mainwindow)   {

        GtkWidget *table, *notebook;
        GtkWidget *Mainframe, *Spkrframe, *Infoframe;
        GtkWidget *MainPage, *SpkrPage, *InfoPage;
        GtkWidget *mainvbox1, *mainhbox1, *mainhbox2, *bannerimage, *nvimage;
        GtkWidget *recframe, *recvbox, *reccombo;
        GtkWidget *outframe, *outhbox, *outtable,*label, *separator;
        GtkWidget *micframe, *mictable;
        GtkWidget *spkrvbox1, *spkrhbox1, *spkrhbox2;
        GtkWidget *spkroutframe, *spkrnvframe, *logoimage, *spkrouttable;
        GtkWidget *spkrmode, *spkrcombo;
        GtkWidget *premixframe, *spkrsetframe, *premixtable;
        GList     *spkrcombo_items = NULL, *reccombo_items = NULL;
        GtkWidget *infotable, *infologo, *infohead, *infosep;
        GdkPixbuf *pixbuf = NULL;
        int regindex = 0, hindex = 0, vindex = 0;
         

        /* Create Main Window */
        Mainwindow = gtk_window_new (GTK_WINDOW_TOPLEVEL);
        gtk_window_set_title (GTK_WINDOW (Mainwindow), "NVAUDIO");
        g_signal_connect(Mainwindow,"destroy",
                G_CALLBACK(delete),&Mainwindow);

        gtk_container_set_border_width(GTK_CONTAINER(Mainwindow),1);
        pixbuf  = gdk_pixbuf_new_from_xpm_data((const char **)nvicon_xpm);
        gtk_window_set_icon((GtkWindow *)Mainwindow,pixbuf);
        g_object_unref(G_OBJECT(pixbuf));

        /* create a table and attach the notebook */
        table = gtk_table_new(1,1,FALSE);
        gtk_container_add(GTK_CONTAINER(Mainwindow),table);

        /* create notebook */
        notebook = gtk_notebook_new();
        gtk_notebook_set_tab_pos(GTK_NOTEBOOK(notebook),GTK_POS_TOP);
        gtk_table_attach_defaults(GTK_TABLE(table),notebook,0,6,0,1);
        gtk_container_set_border_width( GTK_CONTAINER(notebook), 1);

        /*create mainframe page , label and attach to notebook*/
        /******** MAIN PAGE STARTS ******************/
        Mainframe = gtk_frame_new(NULL);
        MainPage = gtk_label_new("Main");
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook),Mainframe,MainPage);

        mainvbox1 = gtk_vbox_new(FALSE,2);
        gtk_container_add(GTK_CONTAINER(Mainframe),mainvbox1);

        mainhbox1 = gtk_hbox_new(FALSE,2);
        gtk_box_pack_start(GTK_BOX (mainvbox1), mainhbox1, FALSE, TRUE, 10);

        bannerimage = create_nvpixmap((const char **)NVBanner_xpm);
        gtk_container_add(GTK_CONTAINER(mainhbox1),bannerimage);

        nvimage = create_nvpixmap((const char **)nforce_xpm);
        gtk_container_add(GTK_CONTAINER(mainhbox1),nvimage);

        mainhbox2 = gtk_hbox_new(FALSE,2);
        gtk_box_pack_start(GTK_BOX (mainvbox1), mainhbox2, FALSE, FALSE, 0);

        recframe = gtk_frame_new(" Input ");
        gtk_box_pack_start(GTK_BOX(mainhbox2),recframe, FALSE, FALSE , 0);
        gtk_container_set_border_width(GTK_CONTAINER(recframe), 2);

        recvbox = gtk_vbox_new(FALSE, 2);
        gtk_container_add(GTK_CONTAINER(recframe),recvbox);
        label = gtk_label_new(volregs[VOL_IGAIN].name);
        gtk_box_pack_start(GTK_BOX(recvbox),label,FALSE,FALSE,0);
        gtk_misc_set_alignment (GTK_MISC (label), 0.07, 0.7);
        gtk_misc_set_padding (GTK_MISC (label), 1, 5);

        reccombo = gtk_combo_new();
        gtk_box_pack_start (GTK_BOX (recvbox), reccombo, FALSE, FALSE, 3);
        gtk_container_set_border_width (GTK_CONTAINER (reccombo), 3);
        gtk_combo_set_value_in_list (GTK_COMBO (reccombo), TRUE, TRUE);
        gtk_combo_set_use_arrows(GTK_COMBO (reccombo), TRUE);
        gtk_entry_set_editable((GtkEntry *)GTK_COMBO(reccombo)->entry, FALSE);

        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_LINE_IN].name);
        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_MIC].name);
        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_CD].name);
        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_VIDEO].name);
        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_AUX].name);
/*         reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_MONOMIX].name);*/
        reccombo_items = g_list_append (reccombo_items, (gpointer) recchannels[REC_STEREOMIX].name);
        gtk_combo_set_popdown_strings (GTK_COMBO (reccombo), reccombo_items);
        g_list_free (reccombo_items);

        volregs[VOL_IGAIN].scale = gtk_vscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (50, 0, 100, 1, 0, 0)));
        gtk_box_pack_start (GTK_BOX (recvbox), volregs[VOL_IGAIN].scale, FALSE, FALSE, 0);
        gtk_widget_set_size_request (volregs[VOL_IGAIN].scale, 35, 150);
        gtk_scale_set_draw_value (GTK_SCALE (volregs[VOL_IGAIN].scale), FALSE);
        gtk_range_set_inverted (GTK_RANGE (volregs[VOL_IGAIN].scale), TRUE);

        if(Init_recselect((gpointer)GTK_COMBO(reccombo)->entry) == -1) {
            Mainwindow = NULL;
            return Mainwindow;
        }

        g_signal_connect((gpointer)GTK_COMBO(reccombo)->entry,"changed" ,
                G_CALLBACK (on_recmode_change),NULL);

        outframe = gtk_frame_new (" Output");
        gtk_box_pack_start (GTK_BOX (mainhbox2), outframe, TRUE, TRUE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (outframe), 2);

        outhbox = gtk_hbox_new (FALSE, 2);
        gtk_container_add (GTK_CONTAINER (outframe), outhbox);

        outtable = gtk_table_new (3, 12, FALSE);
        gtk_box_pack_start (GTK_BOX (outhbox), outtable, FALSE, FALSE, 0);

        for (regindex = 0, hindex = 0; regindex < VOL_MIC; regindex++, hindex += 2) {

            label =  gtk_label_new (volregs[regindex].name);

            gtk_table_attach (GTK_TABLE (outtable), label, hindex, hindex+1, 0, 1,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions) (0), 2, 0);

            volregs[regindex].scale = gtk_vscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (50, 0, 100, 1, 0, 0)));

            gtk_table_attach (GTK_TABLE (outtable), volregs[regindex].scale, hindex, hindex+1, 1, 2,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);
            gtk_scale_set_draw_value (GTK_SCALE (volregs[regindex].scale), FALSE);
            gtk_range_set_inverted (GTK_RANGE (volregs[regindex].scale), TRUE);

            volregs[regindex].mutebutton = gtk_check_button_new_with_mnemonic ("Mute");
            gtk_table_attach (GTK_TABLE (outtable), volregs[regindex].mutebutton, hindex, hindex+1, 2, 3,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions) (GTK_FILL), 2, 0);

            if(regindex < VOL_VIDEO) {
                separator = gtk_vseparator_new();
                gtk_table_attach (GTK_TABLE (outtable), separator, hindex+1, hindex+2, 0, 2,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);
            }

        }

        /* check do we make it same as master volume register value */
        gtk_range_set_value((GtkRange*)(volregs[VOL_MASTER].scale),100);
        volregs[VOL_MASTER].range = 100;

        micframe = gtk_frame_new(volregs[VOL_MIC].name);
        gtk_box_pack_start (GTK_BOX (outhbox), micframe, TRUE, TRUE, 0);
        gtk_container_set_border_width (GTK_CONTAINER (micframe), 10);
        mictable = gtk_table_new (2, 2, FALSE);
        gtk_container_add(GTK_CONTAINER(micframe),mictable);

        volregs[VOL_MIC].scale = gtk_vscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (50, 0, 100, 1, 0, 0)));
        gtk_table_attach (GTK_TABLE (mictable), volregs[VOL_MIC].scale, 0, 1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);
        gtk_scale_set_draw_value (GTK_SCALE (volregs[VOL_MIC].scale), FALSE);
        gtk_range_set_inverted (GTK_RANGE (volregs[VOL_MIC].scale), TRUE);

        volregs[VOL_MIC].mutebutton = gtk_check_button_new_with_mnemonic ("Mute");
        gtk_table_attach (GTK_TABLE (mictable), volregs[VOL_MIC].mutebutton, 0, 1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);

        dbbutton = gtk_check_button_new_with_mnemonic ("Boost");
        gtk_table_attach (GTK_TABLE (mictable), dbbutton, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_FILL), 2, 0);

        /******** MAIN PAGE ENDS ******************/

        /*create Spkrframe page , label and attach to notebook*/
        /******** SPEAKER SETUP PAGE STARTS ******************/


        Spkrframe = gtk_frame_new(NULL);
        SpkrPage = gtk_label_new("Speaker Setup");

        gtk_notebook_append_page(GTK_NOTEBOOK(notebook),Spkrframe,SpkrPage);

        spkrvbox1 = gtk_vbox_new(FALSE, 4);
        gtk_container_add(GTK_CONTAINER(Spkrframe),spkrvbox1);
        gtk_container_set_border_width(GTK_CONTAINER(spkrvbox1), 2 );

        /* Add two hbox to main Vbox */
        spkrhbox1 = gtk_hbox_new(FALSE, 4);
        gtk_box_pack_start_defaults( GTK_BOX(spkrvbox1), spkrhbox1);

        /* two frame to first hbox */
        /****    Output frame starts  ****/

        spkroutframe = gtk_frame_new("Output Settings ");
        gtk_box_pack_start (GTK_BOX (spkrhbox1), spkroutframe, FALSE, FALSE, 0);
        gtk_container_set_border_width(GTK_CONTAINER(spkroutframe), 6);
        gtk_frame_set_shadow_type (GTK_FRAME (spkroutframe), GTK_SHADOW_ETCHED_IN);

        spkrouttable = gtk_table_new (4, 2, FALSE);
        gtk_container_add (GTK_CONTAINER (spkroutframe), spkrouttable);
        gtk_table_set_col_spacings (GTK_TABLE (spkrouttable), 20);

        spkrmode = gtk_label_new(" Listening Mode ");
        gtk_table_attach(GTK_TABLE( spkrouttable), spkrmode, 0, 1, 0, 1,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions)(0), 2, 10);
        gtk_label_set_justify(GTK_LABEL(spkrmode), GTK_JUSTIFY_LEFT);
        gtk_misc_set_alignment (GTK_MISC (spkrmode), 0, 0.5);
        gtk_misc_set_padding (GTK_MISC (spkrmode), 13, 2);


        /* create the Listen mode combo box */
        spkrcombo = gtk_combo_new();
        gtk_table_attach(GTK_TABLE(spkrouttable), spkrcombo, 0, 1 , 1, 2,
                        (GtkAttachOptions) (GTK_FILL),
                        (GtkAttachOptions)(0), 1, 6);
        gtk_container_set_border_width (GTK_CONTAINER(spkrcombo), 1);
        gtk_combo_set_use_arrows(GTK_COMBO (spkrcombo), TRUE);
        gtk_combo_set_value_in_list (GTK_COMBO (spkrcombo), TRUE, TRUE);
        spkrcombo_items = g_list_append (spkrcombo_items, (gpointer)(spkr_name[SPKR_2].name));
        spkrcombo_items = g_list_append (spkrcombo_items, (gpointer)(spkr_name[SPKR_4].name));
        spkrcombo_items = g_list_append (spkrcombo_items, (gpointer)(spkr_name[SPKR_6].name));
        gtk_combo_set_popdown_strings (GTK_COMBO (spkrcombo), spkrcombo_items);
        gtk_entry_set_editable((GtkEntry *)GTK_COMBO(spkrcombo)->entry, FALSE);
        g_list_free (spkrcombo_items);

        Analogbutton = gtk_check_button_new_with_mnemonic ("Analog Output");
        gtk_table_attach (GTK_TABLE (spkrouttable), Analogbutton, 1, 2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 2);
        
        Digitalbutton = gtk_check_button_new_with_mnemonic ("Digital Output");
        gtk_table_attach (GTK_TABLE (spkrouttable), Digitalbutton, 1, 2, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 0, 2);
         
		if( Verify_Realtekcodec() ) {
            Swaplinein = gtk_check_button_new_with_mnemonic ("Swap LineIn for Surround Left & Right Speakers ");
            gtk_table_attach (GTK_TABLE (spkrouttable), Swaplinein, 0, 1, 2, 3,
				            (GtkAttachOptions) (GTK_FILL),
				            (GtkAttachOptions) (0), 0, 2);
                
            Swapmicin = gtk_check_button_new_with_mnemonic ("Swap MicIn for Center & Lfe Speakers ");
            gtk_table_attach (GTK_TABLE (spkrouttable), Swapmicin, 0, 1, 3, 4,
				            (GtkAttachOptions) (GTK_FILL),
				            (GtkAttachOptions) (0), 0, 2);
		}

        /****    Output frame ends  ****/

        spkrnvframe = gtk_frame_new(NULL);
        gtk_box_pack_start (GTK_BOX (spkrhbox1), spkrnvframe, TRUE, TRUE, 1);
        //gtk_box_pack_start_defaults(GTK_BOX (spkrhbox1), spkrnvframe);
        gtk_container_set_border_width(GTK_CONTAINER(spkrnvframe), 6);
        gtk_frame_set_shadow_type (GTK_FRAME (spkrnvframe), GTK_SHADOW_NONE);

        logoimage = create_nvpixmap((const char **)nforce_xpm);
        gtk_container_add(GTK_CONTAINER(spkrnvframe),logoimage);


        spkrhbox2 = gtk_hbox_new(FALSE, 4);
        gtk_box_pack_start_defaults( GTK_BOX(spkrvbox1), spkrhbox2);

        /* add Premix frame and Spkrlogo frame */
        premixframe  = gtk_frame_new("Premix Volume Levels");
        gtk_container_add(GTK_CONTAINER(spkrhbox2),premixframe);
        gtk_container_set_border_width(GTK_CONTAINER(premixframe), 6);

        premixtable = gtk_table_new (2, 12, FALSE);
        gtk_container_add (GTK_CONTAINER (premixframe), premixtable);
        gtk_container_set_border_width (GTK_CONTAINER (premixtable), 6);
        gtk_table_set_row_spacings (GTK_TABLE (premixtable), 3);
        gtk_table_set_col_spacings (GTK_TABLE (premixtable), 3);

        for( regindex = VOL_PRELEFT, hindex = 0; regindex < VOL_MAXNODE ; regindex++, hindex += 2)
        {
            label = gtk_label_new (volregs[regindex].name);
            gtk_table_attach (GTK_TABLE (premixtable), label, hindex, hindex+1, 1, 2,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (0), 5, 3);
            volregs[regindex].scale = gtk_vscale_new (GTK_ADJUSTMENT (gtk_adjustment_new (50, 0, 100, 1, 0, 0)));
            gtk_table_attach (GTK_TABLE (premixtable), volregs[regindex].scale, hindex, hindex+1, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);
            gtk_scale_set_draw_value (GTK_SCALE (volregs[regindex].scale), FALSE);
            gtk_range_set_inverted (GTK_RANGE (volregs[regindex].scale), TRUE);

            if(regindex < VOL_PRESUB) {
                separator = gtk_vseparator_new();
                gtk_table_attach (GTK_TABLE (premixtable), separator, hindex+1, hindex+2, 0, 1,
                    (GtkAttachOptions) (GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);
            }
        }

        /* get the speaker select from driver and set */
        spkrsetframe = gtk_frame_new(NULL);
        gtk_container_add(GTK_CONTAINER(spkrhbox2),spkrsetframe);
        gtk_frame_set_shadow_type (GTK_FRAME (spkrsetframe), GTK_SHADOW_NONE);
        gtk_widget_set_size_request (spkrsetframe, 120, 120); 

        if(Init_Speaker((gpointer)GTK_COMBO(spkrcombo)->entry,spkrsetframe) == -1) {
            Mainwindow = NULL;
            return Mainwindow; 
        }

        g_signal_connect((gpointer)GTK_COMBO(spkrcombo)->entry,"changed" ,
                        G_CALLBACK (on_spkrmode_change),spkrsetframe);

        if( Init_outputstate() == -1 ) {
            Mainwindow = NULL;
            return Mainwindow;
        }
        g_signal_connect((gpointer)Analogbutton,"clicked",
                        G_CALLBACK(on_analogbutton_clicked),NULL);

        g_signal_connect((gpointer)Digitalbutton,"clicked",
                        G_CALLBACK(on_digitalbutton_clicked),NULL);

        if(Swaplinein)   {
	        g_signal_connect((gpointer)Swaplinein,"clicked",
                        G_CALLBACK(on_swaplinein_clicked),NULL);
        }
    	
        if( Swapmicin) {
	       g_signal_connect((gpointer)Swapmicin,"clicked",
                       G_CALLBACK(on_swapmicin_clicked),NULL);
        }

        if( Init_commonvolreg() == -1) {
            Mainwindow = NULL;
            return Mainwindow;
        }

        g_signal_connect((gpointer)volregs[VOL_MASTER].scale,"value_changed",
                        G_CALLBACK(on_masterscale_changed),NULL);
                           
        g_signal_connect((gpointer)volregs[VOL_MASTER].mutebutton,"clicked",
                        G_CALLBACK(on_mastermute_clicked),NULL);

        for( regindex = VOL_LINEIN; regindex < VOL_MAXNODE; regindex++) {

            g_signal_connect((gpointer)volregs[regindex].scale,"value_changed",
                G_CALLBACK(on_volume_changed), (gpointer)((long)regindex));

            if( regindex < VOL_IGAIN) {
                g_signal_connect ((gpointer) volregs[regindex].mutebutton, "clicked",
                    G_CALLBACK (on_mutebutton_clicked),
                    (gpointer)((long)regindex));
            }
        }

        /* signal the mic boost */
        g_signal_connect ((gpointer) dbbutton, "clicked",
                    G_CALLBACK (on_micboost_clicked),NULL);

        /********* SPEAKER SETUP PAGE ENDS ******************/

        /********* Info PAGE STARTS ******************/
        /*create Info page , label and attach to notebook*/

        Infoframe = gtk_frame_new(NULL);
        InfoPage  = gtk_label_new("Information");
        gtk_notebook_append_page(GTK_NOTEBOOK(notebook),Infoframe,InfoPage);

        infotable = gtk_table_new (11, 2, FALSE);
        gtk_container_add (GTK_CONTAINER (Infoframe), infotable);
        gtk_container_set_border_width (GTK_CONTAINER (infotable), 6);
        gtk_table_set_row_spacings (GTK_TABLE (infotable), 3);
        gtk_table_set_col_spacings (GTK_TABLE (infotable), 3);

        infologo = create_nvpixmap((const char **)nforce_xpm);
        gtk_table_attach (GTK_TABLE (infotable), infologo, 1, 2, 0, 1,
                        (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                        (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);

        infosep = gtk_hseparator_new();
        gtk_table_attach (GTK_TABLE (infotable), infosep, 0, 2, 1, 2,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);

        infohead = gtk_label_new (" Topic ");
        gtk_table_attach (GTK_TABLE (infotable), infohead, 0, 1, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 5, 3);
        infohead = gtk_label_new ("  Value ");
        gtk_table_attach (GTK_TABLE (infotable), infohead, 1, 2, 2, 3,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 5, 3);

        infosep = gtk_hseparator_new();
        gtk_table_attach (GTK_TABLE (infotable), infosep, 0, 2, 3, 4,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);

        for( regindex = 0, vindex = 4; regindex < INFO_MAXNODE; regindex++, vindex += 1) {
            label = gtk_label_new (info_page[regindex].name);
            gtk_table_attach (GTK_TABLE (infotable), label, 0, 1, vindex, vindex+1,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 5, 3);
            info_page[regindex].valuewidget =  gtk_label_new (info_page[regindex].value);
            gtk_table_attach (GTK_TABLE (infotable), info_page[regindex].valuewidget, 1, 2, vindex, vindex+1,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 5, 3);
            
        }   
    
        infosep = gtk_hseparator_new();
        gtk_table_attach (GTK_TABLE (infotable), infosep, 0, 2, 9, 10,
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL),
                    (GtkAttachOptions) (GTK_EXPAND|GTK_FILL), 0, 0);

        Get_Info();

        /********* Info PAGE ENDS ******************/
    }

    if(!GTK_WIDGET_VISIBLE(Mainwindow)) {
        gtk_widget_show_all(Mainwindow);
    }else {
        gtk_widget_destroy(Mainwindow);
        Mainwindow = NULL;
    }

    return Mainwindow;
}

/*************************************************** 
*   MAIN ENTRY 
***************************************************/ 
int main ( int argc, char* argv[])
{
    gtk_init(&argc,&argv);

    Mainwindow = create_Mainwindow();

    if(Mainwindow) {
        gtk_widget_show_all(Mainwindow);
        gtk_main();
    }

    return 0;
}
