
#include <glib.h>
#include <string.h>
#include "config.h"
#include "dynamic_lib.h"

#ifdef HPUX

/*
 * Use shl_load family of functions on HP-UX. HP-UX does not
 * support dlopen on 32-bit PA-RISC executables
 */

#include <dl.h>
#include <errno.h>

gpointer dynamic_lib_open(const gchar * filename)
{
    return shl_load(filename, BIND_DEFERRED, 0);
}

void dynamic_lib_close(gpointer handle)
{
    shl_t h = handle;
    shl_unload(h);
}

gpointer dynamic_lib_find_symbol(gpointer handle, const gchar * symbol)
{
    gpointer gpi;
    shl_t h = handle;
    if ((shl_findsym(&h, symbol, TYPE_PROCEDURE, &gpi)) != 0)
	gpi = NULL;
    return gpi;
}

const gchar *dynamic_lib_error(void)
{
    return strerr(errno);
}

#else

/*
 * Good old libdl
 */

#include <dlfcn.h>

gpointer dynamic_lib_open(const gchar * filename)
{
    return dlopen(filename, RTLD_NOW);
}

void dynamic_lib_close(gpointer handle)
{
    dlclose(handle);
}

gpointer dynamic_lib_find_symbol(gpointer handle, const gchar * symbol)
{
#ifdef SYMBOL_PREFIX
    gchar *sym = g_strconcat(SYMBOL_PREFIX, symbol, NULL);
    gpointer symh = dlsym(handle, sym);
    g_free(sym);
    return symh;
#else
    return dlsym(handle, symbol);
#endif
}

const gchar *dynamic_lib_error(void)
{
    return dlerror();
}

#endif				/* HPUX */
