/***********************************************************************

Original Developed by:

	2002/10/25 start
	Copyright (C) 2002 I-O DATA DEVICE,INC.

Abstract:

	CFXGA driver for ZAURUS(Linux)

Revision History:

	2002/11/25	Ver1.00	
	2003/01/08	Ver1.01	Support SWIVEL_180 and SWIVEL_270.
						Modify RESUME process.

***********************************************************************/

#define	KK_DEBUG		1
#define	USE_MOD_TIMER	0	//(Ver1.01)

/***********************************************************************/
#include <pcmcia/config.h>
#include <pcmcia/k_compat.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/malloc.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <linux/major.h>
#include <linux/fs.h>
#include <linux/delay.h>
//#include <linux/ioctl.h>
#include <asm/io.h>
#include <asm/system.h>
#include <asm/segment.h>
#include <stdarg.h>
#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>
#include <pcmcia/memory.h>
#include <pcmcia/mem_op.h>

#include "cfxga_cs.h"


//
#define DEVICE_NAME			"cfxga"
#define DEVICE_NR(minor)	((minor)>>4)
#define IS_DIRECT(minor)	(((minor)&8)>>3)
#define REGION_AM(minor)	(((minor)&4)>>2)
#define REGION_NR(minor)	((minor)&7)
#define MINOR_NR(dev,dir,attr,rgn) \
(((dev)<<4)+((dir)<<3)+((attr)<<2)+(rgn))


//#define	PCMCIA_DEBUG 6
#ifdef PCMCIA_DEBUG
static int pc_debug = PCMCIA_DEBUG;
MODULE_PARM(pc_debug, "i");
#define DEBUG(n, args...) if(pc_debug>(n)) printk(args) //printk(KERN_DEBUG args)
#else
#define DEBUG(n, args...)
#endif

//====================================================================
// Parameters that can be set with 'insmod'

static int	resume_delay = 50;	//delay times before resume (Ver1.01)

MODULE_PARM(resume_delay, "i");

//====================================================================

//Maximum number of separate memory devices we'll allow
#define MAX_DEV		4
#define	MAX_PART	1

static void			cfxga_config(dev_link_t *link);
static void			cfxga_release(u_long arg);
static int			cfxga_event(event_t event, int priority, event_callback_args_t *args);
static dev_link_t	*cfxga_attach(void);
static void			cfxga_detach(dev_link_t *);

//Each memory region corresponds to a minor device
typedef struct minor_dev_t {		/* For normal regions */
	region_info_t	region;
	memory_handle_t	handle;
	int				open;
	u_int			offset;
} minor_dev_t;

typedef struct direct_dev_t {		/* For direct access */
	int			flags;
	int			open;
	caddr_t		Base;
	u_int		Size;
	u_int		cardsize;
	u_int		offset;
} direct_dev_t;

//CFXGA infomation structure
typedef struct cfxga_t {
	int			mode;
	u_int		w;
	u_int		h;
	u_int		bpp;
	u_int		stride;
	u_int		mem_busy;
	status_t	status;
} cfxga_t;

typedef struct memory_dev_t {
	dev_link_t		link;
	dev_node_t		node;
	direct_dev_t	direct;
	minor_dev_t		minor[2*MAX_PART];
	cfxga_t			cfxga;
} memory_dev_t;

static dev_info_t dev_info = "memory_cs";
static dev_link_t *dev_table[MAX_DEV] = { NULL, /* ... */ };

//Major device numbers for cfxga device  [ 0 - 255 ]
static int major_dev	 = 0;

#if USE_MOD_TIMER
struct timer_list timer_info;
#endif

//====================================================================
//file_operations structure
//====================================================================
static int 			cfxga_ioctl(struct inode *inode, struct file *file, u_int cmd, u_long arg);
static ssize_t		cfxga_read FOPS(struct inode *inode, struct file *file, char *buf, size_t count, loff_t *ppos);
static ssize_t		cfxga_write FOPS(struct inode *inode, struct file *file, const char *buf, size_t count, loff_t *ppos);
static int			cfxga_open(struct inode *inode, struct file *file);
static FS_RELEASE_T	cfxga_close(struct inode *inode, struct file *file);

static struct file_operations cfxga_chr_fops = {
	open:		cfxga_open,
	release:	cfxga_close,
	read:		cfxga_read,
	write:		cfxga_write,
	ioctl:		cfxga_ioctl,
};


//====================================================================
//CFXGA reset engine function
//====================================================================
static void reset_cfxga(void *reg)
{
	writew(0x8080, reg);
	mdelay(25);
	writew(0x0000, reg);
	mdelay(25);
	readw((u_char *)reg + 0x400);
	return;
}

//====================================================================
//CFXGA engine busy check function
//====================================================================
static void blt_busy(void *reg)
{
	int	timeout = BUSY_TIMEOUT;
	u_char	*bltreg, *bltdat;

	bltreg = (u_char *)reg + 0x100;
	bltdat = (u_char *)reg + 0x400;

	//////readw(bltdat);
	while((readb(bltreg) & 0x80 ) && --timeout);
	readw(bltdat);
#if KK_DEBUG
	if(timeout == 0)
	{
		DEBUG(5, "***** B U S Y   R E S E T *****\n");
		reset_cfxga(reg);
	}
#endif
	return;
}

static void blt_start(void *reg)
{
	int	timeout = BLTS_TIMEOUT;
	u_char	*bltreg;

	bltreg = (u_char *)reg + 0x100;
	while(((readb(bltreg) & 0x80) == 0) && --timeout);
	return;
}


//====================================================================
//CFXGA FIFO status check function
//	[ret]
//		0: pass
//		1: timeout error
//====================================================================
static int fifo_check(void *reg, int *nFIFO)
{
	int		timeout;
	u_char	stat;
	u_char	*bltreg;
	
//printk("FIFO=%d ", *nFIFO);
	if(--(*nFIFO) == 0)
	{
		bltreg = (u_char *)reg + 0x100;
		stat = readb(bltreg) & 0x70;
		*nFIFO = 1;
		timeout = FIFO_TIMEOUT;
		if(stat == 0x00)		*nFIFO = 15; //16;
		else if(stat == 0x40)	*nFIFO = 7; //8;
		else if(stat == 0x70)
		{
			//while((readb(bltreg) & 0x10) && --timeout);
			while(readb(bltreg) & 0x10); 
			if(timeout == 0)
			{
				DEBUG(5,"*****  F I F O   R E S E T *****\n");
				reset_cfxga(reg);
				return 1;
			}
		}
	}
	return 0;
}

//====================================================================
//CFXGA FIFO empty check function
//====================================================================
static void fifo_empty(void *reg)
{
	u_char	*bltreg;
	int		timeout;

	timeout = FIFO_TIMEOUT;
	bltreg = (u_char *)reg + 0x100;

	while((readb(bltreg) & 0x10) && --timeout);

	if(timeout == 0)
	{
		DEBUG(5,"***** F I F O   E M P T Y   R E S E T *****\n");
		reset_cfxga(reg);	
	}

	return;
}

//====================================================================
//CFXGA set resolution function
//====================================================================
static void set_reso(u_int mode, memory_dev_t *dev)
{
	int			i;
	u_char		*regadr;
	u_short		*regw;
	pDefReso	pReso;
	u_short		regw60, regw66, regw1fc;

	if(mode >= CFXGA_RESONUM) mode = 0;	//default mode

	//
	pReso = reso_tbl[mode];
	dev->cfxga.mode		= mode;
	dev->cfxga.w		= pReso->reso.cx;
	dev->cfxga.h		= pReso->reso.cy;
	dev->cfxga.bpp		= pReso->reso.bpp;
	dev->cfxga.stride	= pReso->reso.cx * pReso->reso.bpp / 8;

	regadr = dev->direct.Base;
	reset_cfxga(regadr);

DEBUG(1, "set_reso: ChipID = %x\n", readb(regadr));
DEBUG(1, "set_reso: <No=%d> %dx%dx%dHzx%dbpp\n", dev->cfxga.mode,
		 dev->cfxga.w, dev->cfxga.h, pReso->reso.ref, dev->cfxga.bpp);

	i = 0;
	regw = common_reg;
	while(regw[i] != 0xffff)
	{
		writew(regw[i+1], regadr+regw[i]);
		//ksleep(20);
		DEBUG(10, "reg%x=%04x ", regw[i], readw(regadr+regw[i]));
		i += 2;
	}
	DEBUG(10, "\n");
	
	regw   = pReso->reg;
	i = 0;
	while(regw[i] != 0xffff )
	{
		writew(regw[i+1], regadr+regw[i]);
		DEBUG(10, "reg%x=%04x ", regw[i], readw(regadr+regw[i]));
		i += 2;
	}
	DEBUG(10, "\n");
	
	//assumes 16bpp,TVout(Composite/SVideo)
	regw60  = 0x0005;
	regw66  = pReso->reso.cx;
	regw1fc = 0x0006;
	if(pReso->reso.bpp == 8)
	{
		regw60  = 0x0003;
		regw66  = pReso->reso.cx >> 1;
	}
	if(pReso->reso.flags & OUT_VGA)	regw1fc = 0x0002;
	writew(regw60,  regadr+0x60);
	writew(regw66,  regadr+0x66);
	writew(regw1fc, regadr+0x1fc);
	DEBUG(10, "reg60=%04x ",  readw(regadr+0x60));
	DEBUG(10, "reg66=%04x ",  readw(regadr+0x66));
	DEBUG(10, "reg1fc=%04x ", readw(regadr+0x1fc));
	DEBUG(10, "\n");

	mdelay(25);

	dev->cfxga.status.flags |= CFXGA_SCREEN_ON;
	return;
}

//====================================================================
//CFXGA memory to screen engine function
//====================================================================
static void mem_to_screen(copy_info_t *info, memory_dev_t *dev, int swivel)
{
	u_char	*reg, *bltreg, *bltdat;
	u_short	*src, *dst;
	u_short	bltcmd, temp, d_ww, d_hh;
	int		x,y,nFIFO;
	int		sx,sy,ex,ey,dx,dy,ww,hh;

	if(dev->cfxga.mem_busy) return;
	dev->cfxga.mem_busy = 1;

	dx = info->x;
	dy = info->y;
	ww = info->w;
	hh = info->h;

	//clipping check
	if(dx < 0) dx = 0;
	if(dy < 0) dy = 0;
	switch(swivel)
	{
	case SWIVEL_90:
	case SWIVEL_270:	//(Ver1.01)
		if(ww > dev->cfxga.h) ww = dev->cfxga.h;
		if(hh > dev->cfxga.w) hh = dev->cfxga.w;
		d_ww = hh;
		d_hh = ww;
		break;
	case SWIVEL_0:
	case SWIVEL_180:	//(Ver1.01)
	default:
		if(ww > dev->cfxga.w) ww = dev->cfxga.w;
		if(hh > dev->cfxga.h) hh = dev->cfxga.h;
		d_ww = ww;
		d_hh = hh;
		break;
	}

	//
	sx = sy = 0;
	ex = sx + ww;
	ey = sy + hh;

	dst = (u_short *)((dy * dev->cfxga.stride) + (dx * (dev->cfxga.bpp/8)));

	if(dev->cfxga.bpp == 16)
	{
		bltcmd = 0x0180;
		temp   = dev->cfxga.w;
	}
	else
	{
		bltcmd = 0x0080;
		temp   = dev->cfxga.w >> 1;
		ex     = (ex + 1) >> 1;
	}

	reg		= dev->direct.Base;
	bltreg	= reg + 0x100;
	bltdat	= reg + 0x400;

	blt_busy(reg);
	writew(0x000c,		bltreg+0x02);	//rop
	writew(0x0000,		bltreg+0x04);
	writew(0x0000,		bltreg+0x06);
	writew((int)dst,	bltreg+0x08);	//start address
	writew((int)dst>>16,bltreg+0x0a);	//
	writew(temp,		bltreg+0x0c);	//stride
	writew(d_ww-1,		bltreg+0x10);	//width
	writew(d_hh-1,		bltreg+0x12);	//height
	writew(bltcmd,		bltreg+0x00);

DEBUG(3, "swivel=<%d>, sx=%d, ex=%d, sy=%d, ey=%d, s_stride=%d, bpp=%d, sadr=0x%08lx, dadr=0x%08lx\n",
		swivel, sx, ex, sy, ey, info->stride, info->bpp, (u_long)info->adr, (u_long)dst);

	nFIFO = 1;
	blt_start(reg);		//(Ver1.01)
	switch(swivel)
	{
	default:
	case SWIVEL_0:
		for(y = sy; y < ey; y++)
		{
			src = (u_short *)(info->adr + (y * info->stride) + (sx * (info->bpp/8)));
			for(x = sx; x < ex; x++)
			{
				writew(*src++, bltdat);
				if(fifo_check(reg, &nFIFO)) goto mem_exit;
			}
		}
		break;
	case SWIVEL_90:
		for(x = ex-1; x >= sx; x--)
		{
			src = (u_short *)(info->adr + (x * (info->bpp/8)));
			for(y = sy; y < ey; y++)
			{
				writew(*src, bltdat);
				src += (info->stride >> 1);
				if(fifo_check(reg, &nFIFO)) goto mem_exit;
			}
		}
		break;
	case SWIVEL_180:	//(Ver1.01)
		for(y = ey-1; y >= sy; y--)
		{
			src = (u_short *)(info->adr + (y * info->stride) + ((ex-1) * (info->bpp/8)));
			for(x = ex-1; x >= sx; x--)
			{
				writew(*src--, bltdat);
				if(fifo_check(reg, &nFIFO)) goto mem_exit;
			}
		}
		break;
	case SWIVEL_270:	//(Ver1.01)
		for(x = sx; x < ex; x++)
		{
			src = (u_short *)(info->adr + ((ey-1) * info->stride) + (x * (info->bpp/8)));
			for(y = ey-1; y >= sy; y--)
			{
				writew(*src, bltdat);
				src -= (info->stride >> 1);
				if(fifo_check(reg, &nFIFO)) goto mem_exit;
			}
		}
		break;
	}

mem_exit:
	//readw(bltdat);

	dev->cfxga.mem_busy = 0;
	return;
}

//====================================================================
//CFXGA clear screen function
//====================================================================
static void clear_screen(memory_dev_t *dev)
{
	u_char	*reg, *bltreg;
	u_short	bltcmd, temp, fgcolor;

	if(dev->cfxga.mem_busy)	return;
	dev->cfxga.mem_busy = 1;

	reg		= dev->direct.Base;
	bltreg	= reg + 0x100;

	fgcolor = 0;
	bltcmd  = (dev->cfxga.bpp == 16) ? 0x180 : 0x0080;
	temp    = (dev->cfxga.bpp == 16) ? dev->cfxga.w : (dev->cfxga.w >> 1);

	blt_busy(reg);
	writew(0x0c0c,	bltreg+0x02);	//ROP = P, Operation = Solid Fill
	writew(0x0000,	bltreg+0x04);
	writew(0x0000,	bltreg+0x06);
	writew(0x0000,	bltreg+0x08);
	writew(0x0000,	bltreg+0x0a);
	writew(temp,	bltreg+0x0c);
	writew(fgcolor,	bltreg+0x18);
	writew(dev->cfxga.w-1,bltreg+0x10);
	writew(dev->cfxga.h-1,bltreg+0x12);
	writew(bltcmd,	bltreg+0x00);
	//readw(reg+0x400);

	dev->cfxga.mem_busy = 0;
	return;
}

//====================================================================
//CFXGA screen swith function
//====================================================================
static void screen_off(memory_dev_t *dev)
{
	u_char	*reg;

	reg = dev->direct.Base;
	writew(0x0000, reg+0x1fc);
	dev->cfxga.status.flags &= ~CFXGA_SCREEN_ON;
	return;
}

#if USE_MOD_TIMER
//====================================================================
//CFXGA set resolution and clear screen function
//====================================================================
static void set_reso_clear(memory_dev_t *dev)
{
	set_reso(dev->cfxga.mode, dev);
	clear_screen(dev);
	return;
}
#endif

//====================================================================
//
//====================================================================
static void cs_error(client_handle_t handle, int func, int ret)
{
	error_info_t err = { func, ret };
	CardServices(ReportError, handle, &err);
}

//====================================================================
//Attach function
//====================================================================
static dev_link_t *cfxga_attach(void)
{
	memory_dev_t	*dev;
	dev_link_t		*link;
	client_reg_t	client_reg;
	int				i, ret;

	DEBUG(1, "-- cfxga_attach() start --\n");

	for (i = 0; i < MAX_DEV; i++)
		if (dev_table[i] == NULL) break;

	if (i == MAX_DEV) {
		printk(KERN_NOTICE "cfxga_cs: no devices available\n");
		return NULL;
	}

	/* Create new memory card device */
	dev = kmalloc(sizeof(*dev), GFP_KERNEL);
	if (!dev) return NULL;
	memset(dev, 0, sizeof(*dev));
	link = &dev->link;
	link->priv = dev;

	link->release.function = &cfxga_release;
	link->release.data = (u_long)link;
	dev_table[i] = link;

	//Initialise (Ver1.01)
	dev->cfxga.mode = -1;
#if USE_MOD_TIMER
	init_timer(&timer_info);
#endif

	/* Register with Card Services */
	client_reg.dev_info = &dev_info;
	client_reg.Attributes = INFO_MEM_CLIENT | INFO_CARD_SHARE; 
	//client_reg.Attributes = INFO_IO_CLIENT | INFO_CARD_SHARE;
	client_reg.EventMask =
		CS_EVENT_RESET_PHYSICAL | CS_EVENT_CARD_RESET |
		CS_EVENT_CARD_INSERTION | CS_EVENT_CARD_REMOVAL |
		CS_EVENT_PM_SUSPEND | CS_EVENT_PM_RESUME;
	client_reg.event_handler = &cfxga_event;
	client_reg.Version = 0x0210;
	client_reg.event_callback_args.client_data = link;
	ret = CardServices(RegisterClient, &link->handle, &client_reg);
	if (ret != 0) {
		printk(KERN_DEBUG "RegisterClient Error\n");
		cs_error(link->handle, RegisterClient, ret);
		cfxga_detach(link);
		return NULL;
	}

	return link;
} /* cfxga_attach */

//====================================================================
//Detach function
//====================================================================
static void cfxga_detach(dev_link_t *link)
{
	memory_dev_t *dev = link->priv;
	int nd;

	DEBUG(1, "-- cfxga_detach(0x%p) start --\n", link);

	/* Verify device address */
	for (nd = 0; nd < MAX_DEV; nd++)
		if (dev_table[nd] == link) break;

	if (nd == MAX_DEV)	return;

#if USE_MOD_TIMER
	del_timer(&timer_info);
#endif
	del_timer(&link->release);
	if (link->state & DEV_CONFIG)
	{
		cfxga_release((u_long)link);
		if (link->state & DEV_STALE_CONFIG)
		{
			link->state |= DEV_STALE_LINK;
			return;
		}
	}

	if (link->handle)
		CardServices(DeregisterClient, link->handle);

	/* Unlink device structure, free bits */
	dev_table[nd] = NULL;
	kfree(dev);

} /* cfxga_detach */


//====================================================================
//Config function
//  This function is called after CARD_INSERTION event is received.
//====================================================================
#define CS_CHECK(fn, args...) \
while ((last_ret=CardServices(last_fn=(fn), args))!=0) goto cs_failed

static void cfxga_config(dev_link_t *link)
{
	memory_dev_t *dev = link->priv;
	cs_status_t status;
	win_req_t req;
	int nd, last_ret, last_fn;
	u_char	chipID;

    DEBUG(1, "-- cfxga_config(0x%p) start --\n", link);

	/* Configure card */
	link->state |= DEV_CONFIG;

	for (nd = 0; nd < MAX_DEV; nd++)
		if (dev_table[nd] == link) break;

	/* Allocate a small memory window for direct access */
	req.Attributes = WIN_MEMORY_TYPE_CM | WIN_ENABLE | WIN_DATA_WIDTH_16;
	req.Base = 	req.Size = 0;
	req.AccessSpeed = MEM_SPEED;
	link->win = (window_handle_t)link->handle;
	CS_CHECK(RequestWindow, &link->win, &req);
	DEBUG(10, "Base = 0x%08lx, Size = %d\n", req.Base, req.Size);

	/* Get write protect status */
	CS_CHECK(GetStatus, link->handle, &status);

	dev->direct.Base = ioremap_nocache(req.Base, req.Size);
	dev->direct.Size = req.Size;
	dev->direct.cardsize = 0;

	DEBUG(10, "direct.Base = 0x%08lx, direct.Size = %d\n", (u_long)dev->direct.Base, dev->direct.Size); 

	chipID = readb(dev->direct.Base) & 0xfc;
	DEBUG(3, "ChipID = 0x%x\n", chipID);
	if(chipID != 0x1c)
	{
		DEBUG(3, "***** ChipID error *****");
		goto chip_err;
	}

	sprintf(dev->node.dev_name, "cfxga%d", nd);
	dev->node.major = major_dev;
	dev->node.minor = MINOR_NR(nd, 0, 0, 0);
	link->dev = &dev->node;
	link->state &= ~DEV_CONFIG_PENDING;

DEBUG(3, "dev_name = %s, major = %x, minor = %x\n", dev->node.dev_name, dev->node.major, dev->node.minor);
DEBUG(3, "CFXGA_RESONUM = %d\n", CFXGA_RESONUM);
	//
	dev->cfxga.status.version = DRVIER_VERSION;
	dev->cfxga.status.flags   = CFXGA_INSTALLED;

	printk("\n");
	return;

cs_failed:
	cs_error(link->handle, last_fn, last_ret);
chip_err:
	cfxga_release((u_long)link);
	return;
} /* cfxga_config */

//====================================================================
//release funtion
//====================================================================
static void cfxga_release(u_long arg)
{
	dev_link_t *link = (dev_link_t *)arg;
	int nd;

	DEBUG(1, "-- cfxga_release(0x%p) start --\n", link);

	for (nd = 0; nd < MAX_DEV; nd++)
		if (dev_table[nd] == link) break;

DEBUG(10, "nd index = %d\n", nd);

	if (link->open) {
		DEBUG(1, "cfxga_cs: release postponed, 'mem%d'"" still open\n", nd);
		link->state |= DEV_STALE_CONFIG;
		return;
	}

	link->dev = NULL;
	if (link->win) {
		memory_dev_t *dev = link->priv;
		DEBUG(10, "iounmap adr = 0x%08lx\n", (u_long)dev->direct.Base);
		iounmap(dev->direct.Base);
		CardServices(ReleaseWindow, link->win);
	}
	link->state &= ~DEV_CONFIG;

	if (link->state & DEV_STALE_LINK)
	{
		cfxga_detach(link);
	}

	//DEBUG(1, "cfxga_release(0x%p) exit ~~~~~~~~~~~~~~~~\n", link);

} /* cfxga_release */

//====================================================================
//Event function
//====================================================================
static int cfxga_event(event_t event, int priority,
		       event_callback_args_t *args)
{
	dev_link_t *link  = args->client_data;
	memory_dev_t *dev = link->priv;

	DEBUG(1, "-- cfxga_event(0x%06x) start --\n", event);

	switch (event) {
	case CS_EVENT_CARD_REMOVAL:
		DEBUG(1, "=== CS_EVENT_CARD_REMOVAL ===\n");
		link->state &= ~DEV_PRESENT;
		dev->cfxga.status.flags &= ~CFXGA_INSTALLED;
		if (link->state & DEV_CONFIG)
			mod_timer(&link->release, jiffies + HZ/20);
		break;
	case CS_EVENT_CARD_INSERTION:
		DEBUG(1, "=== CS_EVENT_CARD_INSERTL ===\n");
		link->state |= DEV_PRESENT | DEV_CONFIG_PENDING;
		cfxga_config(link);
		break;
	case CS_EVENT_ERASE_COMPLETE:
		DEBUG(1, "=== CS_EVENT_ERASE_COMPLETE ===\n");
		break;
	case CS_EVENT_PM_SUSPEND:
		DEBUG(1, "=== CS_EVENT_PM_SUSPEND ===\n");
		link->state |= DEV_SUSPEND;
		/* Fall through... */
	case CS_EVENT_RESET_PHYSICAL:
		DEBUG(1, "=== CS_EVENT_RESET_PHYSICAL ===\n");
		/* get_lock(link); */
		break;
	case CS_EVENT_PM_RESUME:
		DEBUG(1, "=== CS_EVENT_PM_RESUME ===\n");
		mdelay(resume_delay);	//(Ver1.01)
		link->state &= ~DEV_SUSPEND;
		/* Fall through... */
	case CS_EVENT_CARD_RESET:
		DEBUG(1, "=== CS_EVENT_CARD_RESET ===\n");
		if((u_int)dev->cfxga.mode < CFXGA_RESONUM)	//(Ver1.01)
		{
#if USE_MOD_TIMER
			timer_info.function = set_reso_clear;
			timer_info.data = dev;
			mod_timer(&timer_info, jiffies + (HZ*3));
#else
			set_reso(dev->cfxga.mode, dev);
			clear_screen(dev);
#endif
		}
		/* free_lock(link); */
		break;
	}
	return 0;
} /* cfxga_event */

//====================================================================
//Open function
//====================================================================
static int cfxga_open(struct inode *inode, struct file *file)
{
	dev_link_t		*link;
	memory_dev_t	*dev;
	int minor = MINOR(inode->i_rdev);

	DEBUG(1, "-- cfxga_open(%d) start --\n", minor);

	link = dev_table[DEVICE_NR(minor)];
	if(!DEV_OK(link))	return -ENODEV;

	dev = (memory_dev_t *)link->priv;

	//CFXGA installed check
	if(!(dev->cfxga.status.flags & CFXGA_INSTALLED))
	{
		printk(KERN_NOTICE "open: CFXGA NOT INSTALLED");
		return -ENODEV;
	}

	dev->direct.open++;
	link->open++;
	MOD_INC_USE_COUNT;
	return 0;
} /* cfxga_open */

//====================================================================
//Close function
//====================================================================
static FS_RELEASE_T cfxga_close(struct inode *inode,  struct file *file)
{
	dev_link_t		*link;
	memory_dev_t	*dev;
	//minor_dev_t		*minor_dev;
	int minor = MINOR(inode->i_rdev);

	DEBUG(1, "-- cfxga_close(%d) start --\n", minor);

	link = dev_table[DEVICE_NR(minor)];
	dev = (memory_dev_t *)link->priv;

	dev->direct.open--;
	link->open--;
	MOD_DEC_USE_COUNT;
	
	//(Ver1.01)
	if(!MOD_IN_USE)
	{
		DEBUG(1,"@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@@\n");
		screen_off(dev);
		dev->cfxga.mode = -1;
	}
	return (FS_RELEASE_T)0;
} /* cfxga_close */

//====================================================================
//Read function
//  << non-support >>
//====================================================================
static ssize_t cfxga_read FOPS(struct inode *inode, struct file *file, char *buf, 
								size_t count, loff_t *ppos)
{
	return 0;
} /* cfxga_read */

//====================================================================
//Write function
//  << non-support >>
//====================================================================
static ssize_t cfxga_write FOPS(struct inode *inode,
				 struct file *file, const char *buf,
				 size_t count, loff_t *ppos)
{
	return 0;
} /* cfxga_write */


//====================================================================
//ioctl function
//====================================================================
static int cfxga_ioctl(struct inode *inode, struct file *file,
						u_int cmd, u_long arg)
{
	int minor = MINOR(inode->i_rdev);
	dev_link_t		*link;
	memory_dev_t	*dev;
	minor_dev_t		*minor_dev;
	u_int			size, mode;
	copy_info_t		info1;
	copy2_info_t	info2;
	pDefReso		pReso;
	void			*vadr;
	int				ret = 0;

DEBUG(1, "-- cfxga_ioctl start --\n");

	link = dev_table[DEVICE_NR(minor)];
	if (!DEV_OK(link))	return -ENODEV;

	dev = (memory_dev_t *)link->priv;
	minor_dev = &dev->minor[REGION_NR(minor)];

	//cmd check!!
	size = (cmd & IOCSIZE_MASK) >> IOCSIZE_SHIFT;
	if(cmd & IOC_IN)
	{
		ret = verify_area(VERIFY_READ, (char *)arg, size);
		if(ret) return ret;
	}
	if(cmd & IOC_OUT)
	{
		ret = verify_area(VERIFY_WRITE, (char *)arg, size);
		if(ret) return ret;
	}

	//
	if( (!(dev->cfxga.status.flags & CFXGA_INSTALLED)) && (cmd != CFXGA_GET_STATUS) )
		return -1;

	switch(cmd) {
	case CFXGA_GET_STATUS:
		DEBUG(1, "*** CFXGA_GET_STATUS ***\n");
		copy_to_user((status_t *)arg, &dev->cfxga.status, sizeof(status_t));
		break;
	case CFXGA_GET_RESONUM:
		DEBUG(1, "*** CFXGA_GET_RESONUM ***\n");
		__put_user(CFXGA_RESONUM ,(u_int *)arg);
		break;
	case CFXGA_GET_RESOINFO:
		DEBUG(1, "*** CFXGA_GET_RESOINFO ***\n");
		__get_user(mode, (u_int *)arg);		//(reso_info_t *)arg->mode_no
		if(mode >= CFXGA_RESONUM) mode = 0;	//default mode
		pReso = reso_tbl[mode];
		copy_to_user((reso_t *)(arg+sizeof(u_int)), &pReso->reso, sizeof(reso_t));
		break;
	case CFXGA_SET_RESO:
		DEBUG(1, "*** CFXGA_SET_RESO ***\n");
		__get_user(mode, (u_int *)arg);
		if(mode >= CFXGA_RESONUM) mode = 0;	//default mode
		set_reso(mode, dev);
		break;
	case CFXGA_COPY_FB:
		DEBUG(1, "*** CFXGA_COPY_FB ***\n");
		copy_from_user(&info1, (copy_info_t *)arg, sizeof(copy_info_t));
		mem_to_screen(&info1, dev, SWIVEL_0);
		break;
	case CFXGA_COPY_FB2:
		DEBUG(1, "*** CFXGA_COPY_FB2 ***\n");
		copy_from_user(&info2, (copy2_info_t *)arg, sizeof(copy2_info_t));
		DEBUG(10, "adr = 0x%08lx, size = 0x%08lx\n", info2.phys_adr, info2.size);
		info1.x = info2.x;
		info1.y = info2.y;
		info1.w = info2.w;
		info1.h = info2.h;
		info1.bpp = info2.bpp;
		info1.stride = info2.stride;
		vadr = ioremap_nocache(info2.phys_adr, info2.size);
		info1.adr = vadr + info2.offset;
		mem_to_screen(&info1, dev, info2.swivel);
		iounmap((caddr_t)(vadr));
		break;
	case CFXGA_CLEAR_SCREEN:
		DEBUG(1, "*** CFXGA_CLEAR_SCREEN ***\n");
		clear_screen(dev);
		break;
	case CFXGA_SCREEN_OFF:
		DEBUG(1, "*** CFXGA_SCREEN_OFF ***\n");
		screen_off(dev);
		break;
	default:
		ret = -EINVAL;
	}

	return ret;
} /* cfxga_ioctl */

//====================================================================
//init_module function
//====================================================================
static int __init init_cfxga_cs(void)
{
	servinfo_t serv;
	int i;

	DEBUG(1, "\n\n\n");
	DEBUG(1, "cfxga_cs: init_module() %d\n", DRVIER_VERSION);

	CardServices(GetCardServicesInfo, &serv);
	if(serv.Revision != CS_RELEASE_CODE)
	{
		printk(KERN_NOTICE "cfxga_cs: Card Services release "
				"does not match!\n");
		return -1;
	}

	register_pccard_driver(&dev_info, &cfxga_attach, &cfxga_detach);

	for(i = MAX_CHRDEV-1; i > 0; i--)
	{
		if (register_chrdev(i, DEVICE_NAME, &cfxga_chr_fops) == 0)
			break;
	}

	if(i == 0)
	{
		printk(KERN_NOTICE "cfxga_cs: unable to grab a device #\n");
		return -ENODEV;
	}

	major_dev = i;
	printk(KERN_DEBUG "<<<< Init_Module PASS >>>\n");
	return 0;
}

//====================================================================
//release_module function
//====================================================================
static void __exit exit_cfxga_cs(void)
{
	int i;
	dev_link_t *link;

	DEBUG(1, "cfxga_cs: cleanup_module()\n");
	DEBUG(1, "*********************************\n\n\n");

	unregister_pccard_driver(&dev_info);
	if (major_dev != 0)
		unregister_chrdev(major_dev, DEVICE_NAME);

	for (i = 0; i < MAX_DEV; i++)
	{
		link = dev_table[i];
		if (link)
		{
			if (link->state & DEV_CONFIG)
				cfxga_release((u_long)link);
			cfxga_detach(link);
		}
	}
}

module_init(init_cfxga_cs);
module_exit(exit_cfxga_cs);

