/*
 * gdict Dictionary for UNIX
 * Copyright (C) 2003 Alexander Zapjagaev <frustum@frustum.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "dictwindow.h"

/*  
 */
static void window_destroy(GtkWidget *widget,dictwindow_t *dictwin) {
	int i;
	gtk_widget_destroy(dictwin->window);
	for(i = 0; i < dictwin->num_dicts; i++) free(dictwin->dicts[i]);
	free(dictwin->dicts);
	free(dictwin->dict_loads);
	free(dictwin);
}

/*  
 */
static void clist_select_row(GtkWidget *widget,int row,int column,
	GdkEventButton *event,dictwindow_t *dictwin) {
	char name[2048];
	if(event && event->type == GDK_2BUTTON_PRESS) {
		dictwin->dict_loads[row] = !dictwin->dict_loads[row];
		strcpy(name,strrchr(dictwin->dicts[row],'/') + 1);
		if(dictwin->dict_loads[row]) strcat(name,_("  [loaded]"));
		gtk_clist_set_text(GTK_CLIST(widget),row,0,name);
	}
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(dictwin->check_button),dictwin->dict_loads[row]);
	dictwin->row = row;
}

/*   
 */
static void check_button_focus_in_event(GtkWidget *widget,GdkEventFocus *event,dictwindow_t *dictwin) {
	dictwin->check_button_focus = 1;
}

/*    
 */
static void check_button_focus_out_event(GtkWidget *widget,GdkEventFocus *event,dictwindow_t *dictwin) {
	dictwin->check_button_focus = 0;
}

/*    
 */
static void check_button_toggled(GtkWidget *widget,dictwindow_t *dictwin) {
	char name[2048];
	if(!dictwin->check_button_focus) return;
	if(!dictwin->num_dicts) return;
	dictwin->dict_loads[dictwin->row] = !dictwin->dict_loads[dictwin->row];
	strcpy(name,strrchr(dictwin->dicts[dictwin->row],'/') + 1);
	if(dictwin->dict_loads[dictwin->row]) strcat(name,_("  [loaded]"));
	gtk_clist_set_text(GTK_CLIST(dictwin->clist),dictwin->row,0,name);
}

/*  
 */
static void dict_window_reload(dictwindow_t *dictwin) {
	int i,j,num_dicts = 0,count = 0;
	char buf[1024];
	mainwindow_t *mainwin = dictwin->mainwin;
	for(i = 0; i < dictwin->num_dicts; i++) {
		if(!dictwin->dict_loads[i]) continue;
		num_dicts++;
		for(j = 0; j < mainwin->num_dicts; j++) {
			if(!strcmp(dictwin->dicts[i],mainwin->dicts[j])) {
				count++;
				break;
			}
		}
	}
	if(num_dicts != count || mainwin->num_dicts != count) {
		main_window_clist_free(mainwin);
		main_window_dict_free(mainwin);
		for(i = 0; i < dictwin->num_dicts; i++) {
			if(dictwin->dict_loads[i]) main_window_dict_load(mainwin,dictwin->dicts[i]);
		}
		main_window_clist_load(mainwin);
	}
	/* number of words */
	sprintf(buf,_("Number of words: %d"),mainwin->num_words);
	gtk_label_set(GTK_LABEL(dictwin->num_words_label),buf);
}

/*   ok
 */
static void button_ok_clicked(GtkWidget *widget,dictwindow_t *dictwin) {
	dict_window_reload(dictwin);
	gtk_widget_destroy(dictwin->window);
}

/*   cancel
 */
static void button_cancel_clicked(GtkWidget *widget,dictwindow_t *dictwin) {
	gtk_widget_destroy(dictwin->window);
}

/*   apply
 */
static void button_apply_clicked(GtkWidget *widget,dictwindow_t *dictwin) {
	dict_window_reload(dictwin);
}

/*  
 */
static void dict_window_find(dictwindow_t *dictwin,char *dirname) {
	DIR *dir;
	struct dirent *dirent;
	char name[2048];
	dir = opendir(dirname);
	if(!dir) {
		fprintf(stderr,"error open dir \"%s\"\n",dirname);
		return;
	}
	while((dirent = readdir(dir)) != NULL) {
		struct stat st;
		if(!strcmp(dirent->d_name,".") || !strcmp(dirent->d_name,"..")) continue;
		sprintf(name,"%s/%s",dirname,dirent->d_name);
		stat(name,&st);
		if(S_ISREG(st.st_mode)) {
			int i;
			char *d;
			dictwin->num_dicts++;
			dictwin->dicts = realloc(dictwin->dicts,sizeof(char*) * dictwin->num_dicts);
			dictwin->dicts[dictwin->num_dicts - 1] = strdup(name);
			dictwin->dict_loads = realloc(dictwin->dict_loads,sizeof(int) * dictwin->num_dicts);
			dictwin->dict_loads[dictwin->num_dicts - 1] = 0;
			d = strrchr(name,'/') + 1;
			for(i = 0; i < dictwin->mainwin->num_dicts; i++) {
				if(!strcmp(name,dictwin->mainwin->dicts[i])) {
					strcat(d,_("  [loaded]"));
					dictwin->dict_loads[dictwin->num_dicts - 1] = 1;
					break;
				}
			}
			gtk_clist_append(GTK_CLIST(dictwin->clist),&d);
		}
	}
	closedir(dir);
}

/*     
 */
dictwindow_t *dict_window_create(mainwindow_t *mainwin) {
	dictwindow_t *dictwin;
	GtkWidget *window;
	GtkWidget *vbox;
	GtkWidget *scrolled_win;
	GtkWidget *clist;
	GtkWidget *check_button;
	GtkWidget *num_words_label;
	GtkWidget *hbox;
	GtkWidget *hbox_;
	GtkWidget *button;
	char *title[] = { _("Dictionaries") };
	char buf[1024];
	
	dictwin = calloc(1,sizeof(dictwindow_t));
	
	/* window */
	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window),_("Dictionaries"));
	gtk_signal_connect(GTK_OBJECT(window),"destroy",
		GTK_SIGNAL_FUNC(window_destroy),dictwin);
	
	/* vbox */
	vbox = gtk_vbox_new(FALSE,0);
	gtk_container_set_border_width(GTK_CONTAINER(window),10);
	gtk_container_add(GTK_CONTAINER(window),vbox);
	
	/* scrolled_win */
	scrolled_win = gtk_scrolled_window_new(NULL,NULL);
	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(scrolled_win),
		GTK_POLICY_AUTOMATIC,GTK_POLICY_AUTOMATIC);
	gtk_widget_set_usize(scrolled_win,320,200);
	gtk_box_pack_start(GTK_BOX(vbox),scrolled_win,TRUE,TRUE,0);

	/* clist */
	clist = gtk_clist_new_with_titles(1,title);
	gtk_clist_set_shadow_type(GTK_CLIST(clist),GTK_SHADOW_OUT);
	gtk_clist_set_selection_mode(GTK_CLIST(clist),GTK_SELECTION_BROWSE);
	gtk_clist_set_column_width(GTK_CLIST(clist),0,150);
	gtk_container_add(GTK_CONTAINER(scrolled_win),clist);
	gtk_signal_connect(GTK_OBJECT(clist),"select_row",
		GTK_SIGNAL_FUNC(clist_select_row),dictwin);
	
	/* hbox */
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox,FALSE,TRUE,0);
	
	/* check button */
	check_button = gtk_check_button_new_with_label(_("Load dictionary"));
	gtk_box_pack_start(GTK_BOX(hbox),check_button,FALSE,FALSE,0);
	gtk_signal_connect(GTK_OBJECT(check_button),"focus_in_event",
		GTK_SIGNAL_FUNC(check_button_focus_in_event),dictwin);
	gtk_signal_connect(GTK_OBJECT(check_button),"focus_out_event",
		GTK_SIGNAL_FUNC(check_button_focus_out_event),dictwin);
	gtk_signal_connect(GTK_OBJECT(check_button),"toggled",
		GTK_SIGNAL_FUNC(check_button_toggled),dictwin);
	
	/* number of words */
	sprintf(buf,_("Number of words: %d"),mainwin->num_words);
	num_words_label = gtk_label_new(buf);
	gtk_box_pack_end(GTK_BOX(hbox),num_words_label,FALSE,FALSE,0);
	
	/* hbox_ */
	hbox_ = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(vbox),hbox_,FALSE,TRUE,0);
	
	/* hbox */
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(hbox_),hbox,TRUE,TRUE,0);
	
	/* hbox */
	hbox = gtk_hbox_new(FALSE,0);
	gtk_box_pack_start(GTK_BOX(hbox_),hbox,TRUE,TRUE,0);
	
	/* ok button */
	button = gtk_button_new_with_label(_("Ok"));
	gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
		GTK_SIGNAL_FUNC(button_ok_clicked),dictwin);
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	gtk_widget_grab_default(button);
	
	/* cancel button */
	button = gtk_button_new_with_label(_("Cancel"));
	gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
		GTK_SIGNAL_FUNC(button_cancel_clicked),dictwin);
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	
	/* apply button */
	button = gtk_button_new_with_label(_("Apply"));
	gtk_box_pack_start(GTK_BOX(hbox),button,TRUE,TRUE,0);
	gtk_signal_connect(GTK_OBJECT(button),"clicked",
		GTK_SIGNAL_FUNC(button_apply_clicked),dictwin);
	GTK_WIDGET_SET_FLAGS(button,GTK_CAN_DEFAULT);
	
	gtk_widget_show_all(window);
	
	dictwin->window = window;
	dictwin->clist = clist;
	dictwin->check_button = check_button;
	dictwin->num_words_label = num_words_label;
	dictwin->mainwin = mainwin;
	
	/*   */
	sprintf(buf,PREFIX"/share/"PACKAGE);
	dict_window_find(dictwin,buf);
	sprintf(buf,"%s/."PACKAGE"/dict",g_get_home_dir());
	dict_window_find(dictwin,buf);
	
	return dictwin;
}
