/* gpsutils.c -- code shared between low-level and high-level interfaces */
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <math.h>
#include <string.h>
#include <stdarg.h>
#define __USE_XOPEN
#include <time.h>

#include "gpsd.h"

double timestamp(void) 
{
    struct timeval tv; 
    gettimeofday(&tv, NULL); 
    return(tv.tv_sec + tv.tv_usec/1e6);
}

time_t mkgmtime(register struct tm *t)
/* struct tm to seconds since Unix epoch */
{
    register unsigned short year;
    register time_t result;
    static const int cumdays[12] =
    {0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334};

    year = 1900 + t->tm_year + t->tm_mon / 12;
    result = (year - 1970) * 365 + cumdays[t->tm_mon % 12];
    result += (year - 1968) / 4;
    result -= (year - 1900) / 100;
    result += (year - 1600) / 400;
    result += t->tm_mday - 1;
    result *= 24;
    result += t->tm_hour;
    result *= 60;
    result += t->tm_min;
    result *= 60;
    result += t->tm_sec;
    return (result);
}

double iso8601_to_unix(char *isotime)
/* ISO8601 UTC to Unix UTC */
{
    char *dp = NULL;
    double usec;
    struct tm tm;

    dp = strptime(isotime, "%Y-%m-%dT%H:%M:%S", &tm);
    if (*dp == '.')
	usec = strtod(dp, NULL);
    else
	usec = 0;
    return mkgmtime(&tm) + usec;
}

char *unix_to_iso8601(double fixtime, char *isotime)
/* Unix UTC time to ISO8601, no timezone adjustment */
{
    struct tm when;
    double integral, fractional;
    time_t intfixtime;
    int slen;

    fractional = modf(fixtime, &integral);
    intfixtime = (time_t)integral;
    gmtime_r(&intfixtime, &when);

    strftime(isotime, 28, "%Y-%m-%dT%H:%M:%S", &when);
    slen = strlen(isotime);
    sprintf(isotime + slen, "%.1f", fractional);
    memcpy(isotime+slen, isotime+slen+1, strlen(isotime+slen+1));
    strcat(isotime, "Z");
    return isotime;
}

/*
 * The 'week' part of GPS dates are specified in weeks since 0000 on 06 
 * January 1980, with a rollover at 1024.  At time of writing the last 
 * rollover happened at 0000 22 August 1999.  Time-of-week is in seconds.
 *
 * This code copes with both conventional GPS weeks and the "extended"
 * 15-or-16-bit version with no wraparound that apperas in Zodiac
 * chips and is supposed to appear in the Geodetic Navigation
 * Information (0x29) packet of SiRF chips.  Some SiRF firmware versions
 * (notably 231) actually ship the wrapped 10-bit week, despite what
 * the protocol reference claims.
 *
 * Note: This time will need to be corrected for leap seconds.
 * That's what the offset argument is for.
 */
#define GPS_EPOCH	315964800		/* GPS epoch in Unix time */
#define SECS_PER_WEEK	(60*60*24*7)		/* seconds per week */
#define GPS_ROLLOVER	(1024*SECS_PER_WEEK)	/* rollover period */

double gpstime_to_unix(int week, double tow, int offset)
{
    double fixtime;

    if (week >= GPS_ROLLOVER)
	fixtime = GPS_EPOCH + (week * SECS_PER_WEEK) + tow;
    else {
	time_t now, last_rollover;
	time(&now);
	last_rollover = GPS_EPOCH+((now-GPS_EPOCH)/GPS_ROLLOVER)*GPS_ROLLOVER;
	fixtime = last_rollover + (week * SECS_PER_WEEK) + tow;
    }
    return fixtime + offset;
}

#define Deg2Rad(n)	((n) * DEG_2_RAD)

static double CalcRad(double lat)
/* earth's radius of curvature in meters at specified latitude.*/
{
    const double a = 6378.137;
    const double e2 = 0.081082 * 0.081082;
    // the radius of curvature of an ellipsoidal Earth in the plane of a
    // meridian of latitude is given by
    //
    // R' = a * (1 - e^2) / (1 - e^2 * (sin(lat))^2)^(3/2)
    //
    // where a is the equatorial radius,
    // b is the polar radius, and
    // e is the eccentricity of the ellipsoid = sqrt(1 - b^2/a^2)
    //
    // a = 6378 km (3963 mi) Equatorial radius (surface to center distance)
    // b = 6356.752 km (3950 mi) Polar radius (surface to center distance)
    // e = 0.081082 Eccentricity
    double sc = sin(Deg2Rad(lat));
    double x = a * (1.0 - e2);
    double z = 1.0 - e2 * sc * sc;
    double y = pow(z, 1.5);
    double r = x / y;

    return r * 1000.0;	// Convert to meters
}

double earth_distance(double lat1, double lon1, double lat2, double lon2)
/* distance in meters between two points specified in degrees. */
{
    double x1 = CalcRad(lat1) * cos(Deg2Rad(lon1)) * sin(Deg2Rad(90-lat1));
    double x2 = CalcRad(lat2) * cos(Deg2Rad(lon2)) * sin(Deg2Rad(90-lat2));
    double y1 = CalcRad(lat1) * sin(Deg2Rad(lon1)) * sin(Deg2Rad(90-lat1));
    double y2 = CalcRad(lat2) * sin(Deg2Rad(lon2)) * sin(Deg2Rad(90-lat2));
    double z1 = CalcRad(lat1) * cos(Deg2Rad(90-lat1));
    double z2 = CalcRad(lat2) * cos(Deg2Rad(90-lat2));
    double a = (x1*x2 + y1*y2 + z1*z2)/pow(CalcRad((lat1+lat2)/2),2);
    // a should be in [1, -1] but can sometimes fall outside it by
    // a very small amount due to rounding errors in the preceding
    // calculations (this is prone to happen when the argument points
    // are very close together).  Thus we constrain it here.
    if (abs(a) > 1) 
	a = 1;
    else if (a < -1) 
	a = -1;
    return CalcRad((lat1+lat2) / 2) * acos(a);
}

/*****************************************************************************

Carl Carter of SiRF supplied this algorithm for computing DOPs from 
a list of visible satellites...

For satellite n, let az(n) = azimuth angle from North and el(n) be elevation.
Let:

    a(k, 1) = sin az(k) * cos el(k)
    a(k, 2) = cos az(k) * cos el(k)
    a(k, 3) = sin el(k)

Then form the line-of-sight matrix A for satellites used in the solution:

    | a(1,1) a(1,2) a(1,3) 1 |
    | a(2,1) a(2,2) a(2,3) 1 |
    |   :       :      :   : |
    | a(n,1) a(n,2) a(n,3) 1 |

And its transpose A~:

    |a(1, 1) a(2, 1) .  .  .  a(n, 1) |
    |a(1, 2) a(2, 2) .  .  .  a(n, 2) |
    |a(1, 3) a(2, 3) .  .  .  a(n, 3) |
    |    1       1   .  .  .     1    |

Compute the covariance matrix (A~*A)^-1, which is guaranteed symmetric:

    | s(x)^2    s(x)*s(y)  s(x)*s(z)  s(x)*s(t) | 
    | s(x)*s(y) s(y)^2     s(y)*s(z)  s(y)*s(t) |
    | s(z)*s(t) s(y)*s(z)  s(z)^2     s(z)*s(t) |
    | s(x)*s(t) s(y)*s(t)  s(z)*s(t)  s(z)^2    |

Then:

GDOP = sqrt(s(x)^2 + s(y)^2 + s(z)^2 + s(t)^2)
TDOP = sqrt(s(t)^2)
PDOP = sqrt(s(x)^2 + s(y)^2 + s(z)^2)
HDOP = sqrt(s(x)^2 + s(y)^2)
VDOP = sqrt(s(y)^2)

Here's how we implement it...

First, each compute element P(i,j) of the 4x4 product A~*A.
If S(k=1,k=n): f(...) is the sum of f(...) as k varies from 1 to n, then
applying the definition of matrix product tells us: 

P(i,j) = S(k=1,k=n): B(i, k) * A(k, j) + 1

But because B is the transpose of A, this reduces to 

P(i,j) = S(k=1,k=n): A(k, i) * A(k, j) + 1

******************************************************************************/

static int invert(double mat[4][4], double inverse[4][4])
{
  // Find all NECESSARY 2x2 subdeterminants
  //double Det2_12_01 = mat[1][0]*mat[2][1] - mat[1][1]*mat[2][0];
  //double Det2_12_02 = mat[1][0]*mat[2][2] - mat[1][2]*mat[2][0];
  //double Det2_12_03 = mat[1][0]*mat[2][3] - mat[1][3]*mat[2][0];
  //double Det2_12_12 = mat[1][1]*mat[2][2] - mat[1][2]*mat[2][1];
  //double Det2_12_13 = mat[1][1]*mat[2][3] - mat[1][3]*mat[2][1];
  //double Det2_12_23 = mat[1][2]*mat[2][3] - mat[1][3]*mat[2][2];
  double Det2_13_01 = mat[1][0]*mat[3][1] - mat[1][1]*mat[3][0];
  //double Det2_13_02 = mat[1][0]*mat[3][2] - mat[1][2]*mat[3][0];
  double Det2_13_03 = mat[1][0]*mat[3][3] - mat[1][3]*mat[3][0];
  //double Det2_13_12 = mat[1][1]*mat[3][2] - mat[1][2]*mat[3][1];  
  double Det2_13_13 = mat[1][1]*mat[3][3] - mat[1][3]*mat[3][1];
  //double Det2_13_23 = mat[1][2]*mat[3][3] - mat[1][3]*mat[3][2];  
  double Det2_23_01 = mat[2][0]*mat[3][1] - mat[2][1]*mat[3][0];
  double Det2_23_02 = mat[2][0]*mat[3][2] - mat[2][2]*mat[3][0];
  double Det2_23_03 = mat[2][0]*mat[3][3] - mat[2][3]*mat[3][0];
  double Det2_23_12 = mat[2][1]*mat[3][2] - mat[2][2]*mat[3][1];
  double Det2_23_13 = mat[2][1]*mat[3][3] - mat[2][3]*mat[3][1];
  double Det2_23_23 = mat[2][2]*mat[3][3] - mat[2][3]*mat[3][2];

  // Find all NECESSARY 3x3 subdeterminants
  //double Det3_012_012 = mat[0][0]*Det2_12_12 - mat[0][1]*Det2_12_02 
  //				+ mat[0][2]*Det2_12_01;
  //double Det3_012_013 = mat[0][0]*Det2_12_13 - mat[0][1]*Det2_12_03 
  //				+ mat[0][3]*Det2_12_01;
  //double Det3_012_023 = mat[0][0]*Det2_12_23 - mat[0][2]*Det2_12_03
  //				+ mat[0][3]*Det2_12_02;
  //double Det3_012_123 = mat[0][1]*Det2_12_23 - mat[0][2]*Det2_12_13 
  //				+ mat[0][3]*Det2_12_12;
  //double Det3_013_012 = mat[0][0]*Det2_13_12 - mat[0][1]*Det2_13_02 
  //				+ mat[0][2]*Det2_13_01;
  double Det3_013_013 = mat[0][0]*Det2_13_13 - mat[0][1]*Det2_13_03
				+ mat[0][3]*Det2_13_01;
  //double Det3_013_023 = mat[0][0]*Det2_13_23 - mat[0][2]*Det2_13_03
  //				+ mat[0][3]*Det2_13_02;
  //double Det3_013_123 = mat[0][1]*Det2_13_23 - mat[0][2]*Det2_13_13
  //				+ mat[0][3]*Det2_13_12;
  //double Det3_023_012 = mat[0][0]*Det2_23_12 - mat[0][1]*Det2_23_02 
  //				+ mat[0][2]*Det2_23_01;
  //double Det3_023_013 = mat[0][0]*Det2_23_13 - mat[0][1]*Det2_23_03
  //				+ mat[0][3]*Det2_23_01;
  double Det3_023_023 = mat[0][0]*Det2_23_23 - mat[0][2]*Det2_23_03
				+ mat[0][3]*Det2_23_02;
  //double Det3_023_123 = mat[0][1]*Det2_23_23 - mat[0][2]*Det2_23_13
  //				+ mat[0][3]*Det2_23_12;
  double Det3_123_012 = mat[1][0]*Det2_23_12 - mat[1][1]*Det2_23_02 
				+ mat[1][2]*Det2_23_01;
  double Det3_123_013 = mat[1][0]*Det2_23_13 - mat[1][1]*Det2_23_03 
				+ mat[1][3]*Det2_23_01;
  double Det3_123_023 = mat[1][0]*Det2_23_23 - mat[1][2]*Det2_23_03 
				+ mat[1][3]*Det2_23_02;
  double Det3_123_123 = mat[1][1]*Det2_23_23 - mat[1][2]*Det2_23_13 
				+ mat[1][3]*Det2_23_12;

  // Find the 4x4 determinant
  static double det;
          det =   mat[0][0]*Det3_123_123 
		- mat[0][1]*Det3_123_023 
		+ mat[0][2]*Det3_123_013 
		- mat[0][3]*Det3_123_012;

  // ??? Should the test be made: fabs(det) <= epsilon ???
  if (det == 0.0)
      return 0;

  inverse[0][0] =  Det3_123_123 / det;
  //inverse[0][1] = -Det3_023_123 / det;
  //inverse[0][2] =  Det3_013_123 / det;
  //inverse[0][3] = -Det3_012_123 / det;

  //inverse[1][0] = -Det3_123_023 / det;
  inverse[1][1] =  Det3_023_023 / det;
  //inverse[1][2] = -Det3_013_023 / det;
  //inverse[1][3] =  Det3_012_023 / det;

  //inverse[2][0] =  Det3_123_013 / det;
  //inverse[2][1] = -Det3_023_013 / det;
  inverse[2][2] =  Det3_013_013 / det;
  //inverse[2][3] = -Det3_012_013 / det;

  //inverse[3][0] = -Det3_123_012 / det;
  //inverse[3][1] =  Det3_023_012 / det;
  //inverse[3][2] = -Det3_013_012 / det;
  //inverse[3][3] =  Det3_012_012 / det;

  return 1;
}  

void dop(int channels, struct gps_data_t *gpsdata)
{
    double prod[4][4];
    double inv[4][4];
    double satpos[MAXCHANNELS][4];
    int i, j, k, n;

    gpsd_report(0, "Satellite picture:\n");
    for (k = 0; k < MAXCHANNELS; k++) {
	if (gpsdata->used[k])
	    gpsd_report(0, "az: %d el: %d  SV: %d\n",
			gpsdata->azimuth[k], gpsdata->elevation[k], gpsdata->used[k]);
    }

    for (n = k = 0; k < channels; k++) {
	if (!gpsdata->used[k])
	    continue;
	satpos[n][0] = sin(gpsdata->azimuth[k]*DEG_2_RAD)
	    * cos(gpsdata->elevation[k]*DEG_2_RAD);
	satpos[n][1] = cos(gpsdata->azimuth[k]*DEG_2_RAD)
	    * cos(gpsdata->elevation[k]*DEG_2_RAD);
	satpos[n][2] = sin(gpsdata->elevation[k]*DEG_2_RAD);
	satpos[n][3] = 1;
	n++;
    }

    gpsd_report(0, "Line-of-sight matrix:\n");
    for (k = 0; k < n; k++) {
	gpsd_report(0, "%f %f %f %f\n",
		    satpos[k][0], satpos[k][1], satpos[k][2], satpos[k][3]);
    }

    for (i = 0; i < 4; i++)
	for (j = 0; j < 4; j++) {
	    prod[i][j] = 1;
	    for (k = 0; k < n; k++)
		prod[i][j] += satpos[k][i] + satpos[k][j];
	}

    gpsd_report(0, "product:\n");
    for (k = 0; k < 4; k++) {
	gpsd_report(0, "%f %f %f %f\n",
		    prod[k][0], prod[k][1], prod[k][2], prod[k][3]);
    }

    if (invert(prod, inv)) {
	gpsd_report(0, "inverse:\n");
	for (k = 0; k < 4; k++) {
	    gpsd_report(0, "%f %f %f %f\n",
			inv[k][0], inv[k][1], inv[k][2], inv[k][3]);
	}
	gpsd_report(0, "HDOP: reported = %f, computed = %f\n",
		    gpsdata->hdop, sqrt(inv[0][0] + inv[1][1]));
    } else
	gpsd_report(0, "Matrix is singular.\n");

    //gpsdata->hdop = sqrt(diag[0] + diag[1]);
    //gpsdata->vdop = sqrt(diag[1]);
    //gpsdata->pdop = sqrt(diag[0] + diag[1] + diag[2]);
}
#ifdef __UNUSED__
#endif /* __UNUSED */
