/******************************************************************************
//             INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2000 Intel Corporation. All Rights Reserved.
//
//  VSS: 
//		$Workfile: sparseiir.c $
//		$Revision: 1 $
//		$Date: 05-02-28 16:03 $
//
//  Description:
//		Prototype sparse infinite impulse response (IIR) filter <Direct Form II> 
//
******************************************************************************/

#include <ippdefs.h>
#include <ippSP.h>

/* Ipp16s saturation */
#ifndef		SATURATETOIPP16S
#define		SATURATETOIPP16S(x) ((Ipp16s) ((x)>IPP_MAX_16S? IPP_MAX_16S: (x)<IPP_MIN_16S? IPP_MIN_16S: (x)))
#endif 

/******************************************************************************
// Name:            ippsIIR_DirectSparse_16s
//
// Description:     Sparse IIR <Direct Form II> block mode filter
//					Allows efficient implementation of high-order IIR filters
//                  that have relatively few (sparse) taps.
//
// Inputs:			pSrc			- Pointer to the input vector
//					len				- Number of samples, input and output vectors
//                  pTaps			- Pointer to tap vector
//                  pDelays			- Pointer to lag vector
//                  pDelayLineIndex	- Pointer to the current delay line sample (state variable)
//                  pDelayLine		- Pointer to the delay line (state variable)
//
// Outputs:			pDst			- Pointer to the output vector
//
// Returns:			ippStsNoErr     - No errors
//
// Notes:			1. Delay line must be of length max(pDelays[.])+1 
//					2. pDelay[0] contains M, the number numerator taps in H(z)
//					3. pDelay[M+1] contains L, the number of denominator taps in H(z)
//					4. pTaps[M+L] contains the coefficient scalefactor
//					5. Example:
//
//	pDelays = { 3, 1000, 2000, 3000, 2, 50, 70 };
//	pTaps =   { 10, 20, 30, -100, -200, 12 };
//					
//  Implements the following system transfer function, H(z)
//
//                      0.0024 z^-1000 +  0.0049 z^-2000 + 0.0073 z^-3000
//	        H(z) =    ------------------------------------------------------  
//							1 - 0.0244 z^-50 - 0.0488 z^-70
//
//
//  For this example, pDelayLine must be of size 3001 (or larger)
//  Note that the coefficient scalefactor is 12 
******************************************************************************/

IppStatus  ippsIIR_DirectSparse_16s( const Ipp16s	*pSrc, 
									 Ipp16s			*pDst,
									 int			len, 
									 const Ipp16s	*pTaps, 
									 int			*pDelays,
									 int			*pDelayLineIndex,
									 Ipp32s			*pDelayLine)
{
	int i,j,k,n,m,M,L,W;
	int ScaleFactor;
	Ipp32s *w;
	Ipp32s y;

	/* Initialize internal filter variables */
	w=pDelayLine;
	M=pDelays[0];
	L=pDelays[M+1];
	if (pDelays[M]>pDelays[M+L+1])
		W=pDelays[M];
	else
		W=pDelays[M+L+1];
	ScaleFactor=pTaps[M+L];
	n=*pDelayLineIndex;
	
	/* Compute filter output vector */
	for(i=0;i<len;i++)
	{
		/* Init accumulators */
		y=w[n]=0;
		
		/* Update direct form II delay line */
		for (j=M+2;j<=M+L+1;j++)
		{
			m=pDelays[j];
			if (n+m>W)
				k=n+m-W-1;		
			else
				k=n+m;
			w[n]-=pTaps[j-2]*w[k];  /* Match O&S, Matlab sign convention for a(i) */
		}

		/* Apply scalefactor to feedback path */
		w[n]>>=ScaleFactor;
		w[n]+=(*pSrc++);

		/* Compute i-th output sample */
    	for (j=0;j<M;j++)
		{
			m=pDelays[j+1];
			if (n+m>W)
				k=n+m-W-1;		
			else
				k=n+m;
			y+=pTaps[j]*w[k];
		}

		/* Apply scaling */
		y>>=ScaleFactor;

		/* Update delay line index */
		if (n==0)
			n=W;
		else
			n--;

		/* Saturate the output */
		*pDst++ = SATURATETOIPP16S(y);
	}

	/* Update delay line index */
	*pDelayLineIndex=n;

    return ippStsNoErr;

} /* end of ippsIIR_DirectSparse_16s() */


/* EOF */
