/******************************************************************************
//               INTEL CORPORATION PROPRIETARY INFORMATION
//  This software is supplied under the terms of a license agreement or
//  nondisclosure agreement with Intel Corporation and may not be copied
//  or disclosed except in accordance with the terms of that agreement.
//        Copyright (c) 2000 Intel Corporation. All Rights Reserved.
//
//  VSS:
//		$Workfile: SigGen.c $
//      $Revision: 2 $
//      $Date: 05-03-10 18:49 $
//      $Archive: /ipp/branch/5_0/beta/install/bulverde/primitive/arm_linux/sample/SigGen/SigGen.c $
//
//  Description:
//      IPP signal generation example
//
******************************************************************************/

/************
Header files
*************/

/* Linux audio output device interface */
#include	<fcntl.h>					
#include	<unistd.h>
#include	<sys/ioctl.h>
#include	<linux/soundcard.h>
#include	<sched.h>

/* Console, file, and string I/O */
#include	<stdio.h>
#include	<string.h>

/* Standard IPP definitions */
#include	"ippdefs.h"
#include	"ippSP.h"

/* Interface to IPP-based examples */
#include	"signal.h"

/* Local constants */
#define		FS				44100		/* Sample rate */
#define		N				1024		/* Buffer size */
#define		DEFAULT_FREQ	500			/* Default fundamental frequency */

/* Initial conditions */
#define		INIT_SIGMA		1000		/* Startup standard deviation for noise generator */
#define		INIT_FRQ		500			/* Startup frequency for tone and triangle generators */						

/* User interface constants */
#define		UI_STACK_SIZE	2500		/* User interface stack size, in 32-bit words */					
#define		MSGBUF_LEN      1000		/* User interface keyboard buffer size, in bytes */
#define		PBUF_LEN		100			/* User interface parsing buffer */

/*************
 Prototypes 
**************/
void			DisplayConsoleStartupBanner		(void);
void			ProcessCommandLineArguments		(int, char **, FILE **, Ipp16s *);
void			RenderSound						(int *, Ipp16s *, int);
int				OpenAudioDevice					(int *, int, int);
void			CloseOutputFile					(FILE *);
void			CloseAudioDevice				(int);
void			WriteOutputFile					(FILE *, Ipp16s *, int);
int				UserInterface					(void);

/*************
 Global data 
**************/
int g_Quit=0;					/* Quit flag, set asynchronously by the user interface */

/*************************************
 Console-mode application entry point 
**************************************/
/******************************************************************************************
// Name:			 main
//
// Description:		 Signal generation sample code
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//
*******************************************************************************************/
int main(int argc, char **argv)
{
	FILE *fpo;					/* Output file */							
	Ipp16s sig[N];				/* Signal buffer */
	Ipp16s freq;				/* Fundamental frequency */
	int AudioDevice;			/* Pointer to the (Linux) audio output device */
	int StackUI[UI_STACK_SIZE];	/* User interface stack (child thread) */
	int frameNum;
	frameNum =0;
	
	/****************************************************
	Startup banner, parse command line, open I/O devices  
	*****************************************************/
	DisplayConsoleStartupBanner();	
	ProcessCommandLineArguments(argc,argv,&fpo,&freq);

	/* Spawn user interface thread */
	__clone(UserInterface,&(StackUI[UI_STACK_SIZE]),CLONE_VM|CLONE_FS);

	/* Open audio device if no output file was specified */
	if (fpo==NULL)
		OpenAudioDevice(&AudioDevice,2,FS);

	/* Establish initial frequency */
	SetFrequency(freq,FS);
	SetStdDev(INIT_SIGMA);
	SetSignalType(TONE);

	/* Generate signal */
//	while(!g_Quit)
	while(frameNum <100)
	{
		GenerateSignal(sig,N);
		frameNum ++;
		/* Copy to audio device or output file */
		if (fpo==NULL)
			RenderSound(&AudioDevice,sig,N);
		else
			WriteOutputFile(fpo,sig,N);
	}

	/* Close I/O device */	
	if (fpo!=NULL)
		CloseOutputFile(fpo);
	else
		CloseAudioDevice(AudioDevice);

	/* Shtdown message */
	printf("Done.\n\n");

} /* End of main() */

/********************************
 Console management and file I/O 
*********************************/

/*********************************************************************************************
// Name:			 DisplayConsoleStartupBanner
//
// Description:		 Display startup message on the text console
//
// Input Arguments:  None.
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void DisplayConsoleStartupBanner()
{
	int i;
	for(i=0;i<50;i++) printf("\n");
	printf("\n\nIntel(R) Integrated Performance Primitives (IPP)\nfor Intel(R) XScale(TM) Microarchitecture.\n\n");
	printf("Signal Generation Example, Version 1.00\n\n");
	printf("Copyright (C) 2000-01, Intel Corporation.  All Rights Reserved.\n\n");
}

/*********************************************************************************************
// Name:			 ProcessCommandLineArguments
//
// Description:		 Extracts run-time parameters from the command line
//					 Opens I/O files
//
// Input Arguments:  argc				- Standard C argument count
//					 argv				- Standard C argument list
//					 fpo				- Pointer to the output file stream
//					 freq				- Pointer to desired sine frequency
//
// Output Arguments: None				
//
// Returns:			 None
**********************************************************************************************/
void ProcessCommandLineArguments(int argc, char **argv, FILE **fpo, Ipp16s *freq)
{
	/* Usage */
	if ((argc<2)||(argc>3)||(strcmp(argv[1],"--help")==0))
	{
		printf("Usage: SigGen <frequency> <output file> [--help], \n\nwhere\n\n");
		printf("<frequency> is the desired sinusoidal frequency\n");
		printf("<output file> is the desired binary pcm output file\n\n");
		printf("No output file forces use of the audio device for output.\n\n");
		printf("--help displays usage\n\n");
		printf("\n");
		exit(0);
	}

	/* Get initial frequency */
	if (argc>=1)
		*freq=atoi(argv[1]);
	else
		*freq=DEFAULT_FREQ;

	/* Display parameters */
	printf("Frequency:\t%d\n",*freq);

	/* Get desired output device 
	   No argument = use audio device
	   Argument present = output file name
	 */  
    /* Open output file */
	if (argc>=3)
	{
		if ((*fpo=fopen(argv[2],"wb"))==NULL)
		{
			printf("Unable to open %s\n",argv[2]);
			exit(1);
		}
		else
			printf("Output file:\t%s\n",argv[2]);	
	}
	else
	{
		*fpo=NULL;
		printf("Output:\tAudio device\n");
	}
}

/*********************************************************************************************
// Name:			 CloseOutputFile
//
// Description:		 Close PCM binary output file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseOutputFile(FILE *fpo)
{
	if (fpo!=NULL) fclose(fpo);
}

/*****************************
 Linux Audio Device Interface
******************************/

/***************************************************************************************
// Name:			 RenderSound
//
// Description:		 Write one granule of pcm audio samples to the audio output device. 
//
// Input Arguments:  fp					- Pointer to the audio output device. 
//					 pcm				- Pointer to a buffer of interleaved pcm audio.
//					 channels			- Number of channels contained in the buffer.
//					 len				- PCM buffer length, in samples (not bytes)
//
// Returns:			 None.
//
// Notes:			 Stereophonic data is interleaved as follows:  R0L0R1L1 ... R575L575; 
//					 For channels=1, right channel data is replicated on to the left.
****************************************************************************************/
void RenderSound(int *fp, Ipp16s *pcm, int len)
{
	static Ipp16s obuf[2*N];
	int i,j;

	/* Replicate signal on both left and right channels */
	for(i=j=0;i<2*N;i+=2,j++)
		obuf[i]=obuf[i+1]=pcm[j];

	/* Pass audio output buffer to the audio output device */
	write(*fp,obuf,sizeof(Ipp16s)*len*2);
}

/***************************************************************************************
// Name:			 OpenAudioDevice
//
// Description:		 Initialize Linux audio output device. 
//
// Input Arguments:  dev				- Pointer to the audio output device. 
//					 channels			- Number of audio channels.
//
// Returns:			 Status				 0	 = success, device ready
//										-100 = unable to open device
//										-101 = unable to select 16-bit stereo PCM
//										-102 = unable to select 44100 Hz sample rate
//										-103 = unable to select specified # of channels 
//
// Notes:			 01/03/01 - Current blob Linux SA-1110 Microprocessor Evaluation Module
//					 UDA1341 audio driver has bugs; 44.1 kHz sample rate is currently not 
//					 working properly; device is actually running at 48 kHz
//					 The MP3 sample application should be modified once the
//					 driver has been fixed to support other sample rates such that 
//                   32, 44.1, and 48 kHz files are rendered with correct sample rate.
****************************************************************************************/
int OpenAudioDevice(int *dev, int channels, int SampleRate)
{
	long param;
	int status=0;

	/* Open audio device (UDA1341 on the Linux SA-1110 Microprocessor Evaluation Module) */
	if ((*dev=open("/dev/dsp",O_RDWR))==-1) status=-100;

	/* Request 16-bit, little-endian, PCM data format */
	param = AFMT_S16_LE;
	if ((ioctl(*dev,SNDCTL_DSP_SETFMT,&param))!=0) { close (*dev); status=-101; }
  
	/* Request sample rate */
	param=SampleRate;
	if ((ioctl (*dev,SNDCTL_DSP_SPEED,&param))!=0) { close (*dev); status=-102; }

	/* Request number of channels */
	param = channels;
	if ((ioctl(*dev,SNDCTL_DSP_CHANNELS,&param))!=0) { close (*dev); status=-103; }

	/* Trap unsuccessful attempts to open audio device */
	if (status<0) { fprintf(stderr,"\nError opening audio device.\n"); fflush(stderr); exit(1); }
	else return 0;
}

/*********************************************************************************************
// Name:			 CloseAudioDevice
//
// Description:		 Close audio output device
//
// Input Arguments:  AudioDevice		- Pointer to the audio output device
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void CloseAudioDevice(int AudioDevice)
{
	close(AudioDevice);
}

/*********************************************************************************************
// Name:			 WriteOutputFile
//
// Description:		 Write PCM buffer contents to a binary file
//
// Input Arguments:  fpo				- Pointer to the output file stream
//					 pcm				- PCM output buffer pointer
//					 N					- Number of samples to write
//
// Output Arguments: None				
//
// Returns:			 None				
**********************************************************************************************/
void WriteOutputFile(FILE *fpo, Ipp16s *pcm, int len)
{
	if (fpo!=NULL) 
		fwrite(pcm,sizeof(Ipp16s),len,fpo);
}

/*********************************************************************************************
// Name:			 User interface thread
//
// Description:		 Keyboard character-based user interface
//					 Thread terminates with the parent
//
// Returns:			 None				
**********************************************************************************************/
int UserInterface()
{
	char msg[MSGBUF_LEN], tstr[PBUF_LEN];	/* Keyboard message and parsing buffer */
	int i,freq;								/* Buffer indices and control params */
	Ipp16s stddev;
	while(1)
	{
		/* Get user input from the keyboard */
		gets(msg);
		
		/* Parse and act on user input */
		switch(msg[0])
		{
			/* q = Quit */
			case 'q':
			case 'Q':
				g_Quit=1;
				break;

			/* f<freq> = new frequency, up to 5 significant digits */
			case 'f':
			case 'F':
				strncpy(tstr,&(msg[1]),5);
				freq=atoi(tstr);
				SetFrequency(freq,FS);
				printf("Frequency: %d Hz\n",freq);
				break;

			/* h = help */
			case 'h':
			case 'H':
				printf("\n\nCommands:\n");
				printf("f<frequency>		- change fundamental frequency\n");
				printf("h                   - display command list\n");
				printf("n					- select Gaussian noise generator\n");
				printf("t					- select tone generator\n");
				printf("v<stddev>			- change Gaussian standard deviation\n\n");
				printf("q                   - QUIT\n");
				printf("\n\n");
				break;

			case 'n':
			case 'N':
				SetSignalType(NOISE);
				printf("Gaussian noise output\n");
				break;
			case 's':
			case 'S':
				SetSignalType(TONE);
				printf("Sinusoidal output\n");
				break;
			case 't':
			case 'T':
				SetSignalType(TRI);
				printf("Triangular output\n");
				break;
			case 'v':
			case 'V':
				strncpy(tstr,&(msg[1]),5);
				stddev=atoi(tstr);
				SetStdDev(stddev);
				printf("StdDev: %d\n",stddev);
				break;
			default:
				break;
		}

		/* Clear keyboard message buffer */
		for(i=0;i<MSGBUF_LEN;i++)
			msg[i]='\0';
	}
	return 0;
}
