#!/bin/sh
# Copyright 2004, 2005, Intel Corporation, All Rights Reserved.
# This script uninstalls Intel(R) Performance Libraries and related products.






DISPLAY_QUESTION(){
	
	local q=$1
	local da=$2
	local rda=$3
	
	IS_COMMAND_LINE_OPTION_EXISTS silent
	local silent=$?

	NON_SILENT_DEFAULT="$rda"
	PRINT_TO_LOG "${0##*/}:$LINENO user-input:DISPLAY_QUESTION(): question is: '$q'"

	[ $silent -eq ${ERR_OK} ] && return ${ERR_OK}

	echo -n "$q"
	
	if [ ! -z "$da" ] ; then
		echo -n " ( $da )"
	fi

	if [ ! -z "$rda" ] ; then
		echo -n " [ $rda ]"
	fi

	echo -n ": "
	[ $silent -eq ${ERR_OK} ] && echo
}

IS_ANSWER_MATCHES(){
	
	local qa_list=`echo $1 | tr [A-Z] [a-z]`
	local qa_given=`echo $2 | tr [A-Z] [a-z]`

	PRINT_TO_LOG "${0##*/}:$LINENO Given answers - '$qa_given'"
	PRINT_TO_LOG "${0##*/}:$LINENO Check answer - '$qa_list'"
	
	if [ -z "$qa_given" ] ; then
		PRINT_TO_LOG "${0##*/}:$LINENO Empty answer given!" 1
		return 1
	fi
	
	local iter=

	for iter in $qa_given ; do
		if [ "x$iter" = "x$qa_list" ] ; then 
			return ${ERR_OK}
		fi
	done

	return 1  # not matched!
}

FIND_STRING()
{
    local s=$1
    local a=
    shift
    PRINT_TO_LOG "${0##*/}:$LINENO checking '$s' against this list: "
    for a in $@
    do
        if [ "x$s" == "x$a" ];then
            PRINT_TO_LOG "${0##*/}:$LINENO found"
            return ${ERR_OK}
        fi
    done
    PRINT_TO_LOG "${0##*/}:$LINENO haven't found '$s'"
    return 1
}

RPM_INIT(){
	[ ${RPM_CHECK} -eq 0 ] || return ${ERR_OK}
	rpm -q rpm &> /dev/null

	
	if [ $? -ne 0 ] ; then
		PRINT_TO_LOG "Cannot get shared lock on RPM Database" 1
		return ${ERR_RPM_LOCK}
	fi

	RPM_CHECK=1
	
	return ${ERR_OK}
} # RPM_INIT(){   

GET_USER_INTERACTIVE_ANSWER() {
	local question_id=$1
	local l_answer=

	PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): getting answer on the question with id '$question_id'"
	if IS_COMMAND_LINE_OPTION_EXISTS silent ; then
		l_answer=`READ_DEFAULT_ANSWER "$question_id"`
		if [ $? -ne ${ERR_OK} ] ; then 	# haven't found the parameter in config file
		    PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): unable to get default answer for '$question_id' from config file" 2
		    PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): using default answer" 2
		    l_answer=$NON_SILENT_DEFAULT
		fi
	else
    		read l_answer
		if [ "${l_answer}x" == "x" ] ; then 	# user has pressed <Enter>
		    PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): user entered empty answer" 2
		    PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): using default answer" 2
		    l_answer=$NON_SILENT_DEFAULT
		fi
		IS_COMMAND_LINE_OPTION_EXISTS duplicate && ADD_ENTRY_TO_CONFIG_FILE $question_id "$l_answer"
	fi
	echo -en "$l_answer\n"
	PRINT_TO_LOG "${0##*/}:$LINENO: interact:GET_USER_INTERACTIVE_ANSWER(): the answer is '$l_answer'"
	return ${ERR_OK}
} # GET_USER_INTERACTIVE_ANSWER


READ_YES_NO_ANSWER ()
{
    local QUESTION=$1
    local DEFAULT_ANSWER=$2
    local question_id=$3
    local YES_ANSWER="y yes yea yeah yep ok"	# these answers are recognized as 'Yes'
    local NO_ANSWER="n no not nop nope"		# these answers are recognized as 'No'
    local answer=

    while [ 1 -eq 1 ] ; do
        IS_COMMAND_LINE_OPTION_EXISTS silent || DISPLAY_QUESTION "$QUESTION" "Yes/No" "$DEFAULT_ANSWER"
        answer=`GET_USER_INTERACTIVE_ANSWER $question_id`
        YESNO="$(echo $answer | tr -s A-Z a-z)"
        if FIND_STRING $YESNO "$YES_ANSWER":;then
	    PRINT_TO_LOG "${0##*/}:$LINENO: interact: READ_YES_NO_ANSWER (): the answer is Yes"
            YESNO="yes"
            return 0
        else
            if FIND_STRING $YESNO "$NO_ANSWER";then
		PRINT_TO_LOG "${0##*/}:$LINENO: interact: READ_YES_NO_ANSWER (): the answer is No"
                YESNO="no"
                return 1
            fi
	    if IS_COMMAND_LINE_OPTION_EXISTS silent;then
		PRINT_TO_LOG "${0##*/}:$LINENO: interact: READ_YES_NO_ANSWER (): couldn't recognize the answer: config file parameter '$question_id' has wrong value '$answer'" 1
		return 2
	    fi
        fi
    done
}


WAIT_ENTER_KEY ()
{
    IS_COMMAND_LINE_OPTION_EXISTS silent || read
}

SAVE_COMMAND_LINE(){
	CMD_STR=$@   
} #
			                                                                                
IS_COMMAND_LINE_OPTION_EXISTS(){
                                            
	local cmd=$1                                    
	if echo $CMD_STR | egrep -i "[[:space:]]*\-\-$cmd" &> /dev/null ; then
		return ${ERR_OK}
	fi
	
	return 1
} # IS_COMMAND_LINE_OPTION_EXISTS()


GET_COMMAND_LINE_OPTION_VALUE(){
	
	local cmd=$1
	local err=0;
	
	IS_COMMAND_LINE_OPTION_EXISTS $cmd
	err=$?
	
	[ $err -eq ${ERR_OK} ] || return 1
	
	echo $CMD_STR | sed s/.*--$cmd[[:blank:]]*//g | sed 's/[[:blank:]]*--.*$//g'

} # GET_COMMAND_LINE_OPTION_VALUE() {

INIT_LOGGING(){
	LOG_FILE_NAME=$1
	if ! touch "$LOG_FILE_NAME" 2> /dev/null; then
            echo "Warning: Unable to initialize \"$LOG_FILE_NAME\" log file"
            LOG_FILE_NAME=
	    return 1
	fi
	PRINT_ERROR=0 # No errors
	return ${ERR_OK}
} # INIT_LOGGING(){

PRINT_TO_LOG(){
	local message=$1
	local success_code=$2
	local time_stamp=$(date +'%a %m %T')

        if [ -z "$LOG_FILE_NAME" ]; then
            return 0
        fi
        
	[ -z $success_code ] && success_code=0

	if [ ${PRINT_ERROR} -ne 0 ] ; then
        	return ${ERR_LOGGING}
	fi # if

	echo -n "$time_stamp: " >> "$LOG_FILE_NAME" 2>/dev/null
	case $success_code in
	0)
	    echo -n " Success - " >> "$LOG_FILE_NAME" 2>/dev/null;
	    ;;
	1)
	    echo -n " Error - " >> "$LOG_FILE_NAME" 2>/dev/null;
	    ;;
	2)
	    echo -n " Warning - " >> "$LOG_FILE_NAME" 2>/dev/null;
	    ;;
	esac
	echo "$message" >> "$LOG_FILE_NAME" 2>/dev/null
	
	PRINT_ERROR=$?
	if [ ${PRINT_ERROR} -ne 0 ] ; then
		local CUR_DIR=`pwd`
		echo "Warning: Can not print log message into '$LOG_FILE_NAME' file."
		return ${ERR_LOGGING}
	fi #if [ ${PRINT_ERROR} -ne 0 ] ; then

	return ${ERR_OK}
} #PRINT_TO_LOG ()

RPM_INFO(){
	local rpm_name=$1
	local rpm_tag=$2
	
	[ $RPM_CHECK -eq 1 ] || RPM_INIT ; local err=$?
	[ $err -eq ${ERR_OK} ] || return $err
	
	local opt="-q"
	if [ -f $rpm_name ]; then
		PRINT_TO_LOG "Using file query for package - '$rpm_name'"
		opt="-qp"
	fi #if[ -x $rpm_name ]; then
	
	local rpm_out_count=`rpm $opt $rpm_name 2>&1 | wc -l`
	if [ $rpm_out_count -eq 1 ] ; then 
		local info=`rpm $opt $rpm_name --qf %{$rpm_tag} 2>&1`
		if ! echo $info | grep installed &>/dev/null ; then
			echo $info
			PRINT_TO_LOG "Package search: '$rpm_name' => '$rpm_tag' => '$info' "
			return $ERR_OK;
		fi #if ! echo $info | grep installed &>/dv/null ; then
	else
		local out=
		local i=0
		while [ $i -lt $rpm_out_count ] ; do
			local info=`rpm $opt $rpm_name --qf %{$rpm_tag}"\n" 2>&1 | sed $i'!d'`
			[ x$out != x ] || out=$out"@"
			out="$out$info"
			i=$(($i+1))
		done
		echo $out
		PRINT_TO_LOG "Multiple package search: '$rpm_name' => '$rpm_tag' => '$out' "
		return ${ERR_OK}
	fi

	PRINT_TO_LOG "Package '$rpm_name' not found" 1
	
	return ${ERR_RPM_NO_PACKAGE}
} # RPM_INFO(){






NONRPM_DB_ENTRY_CHECK_SYNTAX() {
    local entry="$1"
    local regexp='<:[^:]*:intel-[a-z_0-9]+-[0-9]+\.[0-9]+([abp]u?)?-[0-9]+\.[a-z_0-9]+\.rpm:[^:]*:[^:]*:>'
    return $(echo "$entry" | grep -E -x "$regexp" > /dev/null) \
        || FATAL_EXIT "Unexpected error."
} # NONRPM_DB_ENTRY_CHECK_SYNTAX




NONRPM_DB_ENTRY_GET_FIELD() {
    local entry="$1"
    local field="$2"
    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi
    if [ "$field" -lt 2 ] || [ "$field" -gt 5 ]; then
        return 2
    fi    
    local result=$(echo "$entry" | cut -d':' -f"$field") \
        || FATAL_EXIT "Unexpected error"
    echo "$result"    
    return 0
} # NONRPM_DB_ENTRY_GET_FIELD




NONRPM_DB_ENTRY_GET_RPMFILE() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 3     
    return $?
} # NONRPM_DB_ENTRY_GET_RPMFILE

NONRPM_DB_ENTRY_GET_INSTALLDIR() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 4     
    return $?
} # NONRPM_DB_ENTRY_GET_INSTALLDIR

NONRPM_DB_ENTRY_GET_LOGFILE() {
    NONRPM_DB_ENTRY_GET_FIELD "$1" 5     
    return $?
} # NONRPM_DB_ENTRY_GET_LOGFILE




NONRPM_DB_ENTRY_ADD() {
    local entry="$1"
    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi
    echo "$entry" >> "$INTEL_SDP_PRODUCTS_DB"
    if [ $? -ne 0 ]; then
        return 2
    fi
} # NONRPM_DB_ENTRY_ADD




NONRPM_DB_ENTRY_REMOVE() {

    local entry="$1"

    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        return 1
    fi

    cp -p "$INTEL_SDP_PRODUCTS_DB" "$INTEL_SDP_PRODUCTS_DB~" \
        || FATAL_EXIT "Unable to create backup copy of \"$INTEL_SDP_PRODUCTS_DB\" file."
    grep -F -v -x "$entry" "$INTEL_SDP_PRODUCTS_DB~" > "$INTEL_SDP_PRODUCTS_DB"
    local rc=$?
    [ $rc -le 1 ] || FATAL_EXIT "Unable to overwrite \"$INTEL_SDP_PRODUCTS_DB\" file."
    chmod --reference="$INTEL_SDP_PRODUCTS_DB~" "$INTEL_SDP_PRODUCTS_DB" \
        || FATAL_EXIT "Unable to change permissions on \"$INTEL_SDP_PRODUCTS_DB\" file."

    return 0    
    
} # NONRPM_DB_ENTRY_REMOVE




NONRPM_DB_IS_PACKAGE_INSTALLED() {
    local package="$1"
    if [[ -f "$INTEL_SDP_PRODUCTS_DB" ]]; then
        grep -F ":$package" "$INTEL_SDP_PRODUCTS_DB" &> /dev/null 
        local rc=$?    
        if [ $rc -ge 2 ]; then
            FATAL_EXIT "Unexpected error."
        fi    
    else
        rc=1
    fi    
    return $rc
} # NONRPM_DB_IS_PACKAGE_INSTALLED

                            


NONRPM_DB_FIND_FILE_OWNER() {

    local file="$1"

    local entry                        
    local log_file
    local owner
             
    if [[ -f "$INTEL_SDP_PRODUCTS_DB" ]]; then
        while read entry; do
            log_file=$(NONRPM_DB_ENTRY_GET_LOGFILE "$entry")
            if [ $? -eq 0 ] && [ -f "$log_file" ]; then
                owner=$(grep -F -x -l "$file" "$log_file")
                if [ $? -ge 2 ]; then
                    FATAL_EXIT "Unexpected error."
                fi    
                if [ -n "$owner" ]; then
                    echo "$entry"
                    return 0
                fi    
            fi
        done < "$INTEL_SDP_PRODUCTS_DB"
    fi    

    return 0

} # NONRPM_DB_FIND_FILE_OWNER





NONRPM_DB_ENTRY_FIND_BY_RPMFILE() {

    local rpmfile=$(basename "$1")

    local entry                        
    
    if [[ -f "$INTEL_SDP_PRODUCTS_DB" ]]; then
        while read entry; do
            if [ "$rpmfile" == $(NONRPM_DB_ENTRY_GET_RPMFILE "$entry") ]; then
                echo "$entry"
                return 0
            fi
        done < "$INTEL_SDP_PRODUCTS_DB"
    fi
    
    return 0

} # NONRPM_DB_ENTRY_FIND_BY_RPMFILE


COMPARE_VERSION(){
	local A=$1
	local B=$2
        local COMPARE_RESULT=0
        local INDEX=1
        local CA="1"
        local CB="2"

        if [ $(echo $A | grep -v "\.") ] && [ $(echo $B | grep -v "\.") ]; then
		if [ "$A" -gt "$B" ] ; then
		        COMPARE_RESULT=1
		elif [ "$B" -gt "$A" ] ; then
		        COMPARE_RESULT=-1
	        fi # if [ "$A" -gt "$B" ] ; then
		return $COMPARE_RESULT
        fi
       


        while [ "$CA" != "" ] && [ "$CB" != "" ] ; do
		CA=$(echo $A | cut -d'.' -f${INDEX})
		CB=$(echo $B | cut -d'.' -f${INDEX})
		if [ "$CA" != "" ] && [ "$CB" = "" ] ; then
	    		COMPARE_RESULT=1
		elif [ "$CA" = "" ] && [ "$CB" != "" ] ; then
			COMPARE_RESULT=-1
		elif [ "$CA" != "" ] && [ "$CB" != "" ] ; then
	    		if [ "$CA" -gt "$CB" ] ; then
				COMPARE_RESULT=1
			elif [ "$CB" -gt "$CA" ] ; then
				COMPARE_RESULT=-1
	    		fi # if [ "$CA" -gt "$CB" ] ; then
	    		if [ $COMPARE_RESULT -ne 0 ] ; then
				break
	        	fi # if [ "$COMPARE_RESULT" -ne "0" ] ; then
		fi # if [ "$CA" != "" ] && [ "$CB" = "" ] ; then
		INDEX=$(($INDEX+1))
	done #while [ "$CA" != "" ] && [ "$CB" != "" ] ; do
        return $COMPARE_RESULT
	
} # COMPARE_VERSION(){


	
CHECK_COMMANDS(){
        local CHECK_CMD="rpm rm wc grep cp chmod dirname sed egrep mktemp basename date rpm2cpio mkdir find cat tac rmdir cut id arch awk"
        local c=''
        for c in ${CHECK_CMD} ; do
        local cmd=`type -p $c 2>&1` ;
        if [ ${cmd}"x" = "x" ] ; then
                echo "ERROR: unable to find command '$c'."
                echo "Please add the location to the above commands to your PATH and re-run the script"
                echo -n "Press Enter to continue."
		WAIT_ENTER_KEY
                exit 1
        fi
        done
}

CONFIG_ERRORS(){
        
        ERR_OK=0

        ERR_RPM_UNINSTALL=10
	ERR_RPM_NO_PACKAGE=11
	ERR_RPM_NOT_RELOCATABLE=12
	ERR_UNKNOWN_ARC=13
	ERR_LOGGING=14
	ERR_RPM_LOCK=15
}

CONFIG_STD_ENV(){

	CMD_STR=''
	YESNO=''
	RPM_CONFIGURED=-1
	NON_SILENT_DEFAULT=''
	PLATFORM=''
	IA=''
	declare CONFIG=''
	declare DUPLICATE_CONFIG=''
	declare DUPLICATE_CONFIG_TMP=''
	PRINT_ERROR=0
	LOG_FILE_NAME=''
	RPM_CHECK=0
}

CHECK_ROOT_PRIVILEDGES() {
        local USERID=`id -u`
        if [ $USERID -ne 0 ] ; then
            if [ ! -w /dev ] ; then
                local current_dir=`pwd`
                echo "WARNING:  Super-user or \"root\" priviledges are required in order to continue."
		IS_COMMAND_LINE_OPTION_EXISTS silent && return 1
                echo -n "Please enter \"root\" "
                exec su -c "cd '$current_dir' && /bin/sh '${SCRIPT}' $@" # always do the fork.
                echo "" # should never get here
                return 1
            fi
        fi
}

READ_DEFAULT_ANSWER(){	
	local param=$1
	local file=$CONFIG
	local DG=	# config file entry
	local DGV=	# value for the parameter specified within 'DG' entry
	PRINT_TO_LOG "${0##*/}:$LINENO silent:READ_DEFAULT_ANSWER(): Trying to find the parameter '$param' in config file '$file'"
	[ -f $file ] || return 1
	DG=`sed -n "/^$param=/p" $file | sed -n '1p' | grep ''`
	if [ $? -ne 0 ];then
	    PRINT_TO_LOG "${0##*/}:$LINENO silent:READ_DEFAULT_ANSWER(): Didn't found the parameter '$param' in config file '$file'" 2
	    return 1
	fi
    	DGV=${DG#*=} # get the parameter value (starting at first '=')
	PRINT_TO_LOG "${0##*/}:$LINENO silent:READ_DEFAULT_ANSWER(): The parameter '$param' is found, its value is '$DGV'"
	echo -ne "$DGV\n"
	return ${ERR_OK}
} #READ_DEFAULT_ANSWER(){
			                                                                                


SET_CONFIG_FILE(){
        CONFIG=`GET_COMMAND_LINE_OPTION_VALUE silent` || return 1
	[ -f "$CONFIG" ] && [ -r "$CONFIG" ]|| return 1
	PRINT_TO_LOG "CONFIG is set to '$CONFIG'"
        return ${ERR_OK}
} # SET_CONFIG_FILE()

WRITE_ANSWERS_TO_CONFIG_FILE()
{
    PRINT_TO_LOG "${0##*/}:$LINENO: Writing gathered answers into config file '$DUPLICATE_CONFIG'"
    awk 'BEGIN { FS = "=" }
        !/^#/ { if ($0 ~ /.*=/) { entries[$1]=$2 }  }
        END {
        for (entry_name in entries)
        {
            printf "%s=%s\n",entry_name,entries[entry_name]
        }
    }' $DUPLICATE_CONFIG_TMP >> $DUPLICATE_CONFIG 2>/dev/null || return 1
    PRINT_TO_LOG "${0##*/}:$LINENO: Answers have been written into config file '$DUPLICATE_CONFIG'"
    rm -f $DUPLICATE_CONFIG_TMP
    return ${ERR_OK}
}

SET_CONFIG_FILE_TO_DUPLICATE(){
	local product_abbr=$1
	local path_to_existing_duplicate_config=$2
	local duplicate_name="${product_abbr}SilentInstall.ini"
	if [ "x$path_to_existing_duplicate_config" != "x" ];then
	    DUPLICATE_CONFIG="$path_to_existing_duplicate_config"
	else
    	    DUPLICATE_CONFIG="$(pwd)/${duplicate_name}"
    	    if ! touch "$DUPLICATE_CONFIG" &>/dev/null;then
        	PRINT_TO_LOG "${0##*/}:$LINENO: can't open config file $DUPLICATE_CONFIG for writing" 2
		DUPLICATE_CONFIG="/tmp/${product_abbr}/${duplicate_name}"
		if ! touch "$DUPLICATE_CONFIG" &>/dev/null;then
		    PRINT_TO_LOG "${0##*/}:$LINENO: can't open config file $DUPLICATE_CONFIG for writing" 2
        	    return 1
		fi
    	    fi
	fi
    	if [ ! -w "$DUPLICATE_CONFIG" ];then
        	PRINT_TO_LOG "${0##*/}:$LINENO: Unable to write to config file $DUPLICATE_CONFIG" 1
    	    return 1
        fi
        DUPLICATE_CONFIG_TMP=`mktemp -q /tmp/${0##*/}.XXXXXXXXXXX` || return 1
        PRINT_TO_LOG "${0##*/}:$LINENO: DUPLICATE_CONFIG is set to '$DUPLICATE_CONFIG'"
        return ${ERR_OK}
}

ADD_ENTRY_TO_CONFIG_FILE()
{
    echo "$1=$2" >> $DUPLICATE_CONFIG_TMP
    PRINT_TO_LOG "${0##*/}:$LINENO: added entry $1=$2 to config file"
}

EXPAND_TREE(){
	local locale=$1
	
	while [ ! -d $locale ] &> /dev/null ; do
		locale=$(dirname $locale)
	done # while [ ! -d $locale ] &> /dev/null ; do

	echo $locale
}
	
PKG_CHECK(){
	local package=$1
	local pack_ver=$2
	local pack_ver_rel=$3
	local pack='';
	local err=$ERR_OK;
	
	local equal=`echo $pack_ver_rel | sed s/[^e]*e[^e]*//g`
	local greater=`echo $pack_ver_rel | sed s/[^g]*g[^g]*//g`
	local less=`echo $pack_ver_rel | sed s/[^l]*l[^l]*//g`

	local seq=1
	local sgt=1
	local slt=1

	if [ "$pack_ver_rel" = ne ] ; then
		pack=`RPM_INFO $package name`
		err=$?
			
		if [ $err -eq ${ERR_RPM_NO_PACKAGE} ]  ; then 
			return ${ERR_OK}; 
		elif [ $err -ne ${ERR_OK} ] ; then
			return $err;
		fi # if [ $err -eq ${ERR_RPM_NO_PACKAGE} ]  ; then
			
		COMPARE_VERSION `RPM_INFO $package version` $pack_ver
			
		if [ $? -ne 0 ] ; then 
			return ${ERR_OK}; 
		fi # if [ $? -ne 0 ] ; then
		
	fi # if [ "$pack_ver_rel" = ne ] ; then

	if [ "$pack_ver_rel" = eq ] || [ "$pack_ver_rel" = ge ] ||[ "$pack_ver_rel" = lt ] ||[ "$pack_ver_rel" = le ] ||[ "$pack_ver_rel" = gt ] ; then 
		pack=`RPM_INFO $package name`; 
		[ $? -eq ${ERR_OK} ] || err=1; 
		if [ $err -eq ${ERR_OK} ] ; then 
			COMPARE_VERSION `RPM_INFO $package version` $pack_ver
			local error=$?

			if test $equal"x" = "x" ; then
				[ $error -eq 0 ] && seq=0 
			fi # if test $equal"x" = "x" ; then
			if test $greater"x" = "x" ; then
				[ $error -eq 1 ] && sgt=0
			fi # if test $greater"x" = "x" ; then
			if [ $less"x" = "x" ] ; then
				[ $error -eq 255 ] && slt=0
			fi # if [ $less"x" = "x" ] ; then
		fi # if [ $err -eq ${ERR_OK} ] ; then
	fi # if [ "$pack_ver_rel" = eq ] || [ "$pack_ver_rel" = ge...
	

	if [ $err -eq ${ERR_OK} ]; then 
		err=1
		case $pack_ver_rel in
			eq ) [ $seq -ne 0 ] || err=0 ;;
			ge ) [ $seq -ne 0 ] || err=0 ; [ $sgt -eq 1 ] || err=0 ;;
			le ) [ $seq -ne 0 ] || err=0 ; [ $slt -eq 1 ] || err=0 ;;
			gt ) [ $seq -eq 1 ] && [ $sgt -eq 0 ] && err=0 ;;
			lt ) [ $seq -eq 1 ] && [ $slt -eq 0 ] && err=0 ;;
		esac # if [ $err -eq ${ERR_OK} ]; then
	fi # if [ $err -eq ${ERR_OK} ]; then

	return $err

} # PKG_CHECK(){

ARC_GET(){
	[ "${PLATFORM}x" = "x" ] || return ${ERR_OK}
	
	if arch | egrep 'i.86' > /dev/null ; then
		IA='IA32';
		PLATFORM=x32;
	elif arch | grep ia64 > /dev/null ; then 
		IA='IPF'
		PLATFORM=x64
	elif arch | grep x86_64 > /dev/null ; then
		IA='EM64T'
		PLATFORM=x32_64
	else
		PRINT_TO_LOG "Unknown arch found: `arch`" 1
		return ${ERR_UNKNOWN_ARC}
	fi # if [ arch | egrep 'i?86' > /dev/null ] ; then
	return ${ERR_OK}
} # ARC_GET(){

RPM_CONFIG(){
	
	[ $RPM_CHECK -eq 1 ] || RPM_INIT ; local err=$?	# RPM tool hasn't been checked yet, perform the check
	[ $err -eq ${ERR_OK} ] || return $err

	[ ${RPM_CONFIGURED} -eq -1 ] || return ${RPM_CONFIGURED}

	ARC_GET

	local rpms="4.2.1(x64) 4.1 4.0.2 3.0.5 4.2.2(x64)"
	local rpmi='';

	for rpmi in $rpms ; do
		PRINT_TO_LOG "${0##*/}:$LINENO Check if RPM supports relocateable packages - $rpmi"
		local ver=`echo $rpmi | sed s/\(.*\)//g`
		local arc=`echo $rpmi | sed s/.*\(//g | sed s/\)//g`
		if [ "$arc" = "$PLATFORM" ] || [ $arc"x" = $ver"x" ] ; then
			PRINT_TO_LOG
			PKG_CHECK rpm $ver eq
			if [ $? -eq 0 ] ; then
				PRINT_TO_LOG "${0##*/}:$LINENO Non-relocatable version of RPM. RPM version: $ver, ARC: $arc" 2
				PREFIX='';
				RPM_CONFIGURED=${ERR_RPM_NOT_RELOCATABLE}
				return ${ERR_RPM_NOT_RELOCATABLE}
			fi # if [ $? -eq 0 ] ; then
		fi # if [ "$arc" = "$PLATFORM" ] ; then
	done # for rpmi in $rpms ; do
	RPM_CONFIGURED=${ERR_OK};
	return ${ERR_OK}
} # RPM_CONFIG(){

TEMP_CREATE(){
	local locale=$1
	
	if [ "$locale" != "" ] ; then
		locale=`EXPAND_TREE "$locale"`
	else
		locale="/tmp"
	fi #if [ "$locale" != "" ]

	local temp=`mktemp -q ${locale}/temp.XXXXXXX`;
	local err=$?
	
	PRINT_TO_LOG "Create temp '${temp}' in '${locale}'." $err
	
	echo $temp
	return $err
} # INIT_LOGGING(){
	
UNINSTALL_RPM(){

	local ChosenRPM=$1
	shift
	local uninstall_options=$@
	local fcode=$ERR_OK

	RPM_CONFIG
	    
	local pack_ver=`RPM_INFO $ChosenRPM VERSION`
	fcode=$?
    
	[ $fcode -eq ${ERR_OK} ] || return $fcode
    
    	local TEMPFILE=`TEMP_CREATE`
	local err=$?
	[ $err -eq ${ERR_OK} ] || return $err
	
	rpm -vv -e $ChosenRPM $uninstall_options 2>&1 &> $TEMPFILE
	local err=$?
	
	if [ $err -ne 0 ] ; then 
		local RPM_LOGS=$(sed 's/^/    /g' $TEMPFILE)
		PRINT_TO_LOG "${0##*/}:$LINENO rpm-uninstall:UNINSTALL_RPM(): Uninstallation of the $ChosenRPM. RPM logs: \n$RPM_LOGS" 1
		fcode=$ERR_RPM_UNINSTALL
	else
		PRINT_TO_LOG "${0##*/}:$LINENO rpm-uninstall:UNINSTALL_RPM(): Uninstallation of the $ChosenRPM has succeeded."
		fcode=${ERR_OK}
	fi # if [ ! "$RPMERROR" = 0 ] ; then 

	rm -f $TEMPFILE
	return $fcode

} #UNINSTALL_RPM(){



NONRPM_INSTALL_PACKAGE() {

    local rpm_path="$1"     # A path to rpm file to be installed.
    local dst_dir="$2"      # A path to directory to install package to.

    local db="$INTEL_SDP_PRODUCTS_DB"
    local db_dir=$(dirname "$db") \
        || FATAL_EXIT "Unexpected error."
    local cur_dir=$(pwd) \
        || FATAL_EXIT "Unable to find current directory."
    local rpm_file=$(basename "$rpm_path") \
        || FATAL_EXIT "Unexpected error."
    local rpm_prefix=$(rpm -qp --qf %{prefixes} "$rpm_file") \
        || FATAL_EXIT "Unexpected errror."
    local rpm_name=$(rpm -qp --qf %{name} "$rpm_file") \
        || FATAL_EXIT "Unexpected errror."
    local log_file=".$rpm_file"_$(date +'%d%m%y_%H%M%S').log \
        || FATAL_EXIT "Unexpected errror."
    local tmp_dir=$(mktemp -d /tmp/install.XXXXXX) \
        || FATAL_EXIT "Unable to create temp directory in /tmp/."
    local src_dir="${tmp_dir}${rpm_prefix}"
    echo "Copying \"$rpm_path\" to temp directory..."
    cp "$rpm_path" "$tmp_dir" \
        || FATAL_EXIT "Unable to copy \"$rpm_path\" file to \"$tmp_dir\" directory."
    cd "$tmp_dir" \
        || FATAL_EXIT "Unable to change directory to \"$tmp_dir\"."
    echo "Extracting files from \"$rpm_file\" to temp directory..."
    rpm2cpio "$rpm_file" | cpio --quiet -idmu \
        || FATAL_EXIT "Unable to extract files from \"$rpm_file\"."
    find . -name uninstall.sh > uninstall.lst \
        || FATAL_EXIT "Unexpected error."
    wc -l uninstall.lst > 676767_898989tmp.tmp \
	|| FATAL_EXIT "Unexpected error."
    read lines lines_name < 676767_898989tmp.tmp \
	|| FATAL_EXIT "Unexpected error."
    [ "$lines" -eq "1" ] \
        || FATAL_EXIT "Unexpected error."
    rm -f 676767_898989tmp.tmp \
	|| FATAL_EXIT "Unexpected error."
    local uninstall=$(cat "uninstall.lst") \
        || FATAL_EXIT "Unexpected error."
    cp -p "$uninstall" "uninstall.sh.bak" \
        || FATAL_EXIT "Unable to copy \"$uninstall\" file to \"uninstall.sh.bak\"."
    chmod u+w "$uninstall" \
        || FATAL_ERROR "Unable to change mode on \"$uninstall\" file."
    sed s@'^RPM_INSTALLATION=1$'@'RPM_INSTALLATION='@g "uninstall.sh.bak" > "$uninstall" \
        || FATAL_EXIT "Unable to write \"$uninstall\" file."
    chmod --reference="uninstall.sh.bak" "$uninstall" \
        || FATAL_EXIT "Unable to change mode on \"$uninstall\" file."
    mkdir -p "$dst_dir" \
        || FATAL_EXIT "Unable to create \"$dst_dir\" directory."
    rpm2cpio "$rpm_file" | cpio --quiet -t | sed s@"^\.$rpm_prefix"@"$dst_dir"@g > "$tmp_dir/$log_file" \
        || FATAL_EXIT "Unable to create log file."
    mkdir -p "$db_dir" \
        || FATAL_EXIT "Unable to create \"$db_dir\" directory."
    cp "$tmp_dir/$log_file" "$db_dir" \
        || FATAL_EXIT "Unable to copy \"$tmp_dir/$log_file\" file to \"$db_dir\" directory."
    NONRPM_DB_ENTRY_ADD "<:$rpm_name:$rpm_file:$dst_dir:$db_dir/$log_file:>" \
        || FATAL_EXIT "Cannot add entry to database."
    cd "$src_dir" \
        || FATAL_EXIT "Unable to change directory to \"$src_dir\"."
    echo "Copying extracted files from temp directory to \"$dst_dir\" directory..."
    cp -prf * "$dst_dir" \
        || FATAL_EXIT "Unable to copy files to \"$dst_dir\" directory."
    cd "$cur_dir" \
        || FATAL_EXIT "Unable to change directory to \"$cur_dir\"."
    rm -rf "$tmp_dir" \
        || FATAL_EXIT "Unable to delete \"$tmp_dir\" temp directory."

} # NONRPM_INSTALL_PACKAGE




NONRPM_UNINSTALL_PACKAGE() {

    local entry="$1"

    if ! NONRPM_DB_ENTRY_CHECK_SYNTAX "$entry"; then
        echo "Specified db entry to uninstall does not look like a valid one. Probably, internal error."
        return 1
    fi
    
    local log_file=$(NONRPM_DB_ENTRY_GET_LOGFILE "$entry") \
        || FATAL_EXIT "Unexpected error"
    if [ ! -f "$log_file" ]; then
	echo "Uninstalltion cannot continue for this component: Missing \"$log_file\"."
	return 1
    fi
    
    echo "Removing files..." 
    local line           
    tac "$log_file" | \
        while read line; do
            if [ -f "$line" ]; then
                rm -f "$line"
            elif [ -d "$line" ]; then
                rmdir --ignore-fail-on-non-empty "$line"
            else 
                echo "Warning: installed \"$line\" file not found"    
            fi
        done
    rm -f "$log_file"
    
    NONRPM_DB_ENTRY_REMOVE "$entry"
    
    return 0

} # NONRPM_UNINSTALL_PACKAGE







FATAL_EXIT() {
    if [ -n "$LOG_FILE_NAME" ]; then
        echo "ERROR: Uninstallation failed. Please see \"$LOG_FILE_NAME\" for uninstallation details."
    fi
    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
	echo "Press Enter to continue."
        WAIT_ENTER_KEY
    fi
    exit 1
}

IS_INSTALLED()
{

    local package="$1"
    local product
    local rc
    if [ -n "$RPM_INSTALLATION" ]; then
        if [[ "$package" == *.rpm ]]; then 
            product=$(rpm -qp "$package")
        else
            product="$package"
        fi    
        rpm -qa | grep "$product" &> /dev/null
        rc=$?
    else
        NONRPM_DB_IS_PACKAGE_INSTALLED $(basename "$package")
        rc=$?
    fi
    return $rc

}

UNINSTALL_PRODUCT() {
    local rc=0
    IS_COMMAND_LINE_OPTION_EXISTS silent || echo "Uninstalling $CHOSEN_FULL_NAME..."
    if [ -n "$RPM_INSTALLATION" ]; then
        if [ -n "$CRYPTO_RPM" ]; then
            UNINSTALL_RPM "$CRYPTO_RPM" -e
            rc=$?
        fi
        if [[ $rc == 0 ]]; then
            UNINSTALL_RPM "$CHOSEN_RPM" -e
            rc=$?
        fi    
    else
        if [ -n "$CRYPTO_RPM" ]; then
            local crypto_entry=$(NONRPM_DB_ENTRY_FIND_BY_RPMFILE "$CRYPTO_RPM")
            if [ -n "$crypto_entry" ]; then
                NONRPM_UNINSTALL_PACKAGE "$crypto_entry"
                rc=$?
            fi
        fi
        if [ $rc == 0 ]; then
            NONRPM_UNINSTALL_PACKAGE "$ENTRY"
        fi
        rc=$?
    fi    
    if [ $rc -eq 0 ]; then
        IS_COMMAND_LINE_OPTION_EXISTS silent || echo "  Successful."
    else
        IS_COMMAND_LINE_OPTION_EXISTS silent || echo "  Unsuccessful."
        return 1
    fi
}

SCRIPT="$0"
RPM_INSTALLATION=1
CRYPTO_RPM=
CHECK_COMMANDS
CONFIG_ERRORS
CONFIG_STD_ENV
SAVE_COMMAND_LINE $@
PRINT_ERROR=0
INTEL_SDP_PRODUCTS_DB=~/intel/intel_sdp_products.db # TBD...

SCRIPT_ARGS=""
while [ $# -ne 0 ]
do
    SCRIPT_ARGS="$SCRIPT_ARGS \"$1\""
    shift
done

if [ -n "$RPM_INSTALLATION" ]; then   
    if ! IS_COMMAND_LINE_OPTION_EXISTS noroot; then
        if ! CHECK_ROOT_PRIVILEDGES "$SCRIPT_ARGS"; then
            echo "Uninstallation failed"
            exit 1
        fi
    fi
fi    

if [ -n "$RPM_INSTALLATION" ]; then
    RPM_LIST=$(echo $(rpm -qf "$0" | sort -r))
    CHOSEN_RPM=$(echo "$RPM_LIST" | cut -d' ' -f1)
else
    uninstall=$(cd $(dirname "$0") && pwd)/$(basename $0)
    ENTRY=$(NONRPM_DB_FIND_FILE_OWNER "$uninstall")
    CHOSEN_RPM=$(NONRPM_DB_ENTRY_GET_RPMFILE "$ENTRY")
fi

if [ -z "$UNINSTALL_LOG_FILE" ]; then
    UNINSTALL_LOG_FILE=/var/log/$(echo "$CHOSEN_RPM" | cut -d'-' -f2 | sed s/intel-//)_uninstall.log
fi
INIT_LOGGING "$UNINSTALL_LOG_FILE"

if [[ "$CHOSEN_RPM" == *ipp* ]]; then
    CRYPTO_RPM=$(echo "$CHOSEN_RPM" | sed s@ipp@crypto_ipp@)
    if ! IS_INSTALLED "$CRYPTO_RPM"; then
        CRYPTO_RPM=
    fi
fi

if IS_COMMAND_LINE_OPTION_EXISTS silent && IS_COMMAND_LINE_OPTION_EXISTS duplicate; then
    echo "ERROR: command line option --duplicate cannot be used when --silent is specified"
    exit 1
fi

if IS_COMMAND_LINE_OPTION_EXISTS duplicate; then
    SET_CONFIG_FILE_TO_DUPLICATE || FATAL_EXIT
fi

IS_COMMAND_LINE_OPTION_EXISTS silent || echo

IS_INSTALLED=$(echo "$CHOSEN_RPM" | grep intel)
if [ "$IS_INSTALLED" = "" ]; then 
    if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
	echo "Nothing to uninstall. Press Enter to exit..."
	WAIT_ENTER_KEY
    fi
    exit 1
fi

if [ -n "$RPM_INSTALLATION" ]; then
    CHOSEN_FULL_NAME=$(rpm -q "$CHOSEN_RPM" --qf %{summary})
    if [ -n "$CRYPTO_RPM" ]; then
        CHOSEN_FULL_NAME="$CHOSEN_FULL_NAME and "$(rpm -q "$CRYPTO_RPM" --qf %{summary})
    fi
else
    CHOSEN_FULL_NAME="$CHOSEN_RPM" # TBD.
    if [ -n "$CRYPTO_RPM" ]; then
        CHOSEN_FULL_NAME="$CHOSEN_FULL_NAME and $CRYPTO_RPM"
    fi
fi    

if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
    READ_YES_NO_ANSWER "Do you really want to uninstall $CHOSEN_FULL_NAME?" "Yes" "WANT_TO_UNINSTALL_PRODUCT"
    rc=$?
else
    rc=0
fi    
case $rc in
0)
    UNINSTALL_PRODUCT || FATAL_EXIT
    ;;
1)
    exit 0
    ;;
*)
    PRINT_TO_LOG "${0##*/}:$LINENO: couldn't recognize user's answer on the question id 'WANT_TO_UNINSTALL_PRODUCT' - probably wrong entry value in the config file" 1
    FATAL_EXIT
    ;;
esac

if IS_COMMAND_LINE_OPTION_EXISTS duplicate; then
    WRITE_ANSWERS_TO_CONFIG_FILE || FATAL_EXIT
fi

if ! IS_COMMAND_LINE_OPTION_EXISTS silent; then
    if [ -n "$LOG_FILE_NAME" ]; then
        echo "$CHOSEN_FULL_NAME was successfully uninstalled. Please see \"$LOG_FILE_NAME\" for uninstallation details."
    fi
    echo "Press Enter to continue."
    WAIT_ENTER_KEY
fi


