/*
    This file is part of KOrganizer.
    Copyright (c) 2001 Cornelius Schumacher <schumacher@kde.org>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

    As a special exception, permission is given to link this program
    with any edition of Qt, and distribute the resulting executable,
    without including the source code for Qt in the source distribution.
*/

//
// Journal Entry

#include <qlabel.h>
#include <qlayout.h>
#include <qvbox.h>
#include <qfile.h>
#include <qdir.h>
#include <qtextstream.h>
#include <qtextcodec.h>
#include <qpixmap.h>
#include <qpushbutton.h>
#include <qapplication.h>

#include <kdebug.h>
#include <kglobal.h>
#include <klocale.h>
#include <ktextedit.h>
#include <kfiledialog.h>
#include <kmessagebox.h>
#include "koprefs.h"

#include <libkcal/journal.h>
#include <libkcal/calendarresources.h>
#include <libkcal/resourcecalendar.h>
#include <kresources/resourceselectdialog.h>

#include "journalentry.h"
//#include "journalentry.moc"
#ifndef DESKTOP_VERSION
#include <qpe/qpeapplication.h>
#endif
JournalEntry::JournalEntry(Calendar *calendar,QWidget *parent) :
  QFrame(parent)
{
  mCalendar = calendar;
  mJournal = 0;
  mDirty = false;

  QHBox * vb = new QHBox ( this );
  QPushButton * loadTemplate = new QPushButton( vb );
  QPushButton * saveTemplate = new QPushButton( vb );
  QIconSet icon;
  if ( QApplication::desktop()->width() < 321 ) 
      icon = SmallIcon("fileexport16");
  else
      icon = SmallIcon("fileexport");
  saveTemplate->setIconSet (icon ) ;
  int size = saveTemplate->sizeHint().height();
  saveTemplate->setFixedSize(  size, size );
  if ( QApplication::desktop()->width() < 321 ) 
      icon = SmallIcon("fileimport16");
  else
      icon = SmallIcon("fileimport");
  loadTemplate->setIconSet (icon ) ;
  loadTemplate->setFixedSize(  size, size );

  mTitleLabel = new QLabel(i18n("Title"),vb);
  mTitleLabel->setMargin(2);
  mTitleLabel->setAlignment(AlignCenter);
  
  mEditor = new KTextEdit(this);
  connect(mEditor,SIGNAL(textChanged()),SLOT(setDirty()));
#ifndef DESKTOP_VERSION
  QPEApplication::setStylusOperation( mEditor, QPEApplication::RightOnHold );
#endif
  mEditor->setWordWrap( KTextEdit::WidgetWidth );
  QBoxLayout *topLayout = new QVBoxLayout(this);
  topLayout->addWidget(vb);
  topLayout->addWidget(mEditor);
  mEditor->installEventFilter(this);
  connect( saveTemplate, SIGNAL( clicked() ), this , SLOT( slotSaveTemplate() ) );
  connect( loadTemplate, SIGNAL( clicked() ), this , SLOT( slotLoadTemplate() ) );
}

JournalEntry::~JournalEntry()
{
}

void JournalEntry::slotSaveTemplate()
{
    QString fileName =locateLocal( "templates", "journals" ); 
    QDir t_dir;
    if ( !t_dir.exists(fileName) )
        t_dir.mkdir ( fileName );
    fileName += "/journal";
    fileName = KFileDialog::getSaveFileName(  fileName , i18n("Save as Journal template"), this );
    if ( fileName.length() == 0 )
        return;

    QFile fileIn( fileName );
    if (!fileIn.open( IO_WriteOnly ) ) {
        KMessageBox::error( this, i18n("Error saving template file\n '%1'.")
                            .arg( fileName ) );
        return;
    } 
    // QString text;
    QTextStream tsIn( &fileIn );
    tsIn.setCodec( QTextCodec::codecForName("utf8") );
    tsIn << mEditor->text();
    fileIn.close();  
}
void JournalEntry::slotLoadTemplate()
{
    QString fileName =locateLocal( "templates", "journals" ); 
    QDir t_dir;
    if ( !t_dir.exists(fileName) )
        t_dir.mkdir ( fileName );
    fileName += "/journal";
    fileName = KFileDialog::getOpenFileName(  fileName , i18n("Insert Journal template"), this );
    if ( fileName.length() == 0 )
        return;
    QFile fileIn( fileName );
    if (!fileIn.open( IO_ReadOnly ) ) {
        KMessageBox::error( this, i18n("Error loading template file\n '%1'.")
                            .arg( fileName ) );
        return;
    }
    QTextStream tsIn( &fileIn );
    tsIn.setCodec( QTextCodec::codecForName("utf8") );
    QString text =  tsIn.read();
    fileIn.close();  
    int line, col;
    mEditor->getCursorPosition (& line, & col );
    mEditor-> insertAt ( text, line, col, true ); 
    //mEditor->setIgnoreMark( true );
    setDirty();
}
void JournalEntry::setDate(const QDate &date)
{
  writeJournal();

  mTitleLabel->setText(KGlobal::locale()->formatDate(date));

  
  mDate = date;
}

void JournalEntry::setJournal(Journal *journal)
{
  writeJournal();

  mJournal = journal;
  
  mEditor->setText(mJournal->description());

  mDirty = false;
}

Journal *JournalEntry::journal() const
{
  return mJournal;
}

void JournalEntry::setDirty()
{
  mDirty = true;

//  kdDebug() << "JournalEntry::setDirty()" << endl;
}

void JournalEntry::clear()
{
  mJournal = 0;
  mEditor->setText("");
}

bool JournalEntry::eventFilter( QObject *o, QEvent *e )
{
//  kdDebug() << "JournalEntry::event received " << e->type() << endl;

  if ( e->type() == QEvent::FocusOut ) {
    writeJournal();
  }
  if ( e->type() == QEvent::KeyPress ) {
      QKeyEvent * k = (QKeyEvent *) e;
      if ( k->state() == Qt::ControlButton ) {
          k->ignore();
          //return true;
      }
  }

  return QFrame::eventFilter( o, e );    // standard event processing
}

void JournalEntry::writeJournal()
{
//  kdDebug() << "JournalEntry::writeJournal()" << endl;
  if (!mDirty) return;
 
  if (mEditor->text().isEmpty()) {
      if ( mJournal ) {
          mDirty = false;
          bool conf = KOPrefs::instance()->mConfirm;
          KOPrefs::instance()->mConfirm = false;
          emit deleteJournal(mJournal);
          KOPrefs::instance()->mConfirm = conf;
          mJournal = 0;
      }
      return;
  }

//  kdDebug() << "JournalEntry::writeJournal()..." << endl;
  
  if (!mJournal) {
    mJournal = new Journal;
    mJournal->setDtStart(QDateTime(mDate,QTime(0,0,0)));
    mCalendar->addJournal(mJournal);
  }

  mJournal->setDescription(mEditor->text());
  //qDebug("tttt%s ", mEditor->text().latin1());

  mDirty = false;
}

void JournalEntry::flushEntry()
{
  if (!mDirty) return;
  
  writeJournal();
}
void JournalEntry::keyPressEvent ( QKeyEvent * e ) 
{
    e->ignore();

}
