/*
 *	Watchdog driver for the LH79520
 *
 *      (c) Copyright 2000 Oleg Drokin <green@crimea.edu>
 *          Based on SoftDog driver by Alan Cox <alan@redhat.com>
 *
 *	This program is free software; you can redistribute it and/or
 *	modify it under the terms of the GNU General Public License
 *	as published by the Free Software Foundation; either version
 *	2 of the License, or (at your option) any later version.
 *
 *	Neither Oleg Drokin nor iXcelerator.com admit liability nor provide
 *	warranty for any of this software. This material is provided
 *	"AS-IS" and at no charge.
 *
 *	(c) Copyright 2000           Oleg Drokin <green@crimea.edu>
 *
 *      27/11/2000 Initial release
 */

#include <linux/module.h>
#include <linux/config.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/miscdevice.h>
#include <linux/watchdog.h>
#include <linux/reboot.h>
#include <linux/smp_lock.h>
#include <linux/init.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/bitops.h>

#include "lh79520_wdt.h"

unsigned int hclkfreq_get( void);

#define TIMER_MARGIN	60	/* default in seconds */
#ifdef OLDWAY
#define PCLK		51609600 /* ticks per second of AHB clock, 51MHz */
#else
#define PCLK		hclkfreq_get()
#endif 
#define SLEEP_TIME	10	/* number of seconds between each counter reload */

static int lh79520_margin = TIMER_MARGIN;	/* in seconds */
static int lh79520wdt_users;			/* mutex */
#ifdef FIQ_ENABLED
static int irq = 0x1c; /* FIQ. Normally 0x18 for standard irq */
#else
static int irq = 0x18; /* IRQ. Normally the FIQ is 0x1c */
#endif

#define WDTBase 0xFFFE3000L	/* Base Address for all LH79520 Watchdog Registers */
WDTIMERREGS *wtdregs = (WDTIMERREGS *) WDTBase;

#ifdef MODULE
MODULE_PARM(lh79520_margin,"i");
#endif

/*
 *	Allow only one person to hold it open
 */

static int lh79520dog_open(struct inode *inode, struct file *file)
{
	if(test_and_set_bit(1,&lh79520wdt_users))
		return -EBUSY;
	MOD_INC_USE_COUNT;

	if ((lh79520_margin > (PCLK / 0xffffffff)) ||  /* 83 seconds max, 20 sec min margin */
	    (lh79520_margin <= SLEEP_TIME * 2))
		lh79520_margin = TIMER_MARGIN;

/* setting bits 7-4 of WDCTLR to 0x0 through 0xF sets the *initial* counter value upon a reset */
/* 0x0 is 2^16 tics of PCLK, or reset immediatly,  0x10 is 2^17 tics of PCLK, ... */
/* 0xF is 2^31 tics of PCLK, if PCLK is 51 MHz, 2^31 / 51 MHz = 41.6 seconds */
/* WDCTLR=0xF, the F sets initial counter to 41.6 seconds (assuming PCLK is 51 MHz) */
/* The counter will be set to the user selected margin the first time a reload occurs */
	wtdregs->wdctlr |= 0xF0;

	/* Activate LH79520 Watchdog timer */
	wtdregs->wdctlr |= WDT_CTRL_ENABLE;
	///wtdregs->wdctlr |= WDT_CTRL_FRZ_ENABLE;
	// The 1 sets the enable bit (bit 0) to 1 enabling the watchdog fuctionality
	// The freeze or lock bit (bit 4) makes bit 0 read-only (to avoid accidental disabling)
	// Bit 1 is left at 0 signifing that when the counter reaches 0, a machine reset occurs
	// If bit 1 were 1, then the first time the counter reached 0 an interrupt occurs, and
	// the second time the counter reaches 0 the machine is reset.

	// Now reset Watchdog to let the above settings take effect
	wtdregs->wdcntr = WDT_WDCNTR;  // 0x1984 is a special reset value
	return 0;
}

static int lh79520dog_release(struct inode *inode, struct file *file)
{
	/*
	 *	Shut off the timer.
	 * 	Lock it in if it's a module and we defined ...NOWAYOUT
	 */
	///wtdregs->wdctr |= 0x0;   // turns off bit 4 the freeze lock so we can write to bit 0
	wtdregs->wdctlr |= WDT_CTRL_DISABLE;   // turns off watchdog bit 0

	lh79520wdt_users = 0;
	MOD_DEC_USE_COUNT;
	return 0;
}

static ssize_t lh79520dog_write(struct file *file, const char *data, size_t len, loff_t *ppos)
{
	/*  Can't seek (pwrite) on this device  */
	if (ppos != &file->f_pos)
		return -ESPIPE;

	/* Refresh/reload counter */
	if(len) {
		unsigned int count = (lh79520_margin * PCLK);
		wtdregs->wdcnt3 = count && 0xFF000000;
		wtdregs->wdcnt2 = count && 0x00FF0000;
		wtdregs->wdcnt1 = count && 0x0000FF00;
		wtdregs->wdcnt0 = count && 0x000000FF;
		return 1;
	}
	return 0;
}

static int lh79520dog_ioctl(struct inode *inode, struct file *file,
	unsigned int cmd, unsigned long arg)
{
	static struct watchdog_info ident = {
		identity: "LH79520 Watchdog",
	};

	switch(cmd){
	default:
		return -ENOIOCTLCMD;
	case WDIOC_GETSUPPORT:
		return copy_to_user((struct watchdog_info *)arg, &ident, sizeof(ident));
	case WDIOC_GETSTATUS:
		return put_user(0,(int *)arg);
	case WDIOC_GETBOOTSTATUS: /* 1 = last reboot was cause by watchdog, 0 means no */
		return put_user( ! (wtdregs->wdtstr & WDT_WD_NWDRES), (int *)arg);
	case WDIOC_KEEPALIVE:
		{
		unsigned int count = (lh79520_margin * PCLK);
		wtdregs->wdcnt3 = count && 0xFF000000;
		wtdregs->wdcnt2 = count && 0x00FF0000;
		wtdregs->wdcnt1 = count && 0x0000FF00;
		wtdregs->wdcnt0 = count && 0x000000FF;
		}
		return 0;
	}
}

/**
 *	lh79520dog_interrupt:
 *	@irq:		Interrupt number
 *	@dev_id:	Unused as we don't allow multiple devices.
 *	@regs:		Unused.
 *
 *	Handle an interrupt from the board. These are raised when the status
 *	map changes in what the board considers an interesting way. That means
 *	a failure condition occuring.
 */

void lh79520dog_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	/*
	 *	Read the status register see what is up and
	 *	then printk it.
	 */
	
	unsigned char status=wtdregs->wdtstr;
	
/*	status|=FEATUREMAP1;
	status&=~FEATUREMAP2;	*/
	
	printk(KERN_CRIT "WDT status %d\n", status);
	
    wtdregs->wdcntr = WDT_WDCNTR;  // 0x1984 is a special reset value
/*
	if(!(status&WDC_SR_TGOOD))
		printk(KERN_CRIT "Overheat alarm.(%d)\n",inb_p(WDT_RT));
	if(!(status&WDC_SR_PSUOVER))
		printk(KERN_CRIT "PSU over voltage.\n");
	if(!(status&WDC_SR_PSUUNDR))
		printk(KERN_CRIT "PSU under voltage.\n");
	if(!(status&WDC_SR_FANGOOD))
		printk(KERN_CRIT "Possible fan fault.\n");
	if(!(status&WDC_SR_WCCR))
#ifdef SOFTWARE_REBOOT
#ifdef ONLY_TESTING
		printk(KERN_CRIT "Would Reboot.\n");
#else		
		printk(KERN_CRIT "Initiating system reboot.\n");
		machine_restart(NULL);
#endif		
#else
		printk(KERN_CRIT "Reset in 5ms.\n");
#endif	
*/	
}


static struct file_operations lh79520dog_fops=
{
	owner:		THIS_MODULE,
	write:		lh79520dog_write,
	ioctl:		lh79520dog_ioctl,
	open:		lh79520dog_open,
	release:	lh79520dog_release,
};

static struct miscdevice lh79520dog_miscdev=
{
	WATCHDOG_MINOR,
	"LH79520 watchdog",
	&lh79520dog_fops
};

static int __init lh79520dog_init(void)
{
	int ret;

	ret = misc_register(&lh79520dog_miscdev);

	if (ret)
    {
		goto out;
    }

//	ret = request_irq(irq, lh79520dog_interrupt, SA_INTERRUPT, "lh79520wdt", NULL);
//	if(ret) {
//		printk(KERN_ERR "wdt: IRQ %d is not free.\n", irq);
//		goto outmisc;
//	}

    printk("LH79520 Watchdog Timer: timer margin %d sec\n", lh79520_margin);

    ret=0;
    out:
	return ret;

    outmisc:
	misc_deregister(&lh79520dog_miscdev);
	goto out;

}

static void __exit lh79520dog_exit(void)
{
	misc_deregister(&lh79520dog_miscdev);
}

module_init(lh79520dog_init);
module_exit(lh79520dog_exit);
