/*
 *  linux/arch/arm/mach-s3c2410/irq.c
 *
 *  Copyright (C) 2002 SAMSUNG ELECTRONICS 
 *                   SW.LEE (hitchcar@sec.samsung.com)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */


#include <linux/init.h>
#include <linux/module.h>
#include <linux/sched.h>
#include <linux/interrupt.h>
#include <linux/ioport.h>
#include <linux/ptrace.h>

#include <asm/hardware.h>
#include <asm/irq.h> /* contains asm/arch/irqs.h */
#include <asm/mach/irq.h>
#include <asm/arch/irq.h>


static void s3c2410_mask_ack_irq(unsigned int irq)
{
        rINTMSK |= ((unsigned long)1 << irq);
        rSRCPND = ((unsigned long)1 << irq);
	rSRCPND;		/* WriteBack  */
        rINTPND = ((unsigned long)1 << irq);
	rINTPND;		/* WriteBack */
}

static void s3c2410_mask_irq(unsigned int irq)
{
        rINTMSK |= ((unsigned long) 1 << irq);
}

static void s3c2410_unmask_irq(unsigned int irq)
{
        rINTMSK &= ~((unsigned long)1 << irq);
}

static void EINT4_23mask_ack_irq(unsigned int irq)
{
        irq = (irq-NoIntBase)+4;
        rEINTMSK  |= ((unsigned long)1 << irq);
        rEINTPND  = ((unsigned long)1 << irq);
	rEINTPND;		/* WriteBack */
	if(irq<8) {
		ClearPending(1<<BIT_SHIFT_EINT4_7);
	} else {
		ClearPending(1<<BIT_SHIFT_EINT8_23);
	}
}

static void EINT4_23mask_irq(unsigned int irq)
{
        irq = (irq-NoIntBase)+4;
        rEINTMSK |= ((unsigned long) 1 << irq);
}

static void EINT4_23unmask_irq(unsigned int irq)
{
        rEINTMSK &= ~((unsigned long)1 << ((irq-NoIntBase)+4));
	if ( irq <= IRQ_EINT7){ 
		rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_EINT4_7);
	}
	else  {
		rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_EINT8_23);
	}
}

static void SUB_mask_ack_irq(unsigned int irq)
{
        irq = irq-IRQ_RXD0;

        rINTSUBMSK |= ((unsigned long)1 << irq);
        rSUBSRCPND =  ((unsigned long)1 << irq);
	rSUBSRCPND;		/* WriteBack */

	switch(irq) {
	case 0:	case 1:	case 2:
		ClearPending(1<<BIT_SHIFT_UART0);break;
	case 3:case 4: case 5:
		ClearPending(1<<BIT_SHIFT_UART1);break;
	case 6:case 7: case 8:
	  	ClearPending(1<<BIT_SHIFT_UART2);break;
	case 9: case 10:
	  	ClearPending(1<<BIT_SHIFT_ADC);break;
	default:
		printk(KERN_ERR "In SUB_mask_ack_irq:  Unexpected Error  \n");
	}
}


static void SUB_mask_irq(unsigned int irq)
{
        irq = irq - IRQ_RXD0;
        rINTSUBMSK |= ((unsigned long) 1 << irq);
}

static void SUB_unmask_irq(unsigned int irq)
{

        rINTSUBMSK &= ~((unsigned long)1 << (irq-IRQ_RXD0));

	if ( irq <= IRQ_ERR0) {
		rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_UART0); 
	    	return;
        }
	if (irq<= IRQ_ERR1) {
		rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_UART1);
	     	return;
	}
	if(irq <=IRQ_ERR2){
	    	rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_UART2);
	    	return;
        }
        if ( irq <= IRQ_ADCDone ) {
	    	rINTMSK &= ~((unsigned long)1<<BIT_SHIFT_ADC);
           	return ;
        }
}


static void dummy_mask_ack_irq(unsigned int irq)
{
	//rEINTPND = 0xfffff0;
	//rEINTPND;
}

static void dummy_mask_irq(unsigned int irq)
{
}

static void dummy_unmask_irq(unsigned int irq)
{
}

/*
 * For SpeedUP, Split_EINTX_X will n't combine into Split_EINT 
 * Because EINT4_7 is widely used.
 * return 0 means no interrupt
 * SEE arm/kernel/irq.c "do_IRQ function" to know how to use Split_EINTX_X func.
 */
extern unsigned int Split_EINT4_7(int irq)
{
	int i;
	/* Discard parameter irq */
	irq = (rEINTPND & ~rEINTMSK & 0x000000f0);
	if(!irq) return IRQ_DUMMY;
	for (i = 4; i <= 7; ++i)  {
		if (irq & (1<<i)) {
			return (NoIntBase+(i-4));
		}
	}
}

extern unsigned int Split_EINT8_23(int irq)
{
	int i;
	/* Discard parameter irq */
	irq = (rEINTPND & ~rEINTMSK & 0x00ffff00);
	if(!irq) return IRQ_DUMMY;
	for (i = 8; i <= 23; ++i) { 
		if (irq & (1<<i)) {
			return (NoIntBase+(i-4));
		}
	}
}

/*
 * For SpeedUP, Split_Uart[x] will n't combine into Split_Uart 
 * Because Uart0 is widely used.
 */
extern unsigned int Split_Uart0(int irq)
{
	int i;
	/* Discard parameter irq */
	irq = (rSUBSRCPND&0x3);
	for (i = 0; i <= 2; ++i) 
	  if (irq & (1<<i)) break;
	return (IRQ_RXD0+i);
}

extern unsigned int Split_Uart1(int irq)
{
	int i;
	/* Discard parameter irq */
	irq = (rSUBSRCPND&0x38);
	for (i = 3; i <= 5; ++i) 
	  if (irq & (1<<i)) break;
	return (IRQ_RXD0+i);
}

extern unsigned int Split_Uart2(int irq)
{
	int i;
	/* Discard parameter irq */
	irq = (rSUBSRCPND&0x1C0);
	for (i = 6; i <= 8; ++i)  
	  if (irq & (1<<i)) break;
	return (IRQ_RXD0+i);
}

extern unsigned int Split_ADC(int irq)
{
	int i;
	irq = (rSUBSRCPND&0x600);
	if(irq&0x200)  { 
		return IRQ_TC; 
	} 
	else {
		if(irq&0x400) return IRQ_ADCDone;
	 	else {
			printk("In Split_ADC :  UNEXPECTED IRQ NUMBER   \n");
			return NR_IRQS +1 ;
		}
	}
}

/* YOU CAN CHANGE THIS ROUTINE FOR SPEED UP */
extern unsigned int fixup_irq (int i )
{
	unsigned int RetVal;
	if ( i == OS_TIMER ) {
	  RetVal = i;
	}
	else {
 		switch ( i) {
  			case IRQ_UART0:    RetVal=Split_Uart0(i); break;
			case IRQ_UART1:    RetVal=Split_Uart1(i);break;
 			case IRQ_UART2:    RetVal=Split_Uart2(i);break;
  			case IRQ_ADC:      RetVal=Split_ADC(i);break;
			default:   	   RetVal=i;break;
 		}
        }
	return RetVal;
}

extern void __init s3c2410_init_irq(void)
{
	unsigned long flags;
        int irq;

//	request_resource(&iomem_resource, &irq_resource);

	/****** 
	 *  save_flags_cli differ from 2.2.13 save_flags_cli  *^^*   SWLEE 
         *  cli means that disables IRQ  Not FIQ  by means of value 128 (10000000) 
         *************************************************/
	save_flags_cli (flags);
        /* Disable all IRQs */

	rINTSUBMSK = 0x7ff;
        rINTMSK = 0xffffffff; /* all masked */
        /****************
         * All IRQs are IRQ, not FIQ 
         * Write only one register,
         * 0 : IRQ mode
         * 1 : FIQ mode
         *******************************************/
        rINTMOD = 0x00000000; 

	/* rEINTPEND is smilar to INTSRC  */
	rEINTPND = 0xfffff0;
	restore_flags (flags);

	/* !!!!! VERY IMPORTANT !!!!! 
	 * YOU MUST CHECK IO PORT REGISTER in asm/arch/uncompress.h 
	 * We must fix up GPFCON[0:7] to EINT0,1,2,3 
         *
         * rExternal interrupt will be falling edge triggered.  
         *  except CS8900A Network Interrupt  	
	 */

        for (irq=0; irq <= IRQ_ADC;irq++) {
                irq_desc[irq].valid     = 1;
                irq_desc[irq].probe_ok  = 1;
                irq_desc[irq].mask_ack  = s3c2410_mask_ack_irq;
                irq_desc[irq].mask      = s3c2410_mask_irq;
                irq_desc[irq].unmask    = s3c2410_unmask_irq;
        }

	for ( irq =IRQ_EINT4;irq <= IRQ_EINT23;irq++) {
	        irq_desc[irq].valid     = 1;
                irq_desc[irq].probe_ok  = 1;
                irq_desc[irq].mask_ack  = EINT4_23mask_ack_irq;
                irq_desc[irq].mask      = EINT4_23mask_irq;
                irq_desc[irq].unmask    = EINT4_23unmask_irq;
	}

	for ( irq =IRQ_RXD0;irq <IRQ_DUMMY;irq++) {
	        irq_desc[irq].valid     = 1;
                irq_desc[irq].probe_ok  = 1;
                irq_desc[irq].mask_ack  = SUB_mask_ack_irq;
                irq_desc[irq].mask      = SUB_mask_irq;
                irq_desc[irq].unmask    = SUB_unmask_irq;
	}  

#if 1
	{	/*  irq == IRQ_DUMMY */
	        irq_desc[irq].valid     = 1;
                irq_desc[irq].probe_ok  = 1;
                irq_desc[irq].mask_ack  = dummy_mask_ack_irq;
                irq_desc[irq].mask      = dummy_mask_irq;
                irq_desc[irq].unmask    = dummy_unmask_irq;
	}
#endif
#if 0
	irq_desc[IRQ_EINT4_7].valid	= 1;
	irq_desc[IRQ_EINT4_7].probe_ok	= 1;
	irq_desc[IRQ_EINT4_7].mask_ack	= EINT4_7_mask_ack_irq;
	irq_desc[IRQ_EINT4_7].mask	= EINT4_7_mask_irq;
	irq_desc[IRQ_EINT4_7].unmask	= EINT4_7_unmask_irq;

	irq_desc[IRQ_EINT8_23].valid	= 1;
	irq_desc[IRQ_EINT8_23].probe_ok	= 1;
	irq_desc[IRQ_EINT8_23].mask_ack	= EINT8_23_mask_ack_irq;
	irq_desc[IRQ_EINT8_23].mask	= EINT8_23_mask_irq;
	irq_desc[IRQ_EINT8_23].unmask	= EINT8_23_unmask_irq;
#endif
}
