/*
 * Hardware definitions for HP iPAQ Handheld Computers
 *
 * Copyright 2000-2003 Hewlett-Packard Company.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Jamey Hicks.
 *
 * History:
 *
 * 2003-05-14	Joshua Wise        Adapted for the HP iPAQ H1900
 * 2002-08-23   Jamey Hicks        Adapted for use with PXA250-based iPAQs
 * 2001-10-??   Andrew Christian   Added support for iPAQ H3800
 *                                 and abstracted EGPIO interface.
 *
 */
#include <linux/module.h>
#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/tty.h>
#include <linux/sched.h>
#include <linux/pm.h>
#include <linux/bootmem.h>

#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/setup.h>

#include <asm/mach/irq.h>
#include <asm/mach/arch.h>
#include <asm/mach/map.h>
#include <asm/arch/h3900-init.h>
#include <asm/arch/h3900_asic.h>

#include <asm/arch/irq.h>
#include <asm/types.h>

#include <linux/serial_core.h>

#include "generic.h"

#define SET_ASIC3(x) \
   do {if ( setp ) { H3900_ASIC3_GPIO_B_OUT |= (x); } else { H3900_ASIC3_GPIO_B_OUT &= ~(x); }} while(0)

#define SET_ASIC2(x) \
   do {if ( setp ) { H3800_ASIC2_GPIOPIOD |= (x); } else { H3800_ASIC2_GPIOPIOD &= ~(x); }} while(0)

#define CLEAR_ASIC3(x) \
   do {if ( setp ) { H3900_ASIC3_GPIO_B_OUT &= ~(x); } else { H3900_ASIC3_GPIO_B_OUT |= (x); }} while(0)

#define CLEAR_ASIC2(x) \
   do {if ( setp ) { H3800_ASIC2_B_GPIOPIOD &= ~(x); } else { H3800_ASIC2_B_GPIOPIOD |= (x); }} while(0)


static void msleep(unsigned int msec)
{
	current->state = TASK_INTERRUPTIBLE;
	schedule_timeout( (msec * HZ + 999) / 1000);
}

/*
  On screen enable, we get 
  
     h3800_video_power_on(1)
     LCD controller starts
     h3800_video_lcd_enable(1)

  On screen disable, we get
  
     h3800_video_lcd_enable(0)
     LCD controller stops
     h3800_video_power_on(0)
*/

static void h3900_video_power_on( int setp )
{
	if ( setp ) {
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_LCD_ON;
		msleep(30);
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_LCD_NV_ON;
		msleep(5);
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_LCD_9V_ON;
		msleep(50);
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_LCD_5V_ON;
		msleep(5);
	} else {
		msleep(5);
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_LCD_5V_ON;
		msleep(50);
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_LCD_9V_ON;
		msleep(5);
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_LCD_NV_ON;
		msleep(100);
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_LCD_ON;
	}
}

static void h3900_video_lcd_enable( int setp )
{
	if ( setp ) {
		msleep(17);     // Wait one from before turning on
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_LCD_PCI;
	} else {
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_LCD_PCI;
		msleep(30);     // Wait before turning off
	}
}


static void h3900_control_egpio( enum ipaq_egpio_type x, int setp )
{
	switch (x) {
	case IPAQ_EGPIO_LCD_POWER:
		h3900_video_power_on( setp );
		break;
	case IPAQ_EGPIO_LCD_ENABLE:
		h3900_video_lcd_enable( setp );
		break;
	case IPAQ_EGPIO_CODEC_NRESET:
	case IPAQ_EGPIO_AUDIO_ON:
	case IPAQ_EGPIO_QMUTE:
		printk("%s: error - should not be called\n", __FUNCTION__);
		break;
	case IPAQ_EGPIO_OPT_NVRAM_ON:
		SET_ASIC2( GPIO2_OPT_ON_NVRAM );
		break;
	case IPAQ_EGPIO_OPT_ON:
		SET_ASIC2( GPIO2_OPT_ON );
		break;
	case IPAQ_EGPIO_CARD_RESET:
		SET_ASIC2( GPIO2_OPT_PCM_RESET );
		break;
	case IPAQ_EGPIO_OPT_RESET:
		SET_ASIC2( GPIO2_OPT_RESET );
		break;
	case IPAQ_EGPIO_IR_ON:
		CLEAR_ASIC3( GPIO3_IR_ON_N );
		break;
	case IPAQ_EGPIO_IR_FSEL:
		break;
	case IPAQ_EGPIO_RS232_ON:
		SET_ASIC3( GPIO3_RS232_ON );
		break;
	case IPAQ_EGPIO_BLUETOOTH_ON:
		SET_ASIC3( GPIO3_BT_PWR_ON );
		break;
	case IPAQ_EGPIO_VPP_ON:
		if (setp)
			GPSR(GPIO_NR_H3900_FLASH_VPEN) = GPIO_H3900_FLASH_VPEN;
		else
			GPCR(GPIO_NR_H3900_FLASH_VPEN) = GPIO_H3900_FLASH_VPEN;
		break;
	default:
		printk("%s: unhandled egpio=%d\n", __FUNCTION__, x);
	}
}

static unsigned long h3900_read_egpio( enum ipaq_egpio_type x)
{
	switch (x) {
	case IPAQ_EGPIO_PCMCIA_CD0_N:
		return(GPLR(GPIO_NR_H3900_PCMCIA_CD0_N) & GPIO_bit(GPIO_NR_H3900_PCMCIA_CD0_N));
	case IPAQ_EGPIO_PCMCIA_CD1_N:
		return(GPLR(GPIO_NR_H3900_PCMCIA_CD1_N) & GPIO_bit(GPIO_NR_H3900_PCMCIA_CD1_N));
	case IPAQ_EGPIO_PCMCIA_IRQ0:
		return(GPLR(GPIO_NR_H3900_PCMCIA_IRQ0_N) & GPIO_bit(GPIO_NR_H3900_PCMCIA_IRQ0_N));
	case IPAQ_EGPIO_PCMCIA_IRQ1:
		return(GPLR(GPIO_NR_H3900_PCMCIA_IRQ1_N) & GPIO_bit(GPIO_NR_H3900_PCMCIA_IRQ1_N));
        default:
		printk("%s:%d: unknown ipaq_egpio_type=%d\n", __FUNCTION__, __LINE__, x);
		return 0;
	}
}

/* We need to fix ASIC2 GPIO over suspend/resume.  At the moment,
   it doesn't appear that ASIC3 GPIO has the same problem */

static int h3900_pm_callback( int req )
{
	static u16 asic3_data;
	static u16 asic2_data;
	static u8 led_control_0;

	int result = 0;

#ifdef DEBUG_PM	
	printk("%s %d\n", __FUNCTION__, req);
#endif

	switch (req) {
	case PM_RESUME:
		H3800_ASIC2_GPIOPIOD = asic2_data;
		H3800_ASIC2_GPIODIR = GPIO2_PEN_IRQ 
			| GPIO2_SD_DETECT
			| GPIO2_EAR_IN_N 
			| GPIO2_USB_DETECT_N 
			| GPIO2_SD_CON_SLT;

		H3900_ASIC3_GPIO_B_OUT = asic3_data;

		H3800_ASIC2_LED_TimeBase(0) = led_control_0;

		if ( ipaq_model_ops.pm_callback_aux )
			result = ipaq_model_ops.pm_callback_aux(req);
		break;

	case PM_SUSPEND:
		if ( ipaq_model_ops.pm_callback_aux &&
		     ((result = ipaq_model_ops.pm_callback_aux(req)) != 0))
			return result;

		asic3_data = H3900_ASIC3_GPIO_B_OUT;
		asic2_data = H3800_ASIC2_GPIOPIOD;

		H3900_ASIC3_GPIO_B_OUT = (asic3_data & ~GPIO3_SLEEP_FORCE0) | GPIO3_SLEEP_FORCE1;

		/* make sure blue LED is off.  leave yellow and green LEDs alone. */
		led_control_0 = H3800_ASIC2_LED_TimeBase(0);
		H3800_ASIC2_LED_TimeBase(0) = 0;
		break;
	default:
		printk("%s: unrecognized PM callback\n", __FUNCTION__);
		break;
	}
	return result;
}



struct egpio_irq_info h3900_egpio_irq_info[] = {
	{ IPAQ_EGPIO_PCMCIA_CD0_N, GPIO_NR_H3900_PCMCIA_CD0_N, IRQ_GPIO_H3900_PCMCIA_CD0 }, 
	{ IPAQ_EGPIO_PCMCIA_CD1_N, GPIO_NR_H3900_PCMCIA_CD1_N, IRQ_GPIO_H3900_PCMCIA_CD1 },
	{ IPAQ_EGPIO_PCMCIA_IRQ0,  GPIO_NR_H3900_PCMCIA_IRQ0_N, IRQ_GPIO_H3900_PCMCIA_IRQ0 },
	{ IPAQ_EGPIO_PCMCIA_IRQ1,  GPIO_NR_H3900_PCMCIA_IRQ1_N, IRQ_GPIO_H3900_PCMCIA_IRQ1 },
	{ 0, 0 }
}; 

static int h3900_egpio_irq_number(enum ipaq_egpio_type egpio_nr)
{
	struct egpio_irq_info *info = h3900_egpio_irq_info;
	while (info->irq != 0) {
		if (info->egpio_nr == egpio_nr) {
			if (0) printk("%s: egpio_nr=%d irq=%d\n", __FUNCTION__, egpio_nr, info->irq);
			return info->irq;
		}
		info++;
	}

	printk("%s: unhandled egpio_nr=%d\n", __FUNCTION__, egpio_nr); 
	return -EINVAL;
}

static void h3900_set_led (enum led_color color, int duty_time, int cycle_time)
{
	if (duty_time) {
		H3800_ASIC2_LED_TimeBase(color)   = LEDTBS_BLINK | LEDTBS_AUTOSTOP | LEDTBS_ALWAYS | 1;
		H3800_ASIC2_LED_PeriodTime(color) = cycle_time;
		H3800_ASIC2_LED_DutyTime(color)   = duty_time;
	} else {
		H3800_ASIC2_LED_TimeBase(color) = 0;
	}
}

static void h3900_backlight_power (int on)
{
	if (on)
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_FL_PWR_ON;
	else
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_FL_PWR_ON;
}



static struct ipaq_model_ops h3900_model_ops __initdata = {
	.generic_name = "3900",
	.control      = h3900_control_egpio,
	.read         = h3900_read_egpio,
	.pm_callback  = h3900_pm_callback,
	.irq_number   = h3900_egpio_irq_number,
	.set_led      = h3900_set_led,
	.backlight_power = h3900_backlight_power
};

#define MAX_ASIC_ISR_LOOPS    20

/* The order of these is important - see #include <asm/arch/irqs.h> */
static u32 kpio_irq_mask[] = {
	KPIO_KEY_ALL,
	KPIO_SPI_INT,
	KPIO_OWM_INT,
	KPIO_ADC_INT,
	KPIO_UART_0_INT,
	KPIO_UART_1_INT,
	KPIO_TIMER_0_INT,
	KPIO_TIMER_1_INT,
	KPIO_TIMER_2_INT
};

static u32 gpio_irq_mask[] = {
	GPIO2_PEN_IRQ,
	GPIO2_SD_DETECT,
	GPIO2_EAR_IN_N,
	GPIO2_USB_DETECT_N,
	GPIO2_SD_CON_SLT,
};

static void h3900_IRQ_demux( int irq, void *dev_id, struct pt_regs *regs )
{
	int i;

	if (0) printk("%s: interrupt received\n", __FUNCTION__);

	for ( i = 0 ; i < MAX_ASIC_ISR_LOOPS && (GPLR0 & GPIO_H3900_ASIC2_INT) ; i++ ) {
		u32 irq;
		int j;
		
		/* KPIO */
		irq = H3800_ASIC2_KPIINTFLAG;
		if (0) printk("%s: KPIO 0x%08X\n", __FUNCTION__, irq );
		for ( j = 0 ; j < H3800_KPIO_IRQ_COUNT ; j++ )
			if ( irq & kpio_irq_mask[j] )
				do_IRQ( j + H3800_KPIO_IRQ_START, regs );

		/* GPIO2 */
		irq = H3800_ASIC2_GPIINTFLAG;
		if (0) printk("%s: GPIO 0x%08X\n", __FUNCTION__, irq );
		for ( j = 0 ; j < H3800_GPIO_IRQ_COUNT ; j++ )
			if ( irq & gpio_irq_mask[j] )
				do_IRQ( j + H3800_GPIO_IRQ_START, regs );
	}

	if ( i >= MAX_ASIC_ISR_LOOPS )
		printk("%s: interrupt processing overrun\n", __FUNCTION__);
}

static struct irqaction h3900_irq = {
	name:     "h3900_asic",
	handler:  h3900_IRQ_demux,
	flags:    SA_INTERRUPT
};

u32 kpio_int_shadow = 0;

/* mask_ack <- IRQ is first serviced.
       mask <- IRQ is disabled.  
     unmask <- IRQ is enabled 

     The INTCLR registers are poorly documented.  I believe that writing
     a "1" to the register clears the specific interrupt, but the documentation
     indicates writing a "0" clears the interrupt.  In any case, they shouldn't
     be read (that's the INTFLAG register)
*/

static void h3900_mask_ack_kpio_irq( unsigned int irq )
{
	u32 mask = kpio_irq_mask[irq - H3800_KPIO_IRQ_START];
	kpio_int_shadow &= ~mask;
	H3800_ASIC2_KPIINTSTAT = kpio_int_shadow;
	H3800_ASIC2_KPIINTCLR  = mask;
}

static void h3900_mask_kpio_irq( unsigned int irq )
{
	u32 mask = kpio_irq_mask[irq - H3800_KPIO_IRQ_START];
	kpio_int_shadow &= ~mask;
	H3800_ASIC2_KPIINTSTAT = kpio_int_shadow;
}

static void h3900_unmask_kpio_irq( unsigned int irq )
{
	u32 mask = kpio_irq_mask[irq - H3800_KPIO_IRQ_START];
	kpio_int_shadow |= mask;
	H3800_ASIC2_KPIINTSTAT = kpio_int_shadow;
}

static void h3900_mask_ack_gpio_irq( unsigned int irq )
{
	u32 mask = gpio_irq_mask[irq - H3800_GPIO_IRQ_START];
	H3800_ASIC2_GPIINTSTAT &= ~mask;
	H3800_ASIC2_GPIINTCLR   = mask;
}

static void h3900_mask_gpio_irq( unsigned int irq )
{
	u32 mask = gpio_irq_mask[irq - H3800_GPIO_IRQ_START];
	H3800_ASIC2_GPIINTSTAT &= ~mask;
}

static void h3900_unmask_gpio_irq( unsigned int irq )
{
	u32 mask = gpio_irq_mask[irq - H3800_GPIO_IRQ_START];
	H3800_ASIC2_GPIINTSTAT |= mask;
}

extern int g_ipaq_sleeve_opt_int_gpio;

static void __init h3900_init_irq( void )
{
	int i;

	/* Initialize standard IRQs */
	pxa_init_irq();
	
	g_ipaq_sleeve_opt_int_gpio = GPIO_NR_H3900_OPT_INT;

	/* TO DO: Is this appropriate for H1900? */
	/* Disable all IRQs and set up clock */
	H3800_ASIC2_KPIINTSTAT     =  0;     /* Disable all interrupts */
	H3800_ASIC2_GPIINTSTAT     =  0;

	H3800_ASIC2_KPIINTCLR      =  0;     /* Clear all KPIO interrupts */
	H3800_ASIC2_GPIINTCLR      =  0;     /* Clear all GPIO interrupts */

//	H3800_ASIC2_KPIINTCLR      =  0xffff;     /* Clear all KPIO interrupts */
//	H3800_ASIC2_GPIINTCLR      =  0xffff;     /* Clear all GPIO interrupts */

	H3800_ASIC2_CLOCK_Enable       |= ASIC2_CLOCK_EX0;   /* 32 kHZ crystal on */
	H3800_ASIC2_INTR_ClockPrescale |= ASIC2_INTCPS_SET;
	H3800_ASIC2_INTR_ClockPrescale  = ASIC2_INTCPS_CPS(0x0e) | ASIC2_INTCPS_SET;
	H3800_ASIC2_INTR_TimerSet       = 1;

	for ( i = 0 ; i < H3800_KPIO_IRQ_COUNT ; i++ ) {
		int irq = i + H3800_KPIO_IRQ_START;
		irq_desc[irq].valid    = 1;
		irq_desc[irq].probe_ok = 1;
		irq_desc[irq].mask_ack = h3900_mask_ack_kpio_irq;
		irq_desc[irq].mask     = h3900_mask_kpio_irq;
		irq_desc[irq].unmask   = h3900_unmask_kpio_irq;
	}

	for ( i = 0 ; i < H3800_GPIO_IRQ_COUNT ; i++ ) {
		int irq = i + H3800_GPIO_IRQ_START;
		irq_desc[irq].valid    = 1;
		irq_desc[irq].probe_ok = 1;
		irq_desc[irq].mask_ack = h3900_mask_ack_gpio_irq;
		irq_desc[irq].mask     = h3900_mask_gpio_irq;
		irq_desc[irq].unmask   = h3900_unmask_gpio_irq;
	}

	/* Don't start up the ADC IRQ automatically */
	irq_desc[IRQ_H3800_ADC].noautoenable = 1;

        /* note: set_GPIO_IRQ_edge takes a gpio number not a mask on pxa!! */
	set_GPIO_IRQ_edge( GPIO_NR_H3900_ASIC2_INT, GPIO_RISING_EDGE );
	setup_arm_irq( IRQ_GPIO_H3900_ASIC2_INT, &h3900_irq );
}

/*
 * Common map_io initialization
 */
static short ipaq_gpio_modes[] = {
	GPIO1_RTS_MD,
	GPIO18_RDY_MD,
	GPIO15_nCS_1_MD,
	GPIO33_nCS_5_MD,
	GPIO48_nPOE_MD,
	GPIO49_nPWE_MD,
	GPIO50_nPIOR_MD,
	GPIO51_nPIOW_MD,
	GPIO52_nPCE_1_MD,
	GPIO53_nPCE_2_MD,
	GPIO54_pSKTSEL_MD,
	GPIO55_nPREG_MD,
	GPIO56_nPWAIT_MD,
	GPIO57_nIOIS16_MD,
	GPIO78_nCS_2_MD,
	GPIO79_nCS_3_MD,
	GPIO80_nCS_4_MD,
};

static struct map_desc h3900_io_desc[] __initdata = {
 /* virtual            physical           length      domain     r  w  c  b */
  { H3600_BANK_2_VIRT, H3600_BANK_2_PHYS, 0x02800000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 2  CS#2 */
  { H3600_BANK_4_VIRT, H3600_BANK_4_PHYS, 0x00800000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 4  CS#4 */
  { H3600_BANK_5_VIRT, H3600_BANK_5_PHYS, 0x02000000, DOMAIN_IO, 0, 1, 0, 0 }, /* static memory bank 4  CS#5 */
  LAST_DESC
};

static void __init h3900_map_io(void)
{
	int i;

	pxa_map_io();
	iotable_init(h3900_io_desc);

	/* Configure power management stuff. */
	PWER = PWER_GPIO0 | PWER_RTC;
	PFER = PWER_GPIO0 | PWER_RTC;
	PRER = 0;
	PCFR = PCFR_OPDE;
	CKEN = CKEN6_FFUART;

	PGSR0 = GPSRx_SleepValue;
	PGSR1 = GPSRy_SleepValue;
	PGSR2 = GPSRz_SleepValue;

	/* redundant? */
	for (i = 0; i < ARRAY_SIZE(ipaq_gpio_modes); i++) {
		int mode = ipaq_gpio_modes[i];
		if (0)
			printk("ipaq gpio_mode: gpio_nr=%d dir=%d fn=%d\n",
			       mode&GPIO_MD_MASK_NR, mode&GPIO_MD_MASK_DIR, mode&GPIO_MD_MASK_FN);
		set_GPIO_mode(mode);
	}

	/* Set up GPIO direction and alternate function registers */
	GAFR0_L = GAFR0x_InitValue;
	GAFR0_U = GAFR1x_InitValue;
	GAFR1_L = GAFR0y_InitValue;
	GAFR1_U = GAFR1y_InitValue;
	GAFR2_L = GAFR0z_InitValue;
	GAFR2_U = GAFR1z_InitValue;
	
	GPDR0 = GPDRx_InitValue;
	GPDR1 = GPDRy_InitValue;
	GPDR2 = GPDRz_InitValue;
	
	GPCR0 = 0x0fffffff;       /* All outputs are set low by default */

	/* Add wakeup on AC plug/unplug */
	/* TO DO: Detect if we're h1900 and set that up accordingly. */
	PWER  |= PWER_GPIO8;
	PFER  |= PWER_GPIO8;
	PRER  |= PWER_GPIO8;

	/* Select VLIO for ASIC3 */
	MSC2 = (MSC2 & 0x0000ffff) | 0x74a40000; 

	set_GPIO_mode(GPIO33_nCS_5_MD);
	
	/* Set up ASIC #3 */
	H3900_ASIC3_GPIO_A_DIR            = ASIC3GPIO_INIT_DIR;            /* All outputs */
	H3900_ASIC3_GPIO_B_DIR            = ASIC3GPIO_INIT_DIR;            /* All outputs */
	H3900_ASIC3_GPIO_C_DIR            = ASIC3GPIO_INIT_DIR;            /* All outputs */
	H3900_ASIC3_GPIO_D_DIR            = ASIC3GPIO_INIT_DIR;            /* All outputs */

	H3900_ASIC3_GPIO_B_MASK           = ASIC3GPIO_INIT_DIR;            /* No interrupts */
	H3900_ASIC3_GPIO_B_SLEEP_MASK     = ASIC3GPIO_INIT_DIR;
	H3900_ASIC3_GPIO_B_SLEEP_OUT      = ASIC3GPIO_SLEEP_OUT;
	H3900_ASIC3_GPIO_B_BATT_FAULT_OUT = ASIC3GPIO_BATFALT_OUT;
	H3900_ASIC3_GPIO_B_SLEEP_CONF     = 0;                             /* Disable auto sleep */
	
	H3900_ASIC3_GPIO_B_OUT = GPIO3_IR_ON_N | GPIO3_RS232_ON | GPIO3_TEST_POINT_123;

	/* Set up ASIC #2 */
	H3800_ASIC2_GPIOPIOD    = GPIO2_IN_Y1_N | GPIO2_IN_X1_N;
	H3800_ASIC2_GPOBFSTAT   = GPIO2_IN_Y1_N | GPIO2_IN_X1_N;

	H3800_ASIC2_GPIODIR     = GPIO2_PEN_IRQ 
		| GPIO2_SD_DETECT
		| GPIO2_EAR_IN_N 
		| GPIO2_USB_DETECT_N 
		| GPIO2_SD_CON_SLT;
	
	ipaq_model_ops = h3900_model_ops;

	/* Turn off all LEDs */
	ipaq_set_led (GREEN_LED, 0, 0);
	ipaq_set_led (BLUE_LED, 0, 0);
	ipaq_set_led (YELLOW_LED, 0, 0);
}

MACHINE_START(H3900, "HP iPAQ H3900")
	MAINTAINER("HP Labs, Cambridge Research Labs")
	BOOT_MEM(0xa0000000, 0x40000000, io_p2v(0x40000000))
	BOOT_PARAMS(0xa0000100)
	MAPIO(h3900_map_io)
	INITIRQ(h3900_init_irq)
MACHINE_END
