//          Copyright (c) 1999 Symbol Technologies.
//          All Rights Reserved.
//
//	Abstract:
//
//		This is a part of the driver for the CHOICE Microsystems CW10 802.11
//		wireless MAC controller as used in Symbol Technologies systems.  It
//		contains the functions that handle downloading firmware to the adapter.
//		This driver conforms to the NDIS 5.0 interface.
//
//		The overall structure and much of the code is taken from
//		the CW10 driver written by David J. Leach, Jr. of Choice Microsystems,
//		who in turn used the NE2000 NDIS driver from Microsoft as a model.
//
//		IMPORTANT INFORMATION **************************************************
//
//		The most complex part of the Trilogy firmware download is in the handling
//		of the 'Plug Data'.
//
//		There is a section of the firmware on the adapter that is referred to as the
//		Plug Data Area or PDA.  This area contains some number of Plug Data Items or
//		PDIs.  The format of each PDI is:
//										USHORT		length in words (not including this word)
//										USHORT		ID (see list of PDI_xxx defined below)
//										UCHAR[n]	the actual data (n = (length - 1) * 2)
//
//		The binary image to be downloaded contains Plug Data References (PDRs).  The
//		format of each PDR is:
//										ULONG		ID (corresponds to ID in PDI)
//										ULONG		the address within the binary image
//													where the plug data is to be stored
//										ULONG		length in bytes of the actual data
//
//		The logic for processing this Plug Data is:
//		Read the entire PDA from the adapter.
//		Read the PDRs from the binary to be downloaded.
//		Step through each PDR in the image.  If there is a PDI from the PDA that has
//			the same ID, copy the data from the PDI into the binary image at the
//			address specified by the address field in the PDR.
//
//		This process 'plugs' the configuration data (MAC address, MKK call sign,
//			IF frequency, etc. into the proper place in the new image.  This new
//			image is then checksummed (since it's old checksum is no longer valid)
//			and the download proceeds.
//
//		I tried to identify all variables that reference PDI, PDA or PDR by including
//			the appropriate 3 letters in their names.  I hope this helps to keep it
//			straight for you.  This whole issue certainly confused me.  (Still does.)
//
//	Author:
//
//		Robert J. Neilsen
//
//	Revision History:
#include "Spectrum24tApi.h"
#include "Spectrum24t.h"

#if TRILOGY3
#if DBG
static drv_info_t *DbgInfo = &S24tInfo;
STATIC char *S24tRidDebug(__u16 Rid);
#endif

#include "Spectrum24tPrim.h"
#include "Spectrum24tSec.h"

extern void _NdisFreeMemory(void *ptr,ULONG len,ULONG flags,ULONG line);
#define NdisFreeMemory(p,l,f) _NdisFreeMemory((p),(l),(f),__LINE__)

#define DL_MODE_DISABLE		DL_MODE_DIABLE
#define	PRIMARY				0x3E0000l
#define	SECONDARY			0x3F1800l
#define	CIS					0x3EFE00l
#define	SIGNATURE			0x3F17FEl

//	New values for hfa3842

#define PRIMARY42	0x7E0000l
#define SECONDARY42	0x7E1800l
#define CIS42		0x7EFE00l			// ** for now **
#define PDA_START42	0x7F0000l			// ** for now **
#define SIGNATURE42	0x7E17FEl			// 17FE if using Aux Port

#define PRICOMPID	21			// RID FC02 - Primary is ID 21
#define PRICOMPID42	23			// RID FC02 - Primary is ID 21

#define	BUFADDR				0x3000

#define	STATE_IDLE			0
#define	STATE_LOADING		1
#define	STATE_ERASING		2
#define	STATE_PROGRAMMING	3
#define	STATE_VERIFYING		4
#define STATE_READING		5

#define	CIS_SIZE			0x100
#define	MAX_COMPATIBILITY_RECORDS	100
#define	MAX_NUMBER_OF_CRC_RECORDS	20
#define	MAX_NUMBER_OF_REFERENCES	25

//	Plug data area items)

#define PDI_END			0		// End of PDA
#define PDI_PCB			1		// PCB part number (2 bytes)
#define PDI_TRACER		2		// Spare (8 bytes)
#define PDI_SN			3		// Serial Number (12 bytes)
#define PDI_MKKX		4		// MKK measurement (RSSI, ?? - 120 bytes)
#define PDI_RAM			5		// RAM size (2 bytes)
#define PDI_MODLVL		6		// Modem Interface compat (10 bytes)
#define PDI_CTRLVL		7		// Controller IF compat (10 bytes)
#define PDI_NICCP		8		// NIC component version (8 bytes)
#define PDI_IFFREQ		9		// IF Frequency (2 bytes)
#define PDI_OSCFREQ		0x0A	// OSC Freq (MCLK)
#define PDI_EQUIP		0x0B	// Equipment - 1 = EVB, 2 = Radio..

#define PDI_MAC			0x101	// MAC address (6 bytes)
#define PDI_NAME		0x102	// Call Name (8 bytes)
#define PDI_REG			0x103	// Allowed regulatory domain (country) (12 bytes)
#define PDI_CHAN		0x104	// Allowed channel set (2 bytes)
#define PDI_DEFC		0x105	// Default channel (2 bytes)
#define PDI_PRIV		0x106	// Privacy option (2 bytes)
#define PDI_TEMP		0x107	// Temp type (2 bytes)
#define PDI_MKKCALL		0x108	// Call Sign (MKK) (18 bytes) << new >>
#define PDI_MKKRSSI		0x109	// MKK RSSI (2 bytes) << new >>
#define PDI_COUNTRY		0x10A	// Country code (2 bytes) << new >>
#define PDI_COUNTRYNAME	0x10B	// Country name (36 bytes) << new >>
#define PDI_WEPLEN		0x10C	// WEP key length
#define PDI_TXPOWER		0x10D	// Tx Power Calibration (TrilogyII)
#define PDI_RXRSSICAL	0x10E	// Rx RSSI for -60
#define PDI_FLASHCONFIG	0x10F	// match value for firmware address/size
#define PDI_FLASHCONFIG	0x10F	// match value for firmware address/size

#define PDI_MAX			0x200	// max type
#define BIGBLOCK		(16*1024) // download buffer for 2.1 Trilogy (AMD chip)

#define TYPE_EVB		1		// PDI_EQUIP
#define TYPE_ADAPT		2		// PDI_EQUIP

#define C_JAPAN			2

#define PDI_FILENAME	0xFFFFFFFF	// File name PDI

// Error code for Download
#define ERROR_INVALID_MAC_ADDRESS			0x1388		// Invalid MAC Address (XX-XX-XX-FF-00-00)
#define ERROR_INVALID_DOWNLOAD_STATE		0x7D00
#define	ERROR_DOWNLOAD_PLUG_DATA			0x7D01
#define ERROR_INVALID_DOWNLOAD_BLOCK		0x7D02
#define ERROR_INVALID_DOWNLOAD_BLOCK_SIZE	0x7D03
#define	ERROR_DOWNLOAD_PROGRAM_FLASH		0x7D04
#define	ERROR_DOWNLOAD_UPDATE_PDA			0x7D05
#define	ERROR_DOWNLOAD_COMPARE_DATA			0x7D06

// The following constants have been added for Country Change
#define CW_DOWNLOAD		0x0022	// for 2.00 or earlier
#define CW_DOWNLOAD21	0x0023	// for 2.10 or later
#define PDA_START		0x3F0000UL			// PDA storage in Flash
#define PDALEN			512		// max length of production data records, words
#define PDI_SPLMDATA	0x110	// extra bank of init data
#define PDI_FLASHTYPE	0x111	// flash chip type
#define CW_DL_DISABLE	0x0000  // Disables AUX port access & makes the 

// downloaded firmware active
#define CW_DL_EVP		0x0100  // Enables download of firmware image via the 

// AUX port registers directly into volatile
// memory.
#define CW_DL_ENVP		0x0200  // Enables download of firmware  via the AUX

// port registers into a temp download buffer
// in volatile memory.
#define CW_DL_PROG		0x0300  // Writes the download buffer contents into

// the destination area in non-volatile 
// memory.

// The binary image to be downloaded consists of a series of blocks of data.  Each block
// has the following structure.  ulAddress is the actual address in the adapter where
// that block is to be placed.  usLength is the length of the block (in words, not including
// the ulAddress field.  The actual data starts at ucData[0].

#define BLOCKSIZE	(4 * 1024)	// download in chunks this size (bytes)

// Oh, batbarf.  There is >no< guarantee that this structure is aligned to >any< boundary.
// much less a DWORD boundary.  This fact isn't important in the x86 environment, but
// it's critical in a MIPS environment.  Super yucko.
typedef struct _DBLOCK
{
    ULONG   ulAddress1;
    USHORT  usLength1;
    UCHAR   ucData[BLOCKSIZE];
} DBLOCK, *PDBLOCK;

// This is the definition of the PDI structure.  These fields are documented above in the
// initial documentation.

    #if defined(PPDR)

/*
 * Conflicts with an SA1100 register macro definition.
 */
        #undef PPDR
    #endif

typedef struct _PDR
{
    ULONG   id;                 // id field
    ULONG   addr;               // address
    ULONG   len;                // length
} PDR, *PPDR;

// This is the definition of the CRC structure.  There seems to be a provision for multiple
// CRC structure in the binary.  I'm a bit confused on this right now.

typedef struct _CRC
{
    ULONG   ulAddress;          // address
    ULONG   ulLength;           // length
    ULONG   ulType;             // do crc if type = 1
} CRC, *PCRC;

// Function definitions.

BOOLEAN         DnldCompare (PTRILOGY_ADAPTER   pAdapter,
                             ULONG              ulSource,
                             PUCHAR             pucBuffer,
                             USHORT             usLength);

BOOLEAN         DnldCompareBlocks ( PTRILOGY_ADAPTER    pAdapter,
                                    ULONG               ulAddress,
                                    PUCHAR              pucBuffer,
                                    USHORT              usLength);

BOOLEAN         DnldDoCRC (PCRC pCRC);

PUCHAR          DnldFindAddress (ULONG ulAddress);

USHORT          DnldGetID (PUSHORT PDA, USHORT usRecordID);

USHORT          DnldGetPDACRC (PUCHAR pucPDA, USHORT usLength);

USHORT          DnldGetUshort (USHORT usIndex);

NDIS_STATUS     DnldLoadBlocks (PTRILOGY_ADAPTER pAdapter);

NDIS_STATUS     DnldParse (void);

BOOLEAN         DnldPlugData (PTRILOGY_ADAPTER pAdapter);

BOOLEAN         DnldPlugPDI (ULONG ulAddress, 
                             USHORT usLength, 
                             PUCHAR pucData);

VOID            DnldReset (PTRILOGY_ADAPTER pAdapter);

USHORT          DnldScanPastCharacter (USHORT usStartOffset, 
                                       UCHAR ucCharacter);

// Global Static Variables.

NDIS_PHYSICAL_ADDRESS   ndisPhysicalAddress = {-1, -1};
PCRC            pCRCBuffer;                     // points to start of crc records in image
PPDR            pPDReferences;                  // points to start of PD References in image
PDBLOCK         pFirstBlock;                    // points to start of new binary image in memory
PUCHAR          pSourceBuffer = NULL;           // points to start of buffer containing new image
PUCHAR          pucCompareBuffer = NULL;        // points to data read back from adapter
PUSHORT         pusCompatibilityBuffer;         // points to start of compatibility records
UINT            uiMemoryBlockFlags = NDIS_MEMORY_CONTIGUOUS | NDIS_MEMORY_NONCACHED;
UINT            uiSourceBufferLength;           // length of buffer to contain new image
ULONG           ulBlockNumber;                  // number of last block sent from event monitor
ULONG           ulBlockSize;                    // size of blocks sent from event monitor
ULONG           ulBufferAddress;                // address of buffer for download
ULONG           ulDataLength;                   // total length of data sent from event monitor
ULONG           ulTotalBlocks;                  // total number of blocks to be sent from event monitor
USHORT          usCompatibilityRecords[MAX_COMPATIBILITY_RECORDS];
USHORT          usCRCRecords[(sizeof (CRC) * MAX_NUMBER_OF_CRC_RECORDS)];
USHORT          usFilenameIndex;                // index into buffer to start of filename
USHORT          usLengthOfCompatibilityRecords; // length of compatibility records in new image
USHORT          usLengthOfCRCRecords;           // length of CRC records in new image
USHORT          usLengthOfPDReferences;         // length of PDR records in new image
USHORT          usPDReferences[(sizeof(PDR) * MAX_NUMBER_OF_REFERENCES)];
USHORT          usState = STATE_IDLE;           // current state of download
USHORT          usBlockDownLoad;
BOOLEAN         bUpdatePDA = FALSE;
USHORT          SourcePDABuffer[BLOCKSIZE + sizeof(ULONG) + sizeof(USHORT)];
ULONG           BufferAddress = BUFADDR;
DBLOCK          TempBlock;              // store temp block for swap
#if MEMORY_TEST
UCHAR           pucDlBuffer[BIGBLOCK];
#endif
USHORT          usDnloadCommand = CW_DOWNLOAD;
BOOLEAN         bTypePrimary = FALSE;
BOOLEAN         bTypeSecondary = FALSE;
BOOLEAN         bTypeCIS = FALSE;

    #if FIXME

// The following has been added for Country Update support
CHAIN * root;           // pointer to root block
CHAIN * block;          // pointer to current last block
word    dnloadCmd;      // download command id
ULONG   bufaddr;
byte    refs[256];      // buffer to store PD references (FF000000 address)
word    refc;
word    mkkrssif;       // flag has been set
word    rxrssif;        // flag has been set
word    countrycode[3] =    {  2, PDI_COUNTRY, 1};     // IEEE table
word    countryname[21] =   { 19, PDI_COUNTRYNAME, 0}; // text
word    pdmkkrssi[3] =      {  2, PDI_MKKRSSI, 5};     // RSSI
word    channelset[3] =     {  2, PDI_CHAN, 0x3FFF};   // all allowed channels (test)
word    defchannel[3] =     {  2, PDI_DEFC, 3};        // default channel
word    weplen[3] =         {  2, PDI_WEPLEN , 13};
word    flashtype[3] =      {  2, PDI_FLASHTYPE, 0};   // mfg/dev code
word    pdalen;         // flags, buffer to store PDA
word    pda[PDALEN];    // flags, buffer to store PDA
word    endpda[] = { 2, PDI_END, 0};
word    pdi2del;
//	default initial PDA (PDI_PCB must be first)
word initpda[] = {
    5, PDI_PCB, 0x3041, 0x3030, 0x3030, 0x3030, // PCB part number A0000000
    2, PDI_END, 0};                                    // end of pda - 1 word of CRC
word nref, ncrc, ncom;  // number of ref records
byte *  AMDBuffer;          // sort/put buffer
word    trydownload;
byte    cmpbuf[BLOCKSIZE];          // compare buffer
word    uflashType;

    #endif

// CRC16 table for computing firmware checksums.

USHORT usCrc16[256] = {
    0x0000, 0xc0c1, 0xc181, 0x0140, 0xc301, 0x03c0, 0x0280, 0xc241,
    0xc601, 0x06c0, 0x0780, 0xc741, 0x0500, 0xc5c1, 0xc481, 0x0440,
    0xcc01, 0x0cc0, 0x0d80, 0xcd41, 0x0f00, 0xcfc1, 0xce81, 0x0e40, 
    0x0a00, 0xcac1, 0xcb81, 0x0b40, 0xc901, 0x09c0, 0x0880, 0xc841,
    0xd801, 0x18c0, 0x1980, 0xd941, 0x1b00, 0xdbc1, 0xda81, 0x1a40, 
    0x1e00, 0xdec1, 0xdf81, 0x1f40, 0xdd01, 0x1dc0, 0x1c80, 0xdc41,
    0x1400, 0xd4c1, 0xd581, 0x1540, 0xd701, 0x17c0, 0x1680, 0xd641, 
    0xd201, 0x12c0, 0x1380, 0xd341, 0x1100, 0xd1c1, 0xd081, 0x1040,
    0xf001, 0x30c0, 0x3180, 0xf141, 0x3300, 0xf3c1, 0xf281, 0x3240,
    0x3600, 0xf6c1, 0xf781, 0x3740, 0xf501, 0x35c0, 0x3480, 0xf441,
    0x3c00, 0xfcc1, 0xfd81, 0x3d40, 0xff01, 0x3fc0, 0x3e80, 0xfe41,
    0xfa01, 0x3ac0, 0x3b80, 0xfb41, 0x3900, 0xf9c1, 0xf881, 0x3840,
    0x2800, 0xe8c1, 0xe981, 0x2940, 0xeb01, 0x2bc0, 0x2a80, 0xea41,
    0xee01, 0x2ec0, 0x2f80, 0xef41, 0x2d00, 0xedc1, 0xec81, 0x2c40,
    0xe401, 0x24c0, 0x2580, 0xe541, 0x2700, 0xe7c1, 0xe681, 0x2640,
    0x2200, 0xe2c1, 0xe381, 0x2340, 0xe101, 0x21c0, 0x2080, 0xe041,
    0xa001, 0x60c0, 0x6180, 0xa141, 0x6300, 0xa3c1, 0xa281, 0x6240,
    0x6600, 0xa6c1, 0xa781, 0x6740, 0xa501, 0x65c0, 0x6480, 0xa441,
    0x6c00, 0xacc1, 0xad81, 0x6d40, 0xaf01, 0x6fc0, 0x6e80, 0xae41,
    0xaa01, 0x6ac0, 0x6b80, 0xab41, 0x6900, 0xa9c1, 0xa881, 0x6840,
    0x7800, 0xb8c1, 0xb981, 0x7940, 0xbb01, 0x7bc0, 0x7a80, 0xba41,
    0xbe01, 0x7ec0, 0x7f80, 0xbf41, 0x7d00, 0xbdc1, 0xbc81, 0x7c40,
    0xb401, 0x74c0, 0x7580, 0xb541, 0x7700, 0xb7c1, 0xb681, 0x7640,
    0x7200, 0xb2c1, 0xb381, 0x7340, 0xb101, 0x71c0, 0x7080, 0xb041,
    0x5000, 0x90c1, 0x9181, 0x5140, 0x9301, 0x53c0, 0x5280, 0x9241,
    0x9601, 0x56c0, 0x5780, 0x9741, 0x5500, 0x95c1, 0x9481, 0x5440,
    0x9c01, 0x5cc0, 0x5d80, 0x9d41, 0x5f00, 0x9fc1, 0x9e81, 0x5e40,
    0x5a00, 0x9ac1, 0x9b81, 0x5b40, 0x9901, 0x59c0, 0x5880, 0x9841,
    0x8801, 0x48c0, 0x4980, 0x8941, 0x4b00, 0x8bc1, 0x8a81, 0x4a40,
    0x4e00, 0x8ec1, 0x8f81, 0x4f40, 0x8d01, 0x4dc0, 0x4c80, 0x8c41,
    0x4400, 0x84c1, 0x8581, 0x4540, 0x8701, 0x47c0, 0x4680, 0x8641,
    0x8201, 0x42c0, 0x4380, 0x8341, 0x4100, 0x81c1, 0x8081, 0x4040};


// DnldCheckCompatibility
//
//		This function checks for compatibility between primary and secondary modules.
//
//	Parameters:
//		pAdapter - pointer to adapter control structure.
//
//	Expects:
//		New image to be in pSourceBuffer..
//
//	Return Value:
//		NDIS_STATUS_SUCCESS or NDIS_STATUS_FAILURE.

NDIS_STATUS DnldCheckCompatibility (PTRILOGY_ADAPTER pAdapter)
{ // 0
    DBG_FUNC("DnldCheckCompatibility")
    RID_STRUCT  Rid;
    PUSHORT     pUshort;
    USHORT      usEquipmentType;
    USHORT      usPrimaryVersion=0;
    USHORT      usSecondaryVersion;
    USHORT      usTemp;
    PDBLOCK     pBlock;
    BOOLEAN     bPrimCompId = FALSE;
    USHORT      usIndex;
    USHORT      usLength;

    pBlock = pFirstBlock;   // point to first block
    bTypePrimary = FALSE;
    bTypeSecondary = FALSE;
    bTypeCIS = FALSE;
    usDnloadCommand = CW_DOWNLOAD;

    SYMDEBUG(ZDNLD, "+DnldCheckCompatibility");
    for (usIndex = 0; usIndex < usBlockDownLoad; usIndex++)
    {
        SYMDEBUGX(ZDNLD, "Block Address = ", pBlock->ulAddress1);

        if ((pBlock->ulAddress1) == PRIMARY)
        {
            SYMDEBUG(ZDNLD, "Address is Primary Firmware");
            bTypePrimary = TRUE;
            break;
        }

        if ((pBlock->ulAddress1) == SECONDARY)
        {
            SYMDEBUG(ZDNLD, "Address is Secondary Firmware");
            bTypeSecondary = TRUE;
            break;
        }

        if ((pBlock->ulAddress1) == CIS)
        {
            SYMDEBUG(ZDNLD, "Address is CIS Firmware");
            bTypeCIS = TRUE;
            break;
        }

        usLength = pBlock->usLength1;
        pBlock = (PDBLOCK) &pBlock->ucData[usLength];
    }

    //    IF we are doing a primary or secondary,
    //       IF H/W match to PD_EQUIP (1 must match, 2 is ok with 1 or 2)
    //          Else ERROR not compatible
    //       IF we have a secondary and we have FF2000 SECONDRY ID = 0x1F, 
    //          Get the Primary Functions RID FD02 
    //	            AND verify Secondry level is >= Primary level
    //	            Secondary Requires (version 3 or greater for now); 
    //	            Error if not compatible

    //	secondary  currently 5, 1, 0x1F, 1, 1, 1  (1F, 2, 1, 1 for adapter)
    // primary is currently 5, 1, 0x15, 1, 1, 1  (15, 2, 1, 1 for adapter)
    //	as set in common\ver-scnd.asm, common\ver-prim.asm


    // get new dnload command if firmware is version 2.10 or later
    Rid.rid = RID_PRI_IDENTITY;
    Rid.length = 5;
    if (AdapGetRid (pAdapter, &Rid, FALSE) == FALSE)
    {
        SYMDEBUG(ZDNLD, "@FAILED READING RID_PRI_IDENTITY");
        return(NDIS_STATUS_FAILURE);
    }

    usDnloadCommand = CW_DOWNLOAD;
    if (Rid.u.data[0] == PRICOMPID)
    {
        bPrimCompId = TRUE;
        usPrimaryVersion = (Rid.u.data[2] * 10) + Rid.u.data[3];

        if (usPrimaryVersion >= 21)
        {
            SYMDEBUGX(ZDNLD, "@Firmware Version = ", usPrimaryVersion);
            usDnloadCommand = CW_DOWNLOAD21;
        }
    }

    if (bTypePrimary || bTypeSecondary)
    { // 1

        //	Check H/W compatibility
        usTemp = DnldGetID (&pAdapter->pda[0], PDI_EQUIP);
        pUshort = &pAdapter->pda[usTemp];
        usEquipmentType = pUshort[2];

        if (usTemp == MAX_PDA_LEN_WORDS)
        {
            return(NDIS_STATUS_FAILURE);
        }

        if (pusCompatibilityBuffer[0] != 5)
        {
            SYMDEBUGX(ZDNLD, "@ FAIL Unexpected RELOC Record length", pusCompatibilityBuffer[0]);
            return(NDIS_STATUS_FAILURE);
        }

        if (usEquipmentType == TYPE_ADAPT && pusCompatibilityBuffer[3] == TYPE_EVB)
        {
            SYMDEBUG(ZDNLD, "@FAIL Incompatible firmware");
            return(NDIS_STATUS_FAILURE);
        }

        //	Check Firmware Compatibility

        if (bTypeSecondary)
        { // 2

            // Get the address and length of the download buffer.

            Rid.rid = RID_DOWNLOAD_BUFFER;
            Rid.length = 4;

            if (AdapGetRid(pAdapter, &Rid, FALSE) == FALSE)
            {
                SYMDEBUGX(ZDNLD, "@FAIL AdapGetRid, ", Rid.rid);
                return(NDIS_STATUS_FAILURE);
            }

            pAdapter->usDownloadPage = Rid.u.data[0];
            pAdapter->usDownloadOffset = Rid.u.data[1];
            pAdapter->usDownloadLength = Rid.u.data[2];

            if (bPrimCompId)
            {
                usSecondaryVersion = (pusCompatibilityBuffer[4] * 10) + pusCompatibilityBuffer[5];

                if (usPrimaryVersion < usSecondaryVersion)  // IS IT COMPATIBLE with SECONDARY??
                {
                    SYMDEBUGN(ZDNLD, "@FAIL Primary and Secondary mismatch ", usPrimaryVersion, usSecondaryVersion, 0, 0);
                    return(NDIS_STATUS_FAILURE);
                }
            }

        } // 2
        else
        {
            // from DOS utility, only Secondary firmware use multiple blocks programming
            pAdapter->usDownloadLength = 0;
        }

    }

    SYMDEBUG(ZDNLD, "-DnldCheckCompatibility");
    return(NDIS_STATUS_SUCCESS);
}


// DnldCheckCRC
//
//		This function looks for a valid CRC structure in the binary, and
//		if one is found in which the type field is set to 1, a CRC is
//		created and placed in the CRC structure.
//
//	Parameters:
//		None are passed.
//
//	Expects:
//		usCRCRecordsLength to contain the total length of all CRC records in the image.
//		pCRCBuffer to point to the start of these CRC records.
//
//	Return Value:
//		TRUE if CRC is valid, otherwise FALSE

BOOLEAN DnldCheckCRC (void)
{
    DBG_FUNC("DnldCheckCRC")
    USHORT  usIndex;
    USHORT  usNumberOfCRCRecords;

    SYMDEBUG(ZFUNC, "+DnldCheckCRC");

    usNumberOfCRCRecords = usLengthOfCRCRecords / sizeof (CRC);     // how many CRC records

    if (usLengthOfCRCRecords != 0)
    {
        for (usIndex = 0; usIndex < usNumberOfCRCRecords; usIndex++)
        {
            if (pCRCBuffer[usIndex].ulType == 1)
            {

                // A CRC record has been found with ulType set to 1, so
                // call DnldDoCRC to create the CRC.
                if (DnldDoCRC (pCRCBuffer) == FALSE)
                {
                    SYMDEBUG(ZDNLD, "@-DnldCheckCRC, FAIL DnldDoCRC");
                    return(FALSE);
                }
            }
        }
    }

    SYMDEBUG(ZDNLD, "-DnldCheckCRC");
    return(TRUE);

} // 0


// DnldCompare
//
//		This function compares data in the adapter download buffer with
//		the source from which it came.
//
//	Parameters:
//		pAdapter - pointer to adapter control block.
//		ulAddress - address in adapter memory.
//		pucBuffer - address of buffer in driver memory
//		usLength - number of bytes to compare
//
//	Return Value:
//		TRUE if compare is valid, otherwise FALSE.

BOOLEAN DnldCompare (PTRILOGY_ADAPTER   pAdapter,
                     ULONG              ulAddress,
                     PUCHAR             pucBuffer,
                     USHORT             usLength)
{ // 0
    DBG_FUNC("DnldCompare")
    USHORT  usIndex;
    BOOLEAN bResult = TRUE;
    USHORT  usCompareCount = 0;

    DBG_ENTER(DbgInfo);
    DBG_TRACE(DbgInfo,"ulAddress %08x usLength %04x\n",ulAddress,usLength);

    if ((ulAddress == SIGNATURE) && (usLength == 2))
    {
        SYMDEBUG(ZDNLD, "-DnldCompare, Ignore Signature block");
        DBG_LEAVE(DbgInfo);
        return TRUE;
    }

    AdapAuxPortGet (pAdapter, 
                    (PUSHORT) pucCompareBuffer, 
                    ulAddress, 
                    (USHORT) (usLength / sizeof (USHORT)));

    for (usIndex = 0; usIndex < usLength; usIndex++)
    {
        if (pucCompareBuffer[usIndex] != pucBuffer[usIndex])
        {
            bResult = FALSE;
            if (++usCompareCount < 4)
            {
                DBG_TRACE(DbgInfo, "Error : Data mismatch at addr %08x  wrote %02x  read %02x\n",
                          ulAddress, 
                          pucBuffer[usIndex],
                          pucCompareBuffer[usIndex]
                         );
            }
        }

        ulAddress ++;
    }



    if (usCompareCount > 0)
    {
        SYMDEBUGX(ZERROR, "Error : miscompares = ", usCompareCount);
    }

    DBG_LEAVE(DbgInfo);
    return bResult;
}


// DnldCompareBlocks
//
//		This function compares aux port data in the adapter download buffer with
//		the BIG BLOCK source.
//
//	Parameters:
//		pAdapter - pointer to adapter control block.
//		ulAddress - address in adapter memory.
//		pucBuffer - address of buffer in driver memory
//		usLength - number of bytes to compare
//
//	Return Value:
//		TRUE if compare is valid, otherwise FALSE.

BOOLEAN DnldCompareBlocks ( PTRILOGY_ADAPTER    pAdapter,
                            ULONG               ulAddress,
                            PUCHAR              pucBuffer,
                            USHORT              usLength)
{
    DBG_FUNC("DnldCompareBlocks")
    BOOLEAN     bResult = TRUE;
    PUSHORT     pUshort;
    USHORT      usData;
    USHORT      usCompareErrors = 0;

    SYMDEBUG(ZDNLD, "+DnldCompareBlocks");
    if (usLength == 2)
    {
        SYMDEBUG(ZDNLD, "-DnldCompareBlocks, Ignore Signature block");
        return TRUE;
    }

    pUshort = (PUSHORT)pucBuffer;
    usLength /= sizeof(USHORT);

    // set up aux port to read data
    NdisRawWritePortUshort (pAdapter->ulIoPAddr + NIC_AUX_BASE, (USHORT) (BufferAddress >> 7));
    NdisRawWritePortUshort (pAdapter->ulIoPAddr + NIC_AUX_OFFSET, (USHORT) (BufferAddress & 0x7F));

    while (usLength)
    {

        NdisRawReadPortUshort (pAdapter->ulIoPAddr + NIC_AUX_DATA, &usData);
        if (usData != *pUshort)
        {
            bResult = FALSE;
            if (++usCompareErrors < 4)
            {
                SYMDEBUGN(ZDNLD, "Error : Data mismatch @ [address]  [wrote] [read]",
                          ulAddress,
                          *pUshort,
                          usData,
                          0);
            }
        }

        pUshort++;          // jump to next data
        ulAddress += 2;     // jump to next address
        usLength--;         // until all data length compared
    }

    if (usCompareErrors > 0)
    {
        SYMDEBUGX(ZDNLD, "Error : miscompares = ", usCompareErrors);
    }

    return(bResult);
}

// DnldDoCRC
//
//		This calculates a CRC and places it in the specified CRC record.
//
//	Parameters.
//		pCRC - pointer to CRC structure.
//
//	Expects:
//		pFirstBlock to point to the first block in the binary image.
//
//	Return Value:
//		TRUE if CRC is successfully created, otherwise FALSE.  If
//		the data being CRC'd ends with 0x0002, then the CRC is placed
//		in the word immediately following the end of the data.
// 
//	,-----,-----------------------------------,-------,-----,
//	| len |     code image                    | ErrID | CRC |
//	'-----'-----------------------------------'-------'-----'
//	 <----------------- length ---------------------->
//
//	The calculated CRC is appended to the data that was CRC'd.
//
//	NOTE - All blocks in the CRC area must be contiguous!!
//	NOTE - Length is < 64 KB

BOOLEAN DnldDoCRC (PCRC pCRC)
{ // 0

    PDBLOCK pBlock;                 // pointer to current image block
    PUCHAR  pBlockByByte;
    ULONG   ulBlockAddress;
    USHORT  usBlockLength;
    PUCHAR  pByte;
    UCHAR   ucTemp;
    ULONG   ulAddress;
    ULONG   ulNextAddress=0;
    USHORT  usLength;
    USHORT  usNextLengthInWords=0;
    USHORT  usRCRC16 = 0;
    USHORT  usTemp;
    USHORT  usTempData;

    ulAddress = pCRC->ulAddress;        // start of area to be CRC'd
    usLength = (USHORT) pCRC->ulLength; // number of bytes to be included
    pBlock = pFirstBlock;               // point to first block in chain
    pBlockByByte = (PUCHAR) pBlock;
    ulBlockAddress = ((ULONG) pBlockByByte[0]) |
                     (((ULONG) pBlockByByte[1]) << 8) |
                     (((ULONG) pBlockByByte[2]) << 16) |
                     (((ULONG) pBlockByByte[3]) << 24);
    usBlockLength  = ((USHORT) pBlockByByte[4]) |
                     (((USHORT) pBlockByByte[5]) << 8);

    // Search through the blocks of the image until the block is found
    // that contains the starting address.  Stop if we run out of blocks.

    while ((pBlock != NULL) && (usBlockLength != 0))
    { // 1

        // The relevant address must be at or above the starting address of
        // the current block.
        if ((ulAddress >= ulBlockAddress) &&
            (ulAddress < (ulBlockAddress + (ULONG) usBlockLength)))
        { // 2

            // Found the starting block.  Set ulNextAddress to point to the
            // start of the data area of the block plus the offset into the
            // block where the data to be CRC'd starts.

            ulNextAddress = (ULONG) (&pBlock->ucData[0]) + (ulAddress - ulBlockAddress);

            // Set usNextLengthInWords to the count of words left in the
            // current block.  Then break out of the while loop.

            usNextLengthInWords = usBlockLength;
            usNextLengthInWords -= (USHORT) (ulAddress - ulBlockAddress);
            usNextLengthInWords >>= 1;
            break;

        } // 2

        // The starting address is not in the current block.  Chain to the next block.

        pBlock = (PDBLOCK) &pBlock->ucData[usBlockLength];
        pBlockByByte = (PUCHAR) pBlock;
        ulBlockAddress = ((ULONG) pBlockByByte[0]) |
                         (((ULONG) pBlockByByte[1]) << 8) |
                         (((ULONG) pBlockByByte[2]) << 16) |
                         (((ULONG) pBlockByByte[3]) << 24);
        usBlockLength  = ((USHORT) pBlockByByte[4]) |
                         (((USHORT) pBlockByByte[5]) << 8);

    } // 1

    // If the block pointer is NULL or the length field in the block header is zero,
    // we have reached the end of the blocks.  This is an error.
    if ((pBlock == NULL) || (usBlockLength == 0))
    {
        return(FALSE);
    }


    // Set pUshort to point to the first word to be included in the CRC.

    pByte = (PUCHAR) ulNextAddress;

    // Now perform the CRC algorithm until the number of words specified have
    // been included.

    while (usLength != 0)
    { // 3

        // Load the contents of the next word of the image.

        usTempData = ((USHORT) pByte[0]) |
                     (((USHORT) pByte[1]) << 8);

        // Point to the next word in this block.

        pByte += 2;

        // Decrement the number of words left in this block.

        usNextLengthInWords--;

        // If we are at the end of the block we must chain to the next block.
        // However, if the length of the data that is to be included in the
        // CRC drops to 2, we are at the end of the entire area to be CRC'd,
        // so we don't want to try to chain to the next block after all.

        if ((usNextLengthInWords == 0) && (usLength != 2))
        { // 4

            // Add the length of the current block to the starting address.
            // This is the address of the next data to be included, unless
            // we are at the end of the data.

            ulAddress += usBlockLength; // expected next address

            // Now look at the next block (actually at all of the next blocks).
            // The starting address of the next one should be the address that
            // we just calculated to be the next desired address.  If we don't
            // find that, exit FALSE indicating an error.

            while (ulAddress != ulBlockAddress)
            { // 5

                if (pBlock == NULL)
                {

                    return(FALSE);
                }

                // Set pBlock to point to the next data block.
                // Set usNextLengthInWords to the number of words in the next block.

                pBlock = (PDBLOCK) &pBlock->ucData[usBlockLength];
                pBlockByByte = (PUCHAR) pBlock;
                ulBlockAddress = ((ULONG) pBlockByByte[0]) |
                                 (((ULONG) pBlockByByte[1]) << 8) |
                                 (((ULONG) pBlockByByte[2]) << 16) |
                                 (((ULONG) pBlockByByte[3]) << 24);
                usBlockLength  = ((USHORT) pBlockByByte[4]) |
                                 (((USHORT) pBlockByByte[5]) << 8);
                usNextLengthInWords = usBlockLength >> 1;

            } // 5

            pByte = (PUCHAR) &pBlock->ucData[0];

        } // 4

        // Now perform the CRC logic.  This is Greek to me.  It works.

        ucTemp = (UCHAR) usTempData;            // low byte
        ucTemp = (UCHAR) (ucTemp ^ usRCRC16);
        usTemp = usCrc16[ucTemp];
        usRCRC16 >>= 8;
        usRCRC16 ^= usTemp;

        ucTemp = (UCHAR) (usTempData >> 8); // high byte
        ucTemp = (UCHAR) (ucTemp ^ usRCRC16);
        usTemp = usCrc16[ucTemp];
        usRCRC16 >>= 8;
        usRCRC16 ^= usTemp;
        usLength -= 2;

    } // 3

    // If the last word of the image was equal to 0x0002, we are working
    // on the secondary.  (Seems like a weak test to me.)  If this is so,
    // we must add another 2-word block at the end of the current block
    // to contain the CRC.  Now one might ask why we don't just add a word
    // to the last block.  You got me.  I guess it's because it wouldn't
    // work if the last block was of the maximum size allowed in the
    // download buffer.

    pByte -= 2;

    if ((pByte[0] == 0x02) && (pByte[1] == 0x00))
    { // 6

        pBlock = (PDBLOCK) &pByte[2];
        pBlockByByte = (PUCHAR) pBlock;
        pBlockByByte[0]  = (UCHAR) ((pCRC->ulAddress + pCRC->ulLength) & 0xff);
        pBlockByByte[1]  = (UCHAR) (((pCRC->ulAddress + pCRC->ulLength) >> 8) & 0xff);
        pBlockByByte[2]  = (UCHAR) (((pCRC->ulAddress + pCRC->ulLength) >> 16) & 0xff);
        pBlockByByte[3]  = (UCHAR) (((pCRC->ulAddress + pCRC->ulLength) >> 24) & 0xff);
        pBlockByByte[4] = 2;
        pBlockByByte[5] = 0;
        pByte = (PUCHAR) &pBlock->ucData[0];
        pByte[0] = (UCHAR) (usRCRC16 & 0xff);
        pByte[1] = (UCHAR) ((usRCRC16 >> 8) & 0xff);
        pByte[2] = 0;
        pByte[3] = 0;
        pByte[4] = 0;
        pByte[5] = 0;

        // need to increase block counter
        usBlockDownLoad++;

    } // 6


    return(TRUE);

} // 0


//	DnldUpdatePDA
//
//		Routine to update the new block of PDA for download
//
//	Arguments:
//		pAdapter	- pointer to adapter control block.
//
//	Return Value:
//
//		NDIS_STATUS_SUCCESS
//		NDIS_STATUS_FAILURE

NDIS_STATUS DnldUpdatePDA (IN PTRILOGY_ADAPTER  pAdapter)
{ // 0

    USHORT  usLength;
    USHORT  usId;
    USHORT  usPDALengthInBytes;
    USHORT  usPDALengthInWords;
    USHORT  usCRC;
    USHORT  usIndex;
    PUCHAR  pUchar;


    usLength = pAdapter->pda[0];        // first length

    usId = pAdapter->pda[1];            // first ID

    // The first entry should be PDI_PCB (serial number) with a length of 5 words.

    if ((usId != PDI_PCB) || (usLength != 5))
    {

        return(ERROR_DOWNLOAD_UPDATE_PDA);
    }

    // Search the PDA for the last entry, which must be PDI_END, length 3;
    // DnldGetID returns the offset of the found entry relative to the start
    // of the area.  Adding 2 to this gives us the length of the PDA in words.

    usPDALengthInWords = DnldGetID (&pAdapter->pda[0], PDI_END) + 2;    // find the length

    usPDALengthInBytes = usPDALengthInWords << 1;

    // DnldGetPCACRC calculates the CRC of the PDA, starting at offset 0, for
    // usPDALengthInBytes bytes.

    usCRC = DnldGetPDACRC ((PUCHAR) &pAdapter->pda, usPDALengthInBytes);

    //Update new PDA Checksum
    pAdapter->pda[usPDALengthInWords] = usCRC; 

    pFirstBlock = (PDBLOCK) &SourcePDABuffer[0];
    pUchar = (PUCHAR) pFirstBlock;

    pUchar[0] = (UCHAR) (PDA_START_ADDR & 0xff);
    pUchar[1] = (UCHAR) ((PDA_START_ADDR >> 8) & 0xff);
    pUchar[2] = (UCHAR) ((PDA_START_ADDR >> 16) & 0xff);
    pUchar[3] = (UCHAR) ((PDA_START_ADDR >> 24) & 0xff);
    pUchar[4] = (UCHAR) ((MAX_PDA_LEN_WORDS << 1) & 0xff);
    pUchar[5] = (UCHAR) ((MAX_PDA_LEN_WORDS >> 7) & 0xff);
    pUchar = (PUCHAR)(pAdapter->pda);
    for (usIndex = 0; usIndex < (MAX_PDA_LEN_WORDS << 1); usIndex++)
    {
        pFirstBlock->ucData[usIndex] = pUchar[usIndex];
    }

    // Mark end of PDA Block
    usBlockDownLoad = 0;



    return NDIS_STATUS_SUCCESS;
}


//	DnldDownload
//
//		Processes downloading firmware image.
//
//	Arguments:
//		pAdapter	- pointer to adapter control block.
//		pBlock		- pointer to block of data to be downloaded.
//		ulFunction	- function to be performed (DOWNLOAD_START,
//												DOWNLOAD_DATA,
//												DOWNLOAD_END,
//												DOWNLOAD_ABORT)
//
//	Return Value:
//
//		NDIS_STATUS_SUCCESS
//		NDIS_STATUS_FAILURE

NDIS_STATUS DnldDownload (IN PTRILOGY_ADAPTER   pAdapter,
                          IN PDOWNLOAD_BLOCK    pBlock)
{ // 0
    DBG_FUNC("DnldDownload")

    NDIS_STATUS     nsStatus;
    ULONG           ulIndex;

    DBG_ENTER(DbgInfo);
    DBG_TRACE(DbgInfo,"usID = %04x\n", pBlock->usID);

    switch (pBlock->usID)
    {
        case FW_UPDATE_START:
            switch (usState)
            { // 2

                // It is legal to enter this state from either STATE_IDLE
                // or from STATE_LOADING (in case a download was aborted).
                // Reset all of the local variables which will be used during
                // the download.
                
                case STATE_IDLE:
                case STATE_LOADING:
                    SYMDEBUG(ZDNLD, "case STATE_IDLE");
                    ulBlockNumber = 0;
                    ulDataLength = 0;
                    ulBlockSize = pBlock->ulLength;
                    usState = STATE_LOADING;
                    ulTotalBlocks = pBlock->ulBlock;
                    nsStatus = NDIS_STATUS_SUCCESS;

                    uiSourceBufferLength = ulTotalBlocks * ulBlockSize;

#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
                    nsStatus = NdisAllocateMemoryWithTag (&pSourceBuffer,
                                                          uiSourceBufferLength + 8,
                                                          'umyS');
                    nsStatus = NdisAllocateMemoryWithTag (&pucCompareBuffer,
                                                          CHAIN_BLOCK_SIZE,
                                                          'umyS');
#else
                    nsStatus = NdisAllocateMemory (&pSourceBuffer,
                                                   uiSourceBufferLength + 8,
                                                   uiMemoryBlockFlags,
                                                   ndisPhysicalAddress);
                    nsStatus = NdisAllocateMemory (&pucCompareBuffer,
                                                   CHAIN_BLOCK_SIZE,
                                                   uiMemoryBlockFlags,
                                                   ndisPhysicalAddress);
#endif

                    if (nsStatus != NDIS_STATUS_SUCCESS)
                    {
                        SYMDEBUG(ZERROR, "@FAIL Allocate memory");
                        DBG_LEAVE(DbgInfo);
                        usState = STATE_IDLE;
                    }

                    SYMDEBUGX(ZDNLD, "-DnldDownload, status = ", nsStatus);
                    DBG_LEAVE(DbgInfo);
                    return(nsStatus);

                default:
                    SYMDEBUGX(ZDNLD, "@FAIL Invalid Update state =", usState);
                    DBG_LEAVE(DbgInfo);
                    return(ERROR_INVALID_DOWNLOAD_STATE);

            }

            // The firmware binary image is being sent from the event monitor.
            // Make sure that all of the blocks are received, and in the correct
            // order.
        case FW_UPDATE_DATA:

            if (usState != STATE_LOADING)
            {
                SYMDEBUGX(ZDNLD, "@FAIL Invalid Update state =", usState);
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_INVALID_DOWNLOAD_STATE);
            }

            ulBlockNumber++;            // Increment the block number received
            if ((ulBlockSize != pBlock->ulLength) && (ulBlockNumber != ulTotalBlocks))
            {
                SYMDEBUGX(ZDNLD, "@FAIL Invalid Blocksize", ulBlockSize);
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_INVALID_DOWNLOAD_BLOCK_SIZE);
            }

            if (pBlock->ulBlock != ulBlockNumber)
            {
                SYMDEBUGX(ZDNLD, "@FAIL Invalid Block Number", ulBlockSize);
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_INVALID_DOWNLOAD_BLOCK);
            }

            for (ulIndex = 0; ulIndex < pBlock->ulLength; ulIndex++)
                pSourceBuffer[ulDataLength++] = pBlock->aucData[ulIndex];

            DBG_LEAVE(DbgInfo);
            return(NDIS_STATUS_SUCCESS);


            // The firmware or CIS image has been received.
            // Now perform the update.
        case FW_UPDATE_END:
            // If the state is not STATE_LOADING, we shouldn't be here.
            if (usState != STATE_LOADING)
            {
                SYMDEBUGX(ZDNLD, "@FAIL Invalid Update state =", usState);
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_INVALID_DOWNLOAD_STATE);
            }

            // If all of the blocks have not been received, don't continue.
            if (ulBlockNumber != ulTotalBlocks)
            {
                SYMDEBUGX(ZDNLD, "@FAIL Invalid Block Number", ulBlockSize);
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_INVALID_DOWNLOAD_BLOCK);
            }

            // Extract the header information from the first block, checksum the entire
            // image.  Locate the PD References, the CRC records, and the Compatibility
            // information.
            nsStatus = DnldParse();
            if (nsStatus != NDIS_STATUS_SUCCESS)
            {
                SYMDEBUG(ZDNLD, "@FAIL DnldParse()");
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(nsStatus);
            }
            // Process the plug data records.
            if (DnldPlugData (pAdapter) == FALSE)
            {
                SYMDEBUG(ZDNLD, "@FAIL DnldPlugData()");
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(ERROR_DOWNLOAD_PLUG_DATA);
            }

            // Program the SRAM with new Firmware.
            nsStatus = DnldLoadBlocks(pAdapter);

            if (nsStatus != NDIS_STATUS_SUCCESS)
            {
                SYMDEBUG(ZDNLD, "@FAIL DnldLoadBlocks()");
                DnldReset (pAdapter);
                DBG_LEAVE(DbgInfo);
                return(nsStatus);
            }

            DnldReset (pAdapter);
            DBG_LEAVE(DbgInfo);
            return NDIS_STATUS_SUCCESS;

    } // switch

    DBG_LEAVE(DbgInfo);
    return(NDIS_STATUS_FAILURE);
} // 0

// DnldFindAddress
//
//		This function searches the blocks of the new binary image for the one
//		that contains the address that is passed to it as an input parameter.
//
//	Parameters:
//		ulAddress - address to be searched for.
//
//	Expects:
//		pFirstBlock to point to the first of the blocks that contain the new image.
//
//	Return Value:
//		If the address is found in the image, returns a pointer to its location within
//		the image. If it is not found, returns NULL.

PUCHAR DnldFindAddress (ULONG ulAddress)
{ // 0
    DBG_FUNC("DnldFindAddress")
    PDBLOCK pNextBlock;
    PUCHAR  pNextBlockByByte;
    ULONG   ulNextBlockAddress;
    USHORT  usNextBlockLength;
    ULONG   ulTemp;
    USHORT  ulIndex;

    DBG_ENTER(DbgInfo);

    // Start with the first block in the chain.


    pNextBlock = pFirstBlock;
    pNextBlockByByte = (PUCHAR) pNextBlock;
    ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                         (((ULONG) pNextBlockByByte[1]) << 8) |
                         (((ULONG) pNextBlockByByte[2]) << 16) |
                         (((ULONG) pNextBlockByByte[3]) << 24);
    usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                         (((USHORT) pNextBlockByByte[5]) << 8);
    ulIndex = 0;

    while (TRUE)
    { // 1

        // If we run out of blocks, exit.
        if (ulIndex > usBlockDownLoad)
        {
            DBG_LEAVE(DbgInfo);
            return(NULL);
        }

        // If we run out of blocks, exit.
        if (pNextBlock == NULL)
        {
            DBG_LEAVE(DbgInfo);
            return(NULL);
        }

        //If the address of the next block is zero, exit.
        if (ulNextBlockAddress == 0)
        {
            DBG_LEAVE(DbgInfo);
            return(NULL);
        }

        //If the length of the next block is zero, exit.
        if (usNextBlockLength == 0)
        {
            DBG_LEAVE(DbgInfo);
            return(NULL);
        }

        // Set ulTemp to the last address that is included in this block.

        ulTemp = ulNextBlockAddress + (ULONG) usNextBlockLength;

        // See if the desired address is in the current block.

        if ((ulAddress >= ulNextBlockAddress) && (ulAddress < ulTemp))
        { // 2

            // Now calculate the address (in our memory) that contains the
            // data which will be downloaded at the specified address in the
            // adapter.  First calculate the offset within the block where
            // this byte resides, then add in the address of the first byte
            // in the block.

            DBG_LEAVE(DbgInfo);
            return(PUCHAR) ((ulAddress - ulNextBlockAddress + (ULONG) (&pNextBlock->ucData[0])));

        } // 2

        pNextBlock = (PDBLOCK) &pNextBlock->ucData[usNextBlockLength];
        pNextBlockByByte = (PUCHAR) pNextBlock;
        ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                             (((ULONG) pNextBlockByByte[1]) << 8) |
                             (((ULONG) pNextBlockByByte[2]) << 16) |
                             (((ULONG) pNextBlockByByte[3]) << 24);
        usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                             (((USHORT) pNextBlockByByte[5]) << 8);
        ulIndex++;

    } // 1

    DBG_LEAVE(DbgInfo);
    return(NULL);
} // 0


// DnldGetID
//
//		This function scans the PDA, looking for the record with the specified ID.
//
//	Parameters:
//		PDA			- the start of the PDA in memory.
//		usRecordID	- the ID of the desired record.
//
//	Return Value:
//		The offset within the PDA of the desired record if it is found, MAX_PDA_LEN_WORDS
//		if it is not.

USHORT  DnldGetID (PUSHORT PDA, USHORT usRecordID)
{
    DBG_FUNC("DnldGetID")
    USHORT usIndex = 0;

    DBG_ENTER(DbgInfo);
    DBG_TRACE(DbgInfo,"usRecordID %04x\n",usRecordID);

    while (TRUE)
    {
        DBG_TRACE(DbgInfo,"PDA[%u] == %04x\n",(uint)usIndex,PDA[usIndex]);

        // If the next word of the PDA, which should be the length field
        // of the next PDA record, is zero, return an error indication.
        if (PDA[usIndex] == 0)
        {
            SYMDEBUG(ZDNLD, "-DnldGetID - Invalid Length");
            DBG_LEAVE(DbgInfo);
            return(MAX_PDA_LEN_WORDS);
        }

        // If the following word of the PDA, which should be the ID field
        // of the next PDA record, is the ID we are looking for, return
        // the offset into the PDA of this record.
        if (PDA[usIndex + 1] == usRecordID)
        {
            SYMDEBUG(ZDNLD, "-DnldGetID - Found PDI");
            DBG_LEAVE(DbgInfo);
            return(usIndex);
        }

        // If the ID is PDI_END, we are at the end of the PDA, so return
        // an error indication.
        if (PDA[usIndex + 1] == PDI_END)
        {
            SYMDEBUG(ZDNLD, "-DnldGetID - PDI END");
            DBG_LEAVE(DbgInfo);
            return(MAX_PDA_LEN_WORDS);
        }

        // Increment to the next PDA record.
        usIndex += PDA[usIndex];
        usIndex++;

        // If we have gone past the maximum length of the PDA, return an
        // error indication.
        if (usIndex >= MAX_PDA_LEN_WORDS)
        {
            SYMDEBUG(ZDNLD, "-DnldGetID - Invalid PDA");
            DBG_LEAVE(DbgInfo);
            return(MAX_PDA_LEN_WORDS);
        }
    }
}

// DnldGetPDACRC
//
//		This function calculates the CRC of the PDA.
//
//	Parameters:
//		pucPDA - pointer to start of PDA.
//		usLength - number of bytes to include.
//
//	Return Value:
//		returns the calculated CRC.

USHORT DnldGetPDACRC (PUCHAR pucPDA, USHORT usLength)
{ // 0
    DBG_FUNC("DnldGetPDACRC")
    USHORT  i;
    USHORT  usCarry = 0;
    USHORT  usCRC = 0;
    USHORT  usIBit;
    USHORT  usTemp;

    DBG_ENTER(DbgInfo);
    DBG_TRACE(DbgInfo,"usLength %u\n",(uint)usLength);

    if (usLength == 0)
    {
        DBG_LEAVE(DbgInfo);
        return(0);
    }

    for (i = 0; i < usLength; i++)
    { // 1

        usTemp = (usCRC ^ pucPDA[i]) & 0x00ff;
        usCRC >>= 8;

        for (usIBit = 0; usIBit < 8; usIBit++)
        { // 2

            // Shift all bits right.

            usCarry = usTemp & 0x0001;  // see if carry due

            usTemp >>= 1;

            if (usCarry != 0)
                usTemp ^= 0xA001;       // XOR with polynomial

        } // 2

        usCRC ^= usTemp;

    } // 1

    DBG_LEAVE(DbgInfo);
    return(usCRC);

} // 0


// USHORT	DnldGetUshort
//
//		This function converts 2 ascii digits into a binary number.
//
//	Parameters:
//		usIndex - offset into pSourceBuffer of first digit.
//
//	Return Value:
//		returns the binary value.

USHORT  DnldGetUshort (USHORT usIndex)
{ // 0

    USHORT  usTemp = 0;


    while (TRUE)
    { // 1

        if ((pSourceBuffer[usIndex] < '0') || (pSourceBuffer[usIndex] > '9'))
        {

            return(usTemp);
        }

        else
            usTemp = (usTemp * 10) + (pSourceBuffer[usIndex++] - (USHORT) '0');

    } // 1

    return(usTemp);
} // 0


// DnldParse
//
//		This extracts the information from the header of the binary image,
//		verifies the checksum of the image.
//
//	Parameters:
//		none.
//
//	Expects:
//		pSourceBuffer points to the data sent from the event monitor.
//
//	Return Value:
//		NDIS_STATUS_SUCCESS if everything works, otherwise NDIS_STATUS_FAILURE.

NDIS_STATUS DnldParse (void)
{ // 0
    DBG_FUNC("DnldParse")

    PDBLOCK     pNextBlock;
    PUCHAR      pNextBlockByByte;
    USHORT      usNextBlockLength;
    PUCHAR      pPastEnd;
    PUCHAR      pByte;
    USHORT      usChecksum;
    USHORT      usIndex;
    USHORT      usNextLength;
    USHORT      usNumberOfBlocks;
    USHORT      usPointer;
    USHORT      usTemp;
    USHORT      usTypeIndex;

    DBG_ENTER(DbgInfo);

    // Scan past the first blank, which puts us at the filename string.
    usFilenameIndex = DnldScanPastCharacter (0, ' ');
    if (usFilenameIndex == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    // Scan past the next blank, which puts us at the type string.
    usTypeIndex = DnldScanPastCharacter (usFilenameIndex, ' ');
    if (usTypeIndex == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    // Scan past the next blank, then get the number of blocks in the binary.
    usTemp = DnldScanPastCharacter (usTypeIndex, ' ');
    if (usTemp == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }
    usNumberOfBlocks = DnldGetUshort (usTemp);

    // Scan past the next blank, then get the number of bytes of PD References.
    usTemp = DnldScanPastCharacter (usTemp, ' ');
    if (usTemp == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }
    usLengthOfPDReferences = DnldGetUshort (usTemp);

    // Scan past the next blank, then get the number of bytes of CRC records.
    usTemp = DnldScanPastCharacter (usTemp, ' ');
    if (usTemp == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }
    usLengthOfCRCRecords = DnldGetUshort (usTemp);

    // Scan past the next blank, then get the number of bytes of compatibility records.
    usTemp = DnldScanPastCharacter (usTemp, ' ');
    if (usTemp == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }
    usLengthOfCompatibilityRecords = DnldGetUshort (usTemp);

    // Scan past the next blank, which puts us at the start of the binary block..
    usTemp = DnldScanPastCharacter (usTemp, 0x1a);
    if (usTemp == 0xFFFF)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    // The first block of the binary starts in the next byte.

    usBlockDownLoad = usNumberOfBlocks;

    pFirstBlock = (PDBLOCK) &pSourceBuffer[usTemp];

    pNextBlock = pFirstBlock;
    pNextBlockByByte = (PUCHAR) pNextBlock;
    usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                         (((USHORT) pNextBlockByByte[5]) << 8);

    DBG_TRACE(DbgInfo,"pFirstBlock addr %08x len %04x\n",pFirstBlock->ulAddress1,usNextBlockLength);

    usPointer = 0;

    if ((usLengthOfPDReferences / sizeof (PDR))  > MAX_NUMBER_OF_REFERENCES)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    if ((usLengthOfCRCRecords / sizeof (CRC))  > MAX_NUMBER_OF_CRC_RECORDS)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    if (usLengthOfCompatibilityRecords > MAX_COMPATIBILITY_RECORDS)
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    // Now process each of the blocks in the image.  Each starts with a ULONG
    // that contains the starting address (where the data will go in adapter
    // memory), a USHORT indicating how many bytes of data are in the block, and
    // the actual data.

    // Point pByte to the start of the blocks.  Initialize the checksum variable to zero.


    pByte = (PUCHAR) pFirstBlock;
    usChecksum = 0;

    //Original
    while (usNumberOfBlocks--)
    { // 1
        usNextLength = usNextBlockLength;                       // how many bytes there are
        pNextBlock = (PDBLOCK) &pNextBlock->ucData[usNextLength];   // point to the start of the next block
        pNextBlockByByte = (PUCHAR) pNextBlock;
        usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                             (((USHORT) pNextBlockByByte[5]) << 8);

        DBG_TRACE(DbgInfo,"pNextBlock addr %08x len %04x\n",pNextBlock->ulAddress1,usNextBlockLength);

        // If the length is odd, exit.  This cannot be.
        if ((usNextLength & 1) != 0)
        {
            DBG_LEAVE(DbgInfo);
            return(NDIS_STATUS_FAILURE);
        }

        // Checksum the entire block including the header.
        for (usIndex = 0; usIndex < ((usNextLength + 6) / 2); usIndex++)
        {
            usChecksum += ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usPointer += 2;
        }
    } // 1

    pPastEnd = (PUCHAR) &pByte[usPointer];

    // If there is a block of PD References, include that in the checksum.
    // Also move the PD References to their own buffer.

    if (usLengthOfPDReferences != 0)
    { // 2

        // Save the pointer to the references for later use.
        pPDReferences = (PPDR) &usPDReferences[0];

        //Do Checksum
        for (usIndex = 0; usIndex < (usLengthOfPDReferences / 2); usIndex++)
        { // 2a
            usPDReferences[usIndex] = ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usChecksum += ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usPointer += 2;
        } // 2a

    } // 2

    // If there is a block of CRC records, include that in the checksum.
    // Also move the CRC records to their own buffer.

    if (usLengthOfCRCRecords != 0)
    { // 3

        // Save the pointer to the CRC records for later use.
        pCRCBuffer = (PCRC) &usCRCRecords[0];

        for (usIndex = 0; usIndex < (usLengthOfCRCRecords / 2); usIndex++)
        { // 3a

            usCRCRecords[usIndex] = ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usChecksum += ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usPointer += 2;

        } // 3a
    } // 3

    // If there is a block of compatibility records, include that in the checksum.
    // Also move the compatibility records to their own buffer.

    if (usLengthOfCompatibilityRecords != 0)
    { // 4

        // Save the pointer to the compatibility records for later use.

        pusCompatibilityBuffer = &usCompatibilityRecords[0];

        for (usIndex = 0; usIndex < (usLengthOfCompatibilityRecords / 2); usIndex++)
        { // 4a

            usCompatibilityRecords[usIndex] = ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usChecksum += ((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8);
            usPointer += 2;
        } // 4a
    } // 4

    // Immediately following the compatibility records is the checksum.
    // Exit if the checksum is invalid.
    if (usChecksum != (((USHORT) pByte[usPointer]) | (((USHORT) pByte[usPointer + 1]) << 8)))
    {
        DBG_LEAVE(DbgInfo);
        return(NDIS_STATUS_FAILURE);
    }

    // Make sure that the last image block is followed by a NULL pointer.
    pPastEnd[0] = 0;
    pPastEnd[1] = 0;
    pPastEnd[2] = 0;
    pPastEnd[3] = 0;
    pNextBlockByByte[4] = 0;
    pNextBlockByByte[5] = 0;

    DBG_LEAVE(DbgInfo);
    return(NDIS_STATUS_SUCCESS);

} // 0

// USHORT	DnldPlugData
//
//		This function process the Plug Data records and plugs the new image.
//
//	Parameters:
//		pAdapter - pointer to adapter control block.
//
//	Expects:
//		usLengthOfPDReferences contains total length of PDReferences.
//
//	Return Value:
//		returns TRUE is success, otherwise FALSE.

BOOLEAN DnldPlugData (PTRILOGY_ADAPTER pAdapter)
{ // 0
    DBG_FUNC("DnldPlugData")
    PUCHAR  pucP;
    UCHAR   ucFilename[14];
    USHORT  usIndex;
    USHORT  usIndex2;
    USHORT  usNumberOfReferences;
    USHORT  usPDALength;
    USHORT  usReferenceIndex;
    USHORT  usReferenceLength;

    DBG_ENTER(DbgInfo);

    // If there are no references, there is nothing to do.  (There is probably
    // also an error, but we won't get into that just now.
    if (usLengthOfPDReferences == 0)
    {
        SYMDEBUG(ZDNLD, "-DnldPlugData successfully, usLengthOfPDReferences == 0");
        DBG_LEAVE(DbgInfo);
        return(TRUE);
    }

    // Calculate the number of references in the list.
    usNumberOfReferences = usLengthOfPDReferences / sizeof (PDR);

    // Start with the first reference.
    usReferenceIndex = 0;

    // For each reference item, see if there is a value in the plug
    // data buffer.  If so, plug it.  The modified hex image with the
    // configuration data plugged into it will later be written to flash.

    while (TRUE)
    { // 1

        DBG_TRACE(DbgInfo,"pPDReferences[usReferenceIndex].id %04x\n",pPDReferences[usReferenceIndex].id);

        // If the ID of the next reference is 0, exit;

        if (pPDReferences[usReferenceIndex].id == 0)
        {
            SYMDEBUG(ZDNLD, "-DnldPlugData successfully, pPDReferences[usReferenceIndex].id == 0");
            DBG_LEAVE(DbgInfo);
            return(TRUE);
        }

        // If we have processed all of the PD References, exit.
        if (usReferenceIndex >= usNumberOfReferences)
        {
            SYMDEBUG(ZDNLD, "-DnldPlugData successfully, process all PD References");
            DBG_LEAVE(DbgInfo);
            return(TRUE);
        }

        // If the ID is for the filename, process it with unique logic.
        // All other IDs are processed with common logic.

        if (pPDReferences[usReferenceIndex].id == PDI_FILENAME)
        {
            // Set pucP to the address of the file name in the source buffer.
            // Start Index2 == 1, indicating that the actual name starts with
            // the first character.

            pucP = (PUCHAR) &pSourceBuffer[usFilenameIndex];
            usIndex2 = 0;
            // If the first character of the filename field is a :, increment
            // past that and the next character.  This is supposed to bypass
            // a drive letter at the start of the file name.

            if (pucP[1] == ':')
                usIndex2 = 2;

            // Now search ahead for a maximum of 20 characters, looking for a
            // backslash.  If one is found, set usIndex2 to indicate the character
            // that follows the backslash.  Continue to do this until no backslash
            // is found or a comma is founc, indicating the end of the file name.

            while (TRUE)
            { // 3

                for (usIndex = 0; usIndex < 20; usIndex++)
                { // 3a

                    if (pucP[usIndex2] == '\\')
                    { // 3b

                        usIndex2++;
                        break;

                    } // 3b

                    if (pucP[usIndex2] == ',')
                        break;

                    usIndex2++;

                } // 3a

                // If no backslash was found, reset usIndex2 back to where it was
                // and quit looking.

                if ((usIndex == 20) || (pucP[usIndex2] == ','))
                { // 3c

                    usIndex2 = usIndex2 - usIndex;
                    break;

                } // 3c
            } // 3

            for (usIndex = 0; usIndex < 14; usIndex++)
            { // 3d

                if (pucP[usIndex2 + usIndex] == ',')
                    break;

                ucFilename[usIndex] = pucP[usIndex2 + usIndex];

            } // 3d

            for (; usIndex < 14; usIndex++)
                ucFilename[usIndex] = 0;

            // Now call DnldPlugPDI to plug the filename into the image.

            DBG_TRACE(DbgInfo,"PDI_FILENAME %s\n",&ucFilename[0]);
            DnldPlugPDI (pPDReferences[usReferenceIndex].addr,
                         14,
                         &ucFilename[0]);

        } // 2

        else
        { // 4

            // This logic works for all except the filename PD Item.
            // If the ID in the next reference is zero, skip this processing.

            if (pPDReferences[usReferenceIndex].id != 0)
            { // 5

                // Call DnldGetID to find the PDI in the PDA that this PDR applies to.

                usIndex2 = DnldGetID (&pAdapter->pda[0], (USHORT) pPDReferences[usReferenceIndex].id);

                // If a match is found, process it.
                if (usIndex2 != MAX_PDA_LEN_WORDS)
                { // 6


                    // Set usPDALength to the length of the PDI as shown in the PDA.
                    // Set usReferenceLength to the length of the PDI as indicated in
                    // the PD Reference.

                    usPDALength = (pAdapter->pda[usIndex2] - 1) << 1;

                    usReferenceLength = (USHORT) pPDReferences[usReferenceIndex].len;

                    // If the 2 lengths are the same, call DnldPlugPDI to do the
                    // actual plugging.
                    if (usPDALength == usReferenceLength)
                        DnldPlugPDI (pPDReferences[usReferenceIndex].addr,
                                     usReferenceLength,
                                     (PUCHAR) &pAdapter->pda[usIndex2 + 2]);
                } // 6


                // No match for the ID was found in the PDA.  If the reference is
                // either PDI_SN or PDI_MAC, return an error.  These IDs must be found
                // for the firmware to function.

                else
                { // 7

                    if (pPDReferences[usReferenceIndex].id == PDI_MAC)
                    {
                        SYMDEBUG(ZERROR, "-DnldPlugData error, No MAC in PDA");
                        DBG_LEAVE(DbgInfo);
                        return(FALSE);
                    }

                } // 7
            } // 5
        } // 4

        // Increment to the next PD Reference.

        usReferenceIndex++;
    } // 1

    DBG_LEAVE(DbgInfo);
    return(TRUE);
} // 0


// USHORT	DnldPlugPDI
//
//		This function plugs one piece of data into the image.
//
//	Parameters:
//		ulAddress - address of data item relative to start of adapter memory.
//		usLength - number of bytes to plug.
//		pucData - pointer to the data to be plugged.
//
//	Expects:
//		Image to be in pSourceBuffer.
//
//	Return Value:
//		returns TRUE is success, otherwise FALSE.

BOOLEAN DnldPlugPDI (ULONG ulAddress, USHORT usLength, PUCHAR pucData)
{ // 0
    DBG_FUNC("DnldPlugPDI")
    PUCHAR  pucPointer;
    USHORT  usIndex;

    DBG_ENTER(DbgInfo);

    // Process the plug data one byte at a time.  This must be done in
    // case the region to be plugged overlaps 2 blocks in the image.
    for (usIndex = 0; usIndex < usLength; usIndex++)
    { // 1

        // Locate the offset in the image buffer that corresponds to the
        // desired address in adapter memory.
        pucPointer = DnldFindAddress (ulAddress);

        // If that address cannot be found, exit.
        if (pucPointer == NULL)
        {
            SYMDEBUG(ZDNLD, "@FAIL DnldFindAddress");
            DBG_LEAVE(DbgInfo);
            return(FALSE);
        }

        // Move one byte of data.  Increment the desired address.
        pucPointer[0] = pucData[usIndex];
        ulAddress++;
    }

    DBG_LEAVE(DbgInfo);
    return(TRUE);
}


// DnldLoadBlocks (Trilogy 3)
//		Load fw into RAM
//
//	Parameters:
//		None passed.
//
//	Return Value:
//		NDIS_STATUS_SUCCESS or NDIS_STATUS_FAILURE.

NDIS_STATUS DnldLoadBlocks (PTRILOGY_ADAPTER pAdapter)
{
    DBG_FUNC("DnldLoadBlocks")
    PDBLOCK     pNextBlock;
    PUCHAR      pNextBlockByByte;
    ULONG       ulNextBlockAddress;
    USHORT      usNextBlockLength;
    USHORT      usCount;
    ULONG       ulLength;

    DBG_ENTER(DbgInfo);

    pNextBlock = pFirstBlock;
    pNextBlockByByte = (PUCHAR) pNextBlock;
    ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                         (((ULONG) pNextBlockByByte[1]) << 8) |
                         (((ULONG) pNextBlockByByte[2]) << 16) |
                         (((ULONG) pNextBlockByByte[3]) << 24);
    usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                         (((USHORT) pNextBlockByByte[5]) << 8);
    usCount = 0;

    while (usCount < usBlockDownLoad)
    { // 1

        // if address or length == 0, quit
        if ((pNextBlock == NULL) || (usNextBlockLength == 0) || (ulNextBlockAddress == 0))
        {
            DBG_TRACE(DbgInfo,"pNextBlock %p usNextBlockLength %04x ulNextBlockAddress %08x\n",pNextBlock,usNextBlockLength,ulNextBlockAddress);
            break;
        }
        ulLength = usNextBlockLength;

        DBG_TRACE(DbgInfo,"Download usCount %04x addr %08x len %04x\n",usCount,ulNextBlockAddress,usNextBlockLength);

        AdapAuxPortSet (pAdapter,
                        (PUSHORT) &pNextBlock->ucData[0],
                        ulNextBlockAddress,
                        (USHORT) ((usNextBlockLength)));

        usCount++;
        pNextBlock = (PDBLOCK) &pNextBlock->ucData[ulLength];
        pNextBlockByByte = (PUCHAR) pNextBlock;
        ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                             (((ULONG) pNextBlockByByte[1]) << 8) |
                             (((ULONG) pNextBlockByByte[2]) << 16) |
                             (((ULONG) pNextBlockByByte[3]) << 24);
        usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                             (((USHORT) pNextBlockByByte[5]) << 8);

    } // 1

    // verify
    pNextBlock = pFirstBlock;
    pNextBlockByByte = (PUCHAR) pNextBlock;
    ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                         (((ULONG) pNextBlockByByte[1]) << 8) |
                         (((ULONG) pNextBlockByByte[2]) << 16) |
                         (((ULONG) pNextBlockByByte[3]) << 24);
    usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                         (((USHORT) pNextBlockByByte[5]) << 8);
    usCount = 0;

    while (usCount < usBlockDownLoad)
    { // 2

        // if address or length == 0, quit
        if ((pNextBlock == NULL) || (usNextBlockLength == 0) || (ulNextBlockAddress == 0))
            break;

        DBG_TRACE(DbgInfo,"Compare usCount %04x addr %08x len %04x\n",usCount,ulNextBlockAddress,usNextBlockLength);

        if (DnldCompare (pAdapter,
                         ulNextBlockAddress,
                         &pNextBlock->ucData[0],
                         usNextBlockLength) == FALSE)
        {
            SYMDEBUG(ZDNLD, "@FAIL DnldCompare");
            DBG_LEAVE(DbgInfo);
            return(NDIS_STATUS_FAILURE);
        }

        pNextBlock = (PDBLOCK) &pNextBlock->ucData[usNextBlockLength];
        pNextBlockByByte = (PUCHAR) pNextBlock;
        ulNextBlockAddress = ((ULONG) pNextBlockByByte[0]) |
                             (((ULONG) pNextBlockByByte[1]) << 8) |
                             (((ULONG) pNextBlockByByte[2]) << 16) |
                             (((ULONG) pNextBlockByByte[3]) << 24);
        usNextBlockLength  = ((USHORT) pNextBlockByByte[4]) |
                             (((USHORT) pNextBlockByByte[5]) << 8);
        usCount++;
    } // 2

    DBG_LEAVE(DbgInfo);
    return NDIS_STATUS_SUCCESS;
}

// DnldReset
//
//		Reset everything back to normal before an exit.
//
//	Parameters:
//		pAdapter - pointer to adapter control block.
//
//	Return Value:
//		None.

void DnldReset (PTRILOGY_ADAPTER pAdapter)
{ // 0

    if (pucCompareBuffer != NULL)
#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
        NdisFreeMemory (pucCompareBuffer,
                        CHAIN_BLOCK_SIZE,
                        0);
#else
    NdisFreeMemory (pucCompareBuffer,
                    CHAIN_BLOCK_SIZE,
                        uiMemoryBlockFlags);
#endif
    pucCompareBuffer = NULL;
    return;

} // 0


// DnldScanPastCharacter
//
//		This scans the header of the binary image until the specified character is
//		found, then returns the offset of the next byte beyond that.
//
//	Parameters:
//		usOffset	- the offset in the header where the scan is to start.
//		ucCharacter	- the character that we are looking for.
//
//	Return Value:
//		The offset of the following character if the desired character is found,
//		0xffff if it is not.
//
//	Note:	DnldScanPastCharacter scans the GLOBAL VARIABLE pSourceBuffer for a required character.
//			We may change the function parameter to scan any input buffer.
USHORT DnldScanPastCharacter (USHORT usOffset, UCHAR ucCharacter)
{

    USHORT  usIndex;

    for (usIndex = 0; usIndex < 64; usIndex++)
    {

        if (pSourceBuffer[usOffset++] == ucCharacter)
        {
            return(usOffset);
        }

    }

    return(0xFFFF);
}

#if __KERNEL__

__u32 NumAllocs=0;

#define NDIS_MEM_MAGIC 0x01234567

typedef struct {
    unsigned long MemMagic;
    unsigned long MemLen;
    unsigned long pad[2];
} NdisMemHdr, *PNdisMemHdr;

NDIS_STATUS 
NdisAllocateMemory(
                  void *ptr,
                  ULONG len,
                  ULONG flags,
                  NDIS_PHYSICAL_ADDRESS addr
                  )
{
    DBG_FUNC("NdisAllocateMemory")
    PNdisMemHdr MemHdr;
    void **pptr = (void **)ptr;

    MemHdr = kmalloc(sizeof(NdisMemHdr)+len,GFP_KERNEL);
    if (MemHdr)
    {
        NumAllocs++;
        MemHdr->MemMagic = NDIS_MEM_MAGIC;
        MemHdr->MemLen = len;
        *pptr = (MemHdr+1);
        return(NDIS_STATUS_SUCCESS);
    }
    DBG_ASSERT(0);
    return( NDIS_STATUS_FAILURE );
}

void
_NdisFreeMemory(
              void *ptr,
              ULONG len,
              ULONG flags,
              ULONG line
              )
{
    DBG_FUNC("NdisFreeMemory")
    unsigned char *pptr = (unsigned char *)ptr;
    PNdisMemHdr MemHdr;

    if (!NumAllocs)
    {
        DBG_PRINT("_NdisFreeMemory called from line %u\n",line);
        DBG_ASSERT(NumAllocs);
    }
    MemHdr  = (PNdisMemHdr)(pptr-sizeof(NdisMemHdr));
    if (MemHdr->MemMagic != NDIS_MEM_MAGIC)
    {
        DBG_PRINT("_NdisFreeMemory called from line %u, bogus magic %08x\n",line,(uint)MemHdr->MemMagic);
        DBG_ASSERT(MemHdr->MemMagic == NDIS_MEM_MAGIC);
    }
    if (MemHdr->MemLen != len)
    {
        DBG_PRINT("_NdisFreeMemory called from line %u, bogus len %u, expected %u\n",line,len,(uint)MemHdr->MemLen);
        DBG_ASSERT(MemHdr->MemLen == len);
    }
    
    memset(MemHdr,0,sizeof(NdisMemHdr)+MemHdr->MemLen);

    NumAllocs--;
    kfree(MemHdr);
}


BOOLEAN
AdapGetRid(PTRILOGY_ADAPTER lp, PRID_STRUCT Rid, BOOLEAN InfoRec)
{
    int rc = S24tReadRid(lp,Rid->rid,(__u8 *)Rid->u.data, (Rid->length-1)*2 );
    return( (rc<0) ? FALSE : TRUE);
}

int
AdapAuxPortGet(PTRILOGY_ADAPTER lp, PUSHORT Buff, ULONG Addr, USHORT Wlen)
{
    return(S24tReadAuxPort(lp,(__u8 *)Buff,Addr,Wlen*2));
}

int
AdapAuxPortSet(PTRILOGY_ADAPTER lp, PUSHORT Buff, ULONG Addr, USHORT len)
{
    return(S24tWriteAuxPort(lp,(__u8 *)Buff,Addr,len));
}

void
NdisRawWritePortUshort( ULONG Addr, USHORT Val)
{
    outw( Val, Addr );
}

void
NdisRawReadPortUshort( ULONG Addr, USHORT *Val )
{
    *Val = inw( Addr );
}

void
NdisRawWritePortUchar( ULONG Addr, UCHAR Val)
{
    outb( Val, Addr );
}

void NdisRawReadPortUchar(ULONG Addr, PUCHAR Val)
{
    *Val = inb(Addr);
}

void
AdapWriteUshort(PTRILOGY_ADAPTER lp, USHORT Offset, USHORT Val)
{
    S24tWrite16(lp,Offset,Val);
}

void NdisStallExecution(ULONG usec)
{
    udelay(usec);
}

void NdisMoveMemory (void *dest, void *src, ULONG len)
{
    memcpy(dest,src,len);
}

//	AdapInitMode42
//
//		set the COR for Initiate Mode (HFA3842 only) flags are 0x1F to 
//		start init mode, 0x17 to leave it running
//
//	Arguments:
//		pAdapter	- pointer to the adapter block
//		usFlag		- HFA3842 init mode flag
//
//	Return Value:
//		TRUE if the command was issued without any problems.

BOOLEAN AdapInitMode42 (IN PTRILOGY_ADAPTER pAdapter, UCHAR ucFlags)
{
    DBG_FUNC("AdapInitMode42")
    UCHAR   ucHcr;

    // set the COR RESET bit
    if (S24tReadCor(pAdapter) & COR_RESET)
    {
        S24tWriteCor(pAdapter,COR_CLEAR);
        S24tWriteCor(pAdapter,COR_CLEAR);
    }
    S24tWriteCor(pAdapter,COR_RESET);

    mdelay(100);

    ucHcr = S24tRead8(pAdapter,NIC_HCR);

    SYMDEBUGX(0,"AdapInitMode42 ucHcr",(ULONG)ucHcr);
    SYMDEBUGX(0,"AdapInitMode42 ucFlags",(ULONG)ucFlags);

    if (( ucHcr & 0x10 ) == 0)
        ucFlags &= 0xEF;                            // if so, clear 4Wire bit

    S24tWrite8(pAdapter,NIC_HCR,ucFlags);

    mdelay(2000);

    S24tWriteCor(pAdapter,COR_IO_MODE);

    mdelay(100);

#if DBG
    {
        UCHAR Cor = S24tReadCor(pAdapter);
        DBG_IO(DbgInfo,"Cor %02x\n",Cor);
    }
#endif
    return TRUE;
}

BOOLEAN ESquare(IN  PTRILOGY_ADAPTER pAdapter)
{
    word u;

    NdisRawWritePortUshort( pAdapter->ulIoPAddr + NIC_PARAM0 , 0x1234 );
    NdisRawWritePortUshort (pAdapter->ulIoPAddr + NIC_AUX_BASE, 0);
    NdisRawWritePortUshort (pAdapter->ulIoPAddr + NIC_AUX_OFFSET, 2);
    NdisRawReadPortUshort (pAdapter->ulIoPAddr + NIC_AUX_DATA, &u ) ;
    return( u != 0x1234 );
}

//	AdapEnableAux
//
//		Turn ON/OFF the AUX port access.
//
//	Arguments:
//
//		pAdapter	- pointer to the adapter block
//		bEnable		- open/close flag
//
//	Return Value:
//
//		TRUE if operation succeeded.

BOOLEAN AdapEnableAux (IN  PTRILOGY_ADAPTER pAdapter)               
{ // 0

    ULONG   i;
    USHORT  tmp;

    NdisRawReadPortUshort (pAdapter->ulIoPAddr + NIC_CONTROL, &tmp);

    if (tmp == CTL_AUX_OPEN)
        return TRUE;

    NdisRawWritePortUshort(pAdapter->ulIoPAddr + NIC_PARAM0, AUX_PARAM0_KEY);
    NdisRawWritePortUshort(pAdapter->ulIoPAddr + NIC_PARAM1, AUX_PARAM1_KEY);
    NdisRawWritePortUshort(pAdapter->ulIoPAddr + NIC_PARAM2, AUX_PARAM2_KEY);

    NdisRawWritePortUshort(pAdapter->ulIoPAddr + NIC_CONTROL, CTL_AUX_EN);

    for (i = 0; i < KMAXTRY; i++)
    {
        // 10 usec delay after the write before read
        NdisStallExecution (10);

        // Read status
        NdisRawReadPortUshort (pAdapter->ulIoPAddr + NIC_CONTROL, &tmp);

        if (tmp == CTL_AUX_OPEN)
            break;
    }

    if (tmp != CTL_AUX_OPEN)
    {
        return(FALSE);
    }

    return(TRUE);

}

BOOLEAN AdapIssueCommand(PTRILOGY_ADAPTER lp, USHORT cmd, ULONG timeout, PUSHORT pStatus)
{
    int Status = S24tCommand(lp,cmd,pStatus,NULL,NULL,NULL);

    if (Status < 0)
    {
        return(FALSE);
    }
    return(TRUE);
}

// AdapReadEthernetAddress
//
//      Reads in the Ethernet address from the adapter
//
// Arguments:
//
//		pAdapter - pointer to the adapter block.
//
// Return Value:
//
//		The address is stored in pAdapter->uvPermanentAddress.

BOOLEAN AdapReadEE (IN PTRILOGY_ADAPTER pAdapter)
{
    BOOLEAN bReturn = TRUE;
    USHORT  usStatus;

    bReturn = AdapIssueCommand(pAdapter, CW_READEE, 1000, &usStatus);

    if ((bReturn == FALSE) ||(usStatus != CW10_STATUS_OK))
        return FALSE;

    return TRUE;
}

//	AdapGetPDA
//
//		Retrieves the PDA and stores it in pAdapter->pda.
//
//	Arguments:
//
//		pAdapter - pointer to the adapter block

void AdapGetPDA (IN PTRILOGY_ADAPTER    pAdapter)
{ // 0
    DBG_FUNC("AdapGetPDA")

    DBG_ENTER(DbgInfo);

    AdapAuxPortGet (pAdapter,
                    (PUSHORT) &pAdapter->pda,
                    PDA_START_ADDR,
                    MAX_PDA_LEN_WORDS);

    DBG_LEAVE(DbgInfo);
    return;

} // 0

#if MEMORY_TEST

#define MAX_PATTERNS 8
__u16 Patterns[MAX_PATTERNS] =
{
    0xaa55,
    0x55aa,
    0x0000,
    0xffff,
    0x0123,
    0x4567,
    0x89ab,
    0xcdef
};

int TestBuffer(
              PTRILOGY_ADAPTER lp,
              __u32            Addr,
              __u32            Wlen,
              __u16            Pattern,
              __u16            Increment
              )
{
    DBG_FUNC("TestBuffer")
    __u32 i;
    __u32 miscompares;
    __u32 first_index;
    __u16 first_expected;
    __u16 *ppatt;
    __u16 patt;

    DBG_ASSERT( Wlen <= (BIGBLOCK/2) );

    ppatt = (__u16 *)pucDlBuffer;
    patt = Pattern;
    for (i=0; i<Wlen; i++,ppatt++)
    {
        *ppatt = patt;
        patt += Increment;
    }

    AdapAuxPortSet(lp,(PUSHORT)pucDlBuffer,Addr,Wlen*2);
    memset(pucDlBuffer,0,BIGBLOCK);
    AdapAuxPortGet(lp,(PUSHORT)pucDlBuffer,Addr,Wlen);

    first_index = BIGBLOCK;
    miscompares = 0;
    ppatt = (__u16 *)pucDlBuffer;
    patt = Pattern;
    for (i=0; i<Wlen; i++,ppatt++)
    {
        if (patt != *ppatt)
        {
            miscompares++;
            if (first_index >= Wlen)
            {
                first_index = i;
                first_expected = patt;
            }
        }
        patt += Increment;
    }


    if (miscompares)
    {
        DBG_TRACE(DbgInfo,"%u miscompares out of %u words, first index %u expected %04x pattern %04x increment %u\n",miscompares,Wlen,first_index,first_expected,Pattern,(uint)Increment);
        return(-1);
    }
    return(0);
}
#endif

//	TrilogyDownLoadFirmwareToT3Adapter
//
//		Called when the T3 adapter is initialized or reset. Download the Firmware image to
//		the adapters SRAM.
//
//	Arguments:
//		pAdapter - The adapter structure.
// 
// Return Value:
//
//		Indicates the success or failure of the download.

NDIS_STATUS TrilogyDownLoadFirmwareToT3Adapter(IN PTRILOGY_ADAPTER pAdapter)
{
    DBG_FUNC("TrilogyDownLoadFirmwareToT3Adapter")
    NDIS_STATUS             nsStatus;
    PUSHORT                 pucTmpPrimaryFW;
    PUSHORT                 pucTmpSecondaryFW;
    DOWNLOAD_BLOCK          FWBlock;
    USHORT                  pusEEBuf[256];
    PUSHORT                 pPda;
    USHORT                  usIndex;
    PUSHORT                 pUshort;
    USHORT                  usCount;
    NDIS_PHYSICAL_ADDRESS   ndisPhysicalAddress = {-1, -1};

    DBG_ENTER(DbgInfo);

    AdapInitMode42(pAdapter, HCREEHOLD);
    // initial mode: get HCR, verify serial EE

#if MEMORY_TEST
    for (usCount=0; usCount<MAX_PATTERNS; usCount++)
    {
        for (usIndex=0; usIndex<16; usIndex++)
        {
            if (TestBuffer(pAdapter,0x007e17fe,(0x7e4000-0x7e17fe)/2,Patterns[usCount],usIndex) < 0)
            {
                return(!NDIS_STATUS_SUCCESS);
            }
        }
    }
#endif

    SYMDEBUGX(0,"uiEsquare",pAdapter->uiEsquare);

    // allocate memory for download fw 
#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
    nsStatus = NdisAllocateMemoryWithTag (&pucCompareBuffer,
                                          CHAIN_BLOCK_SIZE,
                                          'umyS');
    nsStatus = NdisAllocateMemoryWithTag (&pucTmpPrimaryFW,
                                          primsymLEN * sizeof(USHORT),
                                          'umyS');
#else
    nsStatus = NdisAllocateMemory (&pucCompareBuffer,
                                   CHAIN_BLOCK_SIZE,
                                   uiMemoryBlockFlags,
                                   ndisPhysicalAddress);

    nsStatus = NdisAllocateMemory ( &pucTmpPrimaryFW,
                                    primsymLEN * sizeof(USHORT),
                                    uiMemoryBlockFlags,
                                    ndisPhysicalAddress);
#endif

    // we already have firmware image in header file
    // point to primary fw image
    NdisMoveMemory (pucTmpPrimaryFW, primsym, primsymLEN * sizeof(USHORT));
    pSourceBuffer = (PUCHAR) pucTmpPrimaryFW;
    FWBlock.usID = FW_UPDATE_END;
    usState = 1;        // STATE_LOADING
    ulBlockNumber = ulTotalBlocks;

    // load primary fw
    nsStatus = DnldDownload(pAdapter, &FWBlock);    

    // free temp primary firmware whether download success or not
#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
    NdisFreeMemory ( pucTmpPrimaryFW,
                     primsymLEN * sizeof(USHORT),
                     0);
#else
    NdisFreeMemory ( pucTmpPrimaryFW,
                     primsymLEN * sizeof(USHORT),
                     uiMemoryBlockFlags);
#endif

    if (nsStatus != NDIS_STATUS_SUCCESS)
    {
        SYMDEBUG(0,"Primary firmware download failed.");
        DBG_LEAVE(DbgInfo);
        return(nsStatus);
    }

    // run primary
    AdapInitMode42(pAdapter, HCRRUN);

    // recommand 10ms, but 2 seems to work
    for (usIndex = 0; usIndex < 2000; usIndex++)
        NdisStallExecution (10);        

    if (AdapEnableAux(pAdapter) == FALSE)
    {
        // recommand 10ms, but 2 seems to work
        for (usIndex = 0; usIndex < 2000; usIndex++)
            NdisStallExecution (10);

        if (AdapEnableAux(pAdapter) == FALSE)
        {
            DBG_LEAVE(DbgInfo);
            return(NDIS_STATUS_FAILURE);
        }
    }

    SYMDEBUG(ZFUNC, "primary loaded successfully");

    // load secsym
    if (AdapReadEE(pAdapter) == FALSE)
    {
        SYMDEBUG(ZERROR, "TrilogyRegisterAdapter : error in AdapReadEE");
        DBG_LEAVE(DbgInfo);
        return NDIS_STATUS_FAILURE;
    }

    AdapAuxPortGet (pAdapter,
                    pusEEBuf,
                    BUFADDR,
                    256);

    if (pusEEBuf[1] != BAR_CIS)
    {
        SYMDEBUG(ZERROR, "TrilogyRegisterAdapter : error reading pusEEBuf");
        DBG_LEAVE(DbgInfo);
        return NDIS_STATUS_FAILURE;
    }

    // the following code is copied from DOS Utilities DL.C
    pUshort = pusEEBuf;
    pUshort += 0x80;
    usCount = *pUshort++ >> 1;

    if (usCount > MAX_PDA_LEN_WORDS)
    {
        SYMDEBUGX(ZERROR, "TrilogyRegisterAdapter : invalid PDA Length, pda length = ", usCount);
        DBG_LEAVE(DbgInfo);
        return NDIS_STATUS_FAILURE;
    }

    pPda = pAdapter->pda;
    ++pUshort;

    for (usIndex = 0; usIndex < usCount; usIndex++)
        *pPda++ = *pUshort++;

    // allocate memory for download fw 
#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
    nsStatus = NdisAllocateMemoryWithTag (&pucCompareBuffer,
                                          CHAIN_BLOCK_SIZE,
                                          'umyS');
    nsStatus = NdisAllocateMemoryWithTag (&pucTmpSecondaryFW,
                                          secsymLEN * sizeof(USHORT),
                                          'umyS');
#else
    nsStatus = NdisAllocateMemory ( &pucTmpSecondaryFW,
                                    secsymLEN * sizeof(USHORT),
                                    uiMemoryBlockFlags,
                                    ndisPhysicalAddress);

    nsStatus = NdisAllocateMemory (&pucCompareBuffer,
                                   CHAIN_BLOCK_SIZE,
                                   uiMemoryBlockFlags,
                                   ndisPhysicalAddress);
#endif

    // we already have firmware image in header file
    // point to primary fw image
    NdisMoveMemory (pucTmpSecondaryFW, secsym, secsymLEN * sizeof(USHORT));
    pSourceBuffer = (PUCHAR) pucTmpSecondaryFW;
    FWBlock.usID = FW_UPDATE_END;
    usState = 1;        // STATE_LOADING
    ulBlockNumber = ulTotalBlocks;

    // load primary fw
    nsStatus = DnldDownload(pAdapter, &FWBlock);    

    // free temp secondary firmware whether download success or not
#if defined(NDIS50_MINIPORT) || defined(NDIS51_MINIPORT)
    NdisFreeMemory ( pucTmpSecondaryFW,
                     secsymLEN * sizeof(USHORT),
                     0);
#else
    NdisFreeMemory ( pucTmpSecondaryFW,
                     secsymLEN * sizeof(USHORT),
                     uiMemoryBlockFlags);
#endif
    if (nsStatus != NDIS_STATUS_SUCCESS)
    {
        SYMDEBUG(ZERROR, "TrilogyRegisterAdapter : DnldDownload failed");
    }
#if __KERNEL__
    if (NumAllocs)
    {
        printk(KERN_ERR "Orphaned memory (%u) in TrilogyDownLoadFirmwareToT3Adapter\n",NumAllocs);
    }
#endif
    DBG_LEAVE(DbgInfo);
    return(nsStatus);
}

#endif

#endif

