/*

  Copyright (C) 2000, The MITRE Corporation

  Use of this software is subject to the terms of the GNU General
  Public License version 2.

  Please read the file LICENSE for the exact terms.

*/

/*
 * Handles command line switches.  Instantiates the
 * LinkDiscover object for an IP interface and runs it.
 *
 * Author: Kevin H. Grace, kgrace@mitre.org
 *         The MITRE Corporation
 *         202 Burlington Rd
 *         Bedford, MA  01730
 *         
 *
 * $Id: LnMain.C,v 1.3 2004/01/03 15:04:33 br1 Exp $
 *  
 */
#include <LnDiscover.h>
#include <UtMain.h>
#include <UtString.h>
#include <UtReport.h>
//#include "rsacipher.h"

static LinkDiscover* application = 0;
  
static void Version() {
  char *version = "mmdiscover - Mobile Mesh Link Discovery Protocol v1.0\n";
  Report::Info(version);
}

static void Usage() {
  char *usage = "Usage: mmdiscover [-d file] [-f file] [-h] -i ifname [-vz] \n"\
    " -d file   , use flags 'file' to enable debug output \n"\
    " -f file   , use config 'file' instead of mmdiscover.conf \n"\
    " -h        , display this help message \n"\
    " -i ifname , the name of the interface whose neighbors will be discovered \n"\
    " -v        , display version information \n"\
    " -e 	, set encryption on \n"\
    " -z        , run in the foreground instead of running as a daemon \n";  
  Report::Error(usage);
}

// static bool isEncrypted; Defined in UtUdpsocket.h

int main(int argc, char* argv[]) {
  String ident("mmdiscover");
  String configFile  = "/etc/mobilemesh/mmdiscover.conf";
  String debugFile   = "";
  String ifname = "";
  int    flags       = Report::fError | Report::fWarn | Report::fDebug;
  bool isDaemon = true;
  //InitSSL *initssl;

  Report::UseOstream(&cerr, flags);
  //initssl = new InitSSL();

  // Presets encryption to no
  isEncrypted = false;
  // Parse all command line switches
  bool done = false;
  while(!done) {
    int c = getopt(argc,argv,"d:f:hi:vez");
    switch(c) {
    case 'd':
      debugFile = optarg;
      break;
    case 'f':
      configFile = optarg;
      break;
    case 'h':
      Usage();
      break;
    case 'i':
      ifname = optarg;
      break;
    case 'v':
      Version();
      exit(0);
      break;
    case 'e':
      isEncrypted = true;
      break;
    case 'z':
      isDaemon = false;
      break;
    case EOF:
      done = true;
      break;
    default:
      Usage();
    }
  }
  if(optind < argc) Usage();
  
  if(ifname == "") {
    Report::Error("Must specify interface name with '-i' option");
  }

  if(isDaemon) {
    Daemonize();           
    Report::UseOstream(0); // Don't send anything to cerr

    // Use syslog to record output
    Report::UseSyslog(ident,flags);
  }
    

  // Convert any relative file names to absolute paths so we can access them after
  // changing from the current directory
  debugFile  = AbsoluteFilename(debugFile);
  configFile = AbsoluteFilename(configFile);

  // We must run in the directory where the Unix sockets exists because
  // their names are limited to something like 14 characters
  Chdir("/var/run/mobilemesh");
    
  // In case a signal occurs, register for garbage collection
  GarbageCollection<LinkDiscover>::Register(&application);

  // Build the application and run it
  application = new LinkDiscover(ifname,configFile,debugFile);
  application->Run();

  // Clean up if we get here
  delete(application);
  application = 0;

  return(0);
}












