#include "WirelessQualityMonitor.h"


using namespace std;


void print_sa_ip( const struct sockaddr_in ip ) {
	printf( "%s", inet_ntoa( ip.sin_addr ) );
}


void print_sa_mac( const struct sockaddr hwa ) {
	printf("%02X:%02X:%02X:%02X:%02X:%02X", hwa.sa_data[0], hwa.sa_data[1], hwa.sa_data[2], 	hwa.sa_data[3], hwa.sa_data[4], hwa.sa_data[5]);
}


WirelessQualityMonitor::WirelessQualityMonitor( const char* name, const unsigned int updateThreshold ) {
	skfd = socket(AF_INET, SOCK_DGRAM, 0);
	strncpy( ifname, name, 16 );

	qualMap = new QualMap();
	macMap = new MacMap();
	toArpMap = new QualMap();
	//printf("wireless quality monitor constructed\n");
	cUpdateThreshold = updateThreshold;
}


WirelessQualityMonitor::~WirelessQualityMonitor() {

	// delete all IwQual objects
	QualMap::iterator it;
	IwQual* h;
	for ( it = qualMap->begin(); it != qualMap->end(); it++ ) {
		h = it->second;
		delete(h);
	}
	for ( it = toArpMap->begin(); it != toArpMap->end(); it++ ) {
		h = it->second;
		delete(h);
	}

	delete( qualMap );
	delete( macMap );
	delete( toArpMap );

	close(skfd);
}


int WirelessQualityMonitor::setSpy()
{
	struct iwreq wrq;
	struct sockaddr hw_address[IW_MAX_SPY];

	QualMap::iterator it;
	IwQual* h;

	int i=0;
	for ( it = qualMap->begin(); it != qualMap->end() && i<IW_MAX_SPY; it++ ) {
		h = it->second;
		if (h!=NULL) {
			hw_address[i] = h->getMacSockaddr();
			//printf( "spying " );
			//print_sa_mac(hw_address[i]);
			//printf( "\n" );

			i++;
		}
	}

	if (i>0) {
		// send addresses to the driver
		wrq.u.data.pointer = (caddr_t) hw_address;
		wrq.u.data.length = i;
		wrq.u.data.flags = 0;
		strncpy(wrq.ifr_name, ifname, IFNAMSIZ);
		if (ioctl(skfd, SIOCSIWSPY, &wrq) < 0) {
			throw SetIwSpyException();
		}
		return i;
	}
	return 0;
}


int WirelessQualityMonitor::getSpy()
{
	int n;
	int i;
	struct iwreq wrq;
	char buffer[(sizeof(struct iw_quality) + sizeof(struct sockaddr)) * IW_MAX_SPY];
	struct sockaddr* hwa;
	struct iw_quality* qual;
	IwQual* q;

	// get addresses
	wrq.u.data.pointer = (caddr_t)buffer;
	wrq.u.data.length = IW_MAX_SPY;
	wrq.u.data.flags = 0;
	strncpy(wrq.ifr_name, ifname, IFNAMSIZ);

	if (ioctl(skfd, SIOCGIWSPY, &wrq) < 0) {
		throw GetIwSpyException();
	}

	/* Number of addresses */
	n = wrq.u.data.length;

	hwa = (struct sockaddr *) buffer;
  	qual = (struct iw_quality *) (buffer + (sizeof(struct sockaddr) * n));

	for(i = 0; i < n; i++) {
		EthAddr addr(hwa[i]);
		q = (*macMap)[addr];
		if (q!=NULL) {
			q->setSignalNoise( qual[i].level - 0x100,
					qual[i].noise - 0x100,
					qual[i].updated & 0x7 );
		}
		//else {
			//printf("get: MAC address not found: ");
			//print_sa_mac(hwa[i]);
			//printf("\n");
		//}
	}
	//print();
	return 0;
}


bool WirelessQualityMonitor::add( const sockaddr_in ip ) {
 	IwQual* q = new IwQual();
	q->setIp( ip );

	if ( arp( q ) ) {
		// hwaddr found:
		// insert into IP map and MAC map (need this for get() later)
		(*qualMap)[q->getIp()] = q;
		(*macMap)[q->getMac()] = q;
		//printf("inserted "); print_sa_ip( ip ); printf(" (%d)\n", q->getIp());
		return true;
	} else {
		// remember that we have to look up MAC
		(*toArpMap)[q->getIp()] = q;
		return false;
	}
}


/* first shot:
int WirelessQualityMonitor::getMetric( const __u32 ip ) {
	IwQual* q;
	int m=0;
	q = (*qualMap)[ip];
	if (q != NULL) {
		m = q->getAverageSNR();
		if (m>60) {
			m = 2;
		} else {
			m = (69-m)/3;
		}
	} else {
		printf("no metric found!\n");
		m = 15;
	}
	return m;
}
*/

int WirelessQualityMonitor::getMetric( const __u32 ip ) {
	IwQual* q;
	int m=0;
	q = (*qualMap)[ip];
	if (q != NULL) {
		m = q->getAverageSNR();
		if (m>20) {
			m = 2;
		} else {
			m = 23-m;
		}
	} else {
		printf("no metric found!\n");
		m = 15;
	}
	return m;
}


int WirelessQualityMonitor::getMetric( struct sockaddr_in ip ) {
 	IwQual h;
	h.setIp( ip );
	//print_sa_ip( ip );
	return getMetric( h.getIp() );
}


int WirelessQualityMonitor::getSNR( const __u32 ip ) {
	IwQual* q;
	q = (*qualMap)[ip];
	if (q != NULL) {
		return q->getAverageSNR();
	}
	return -1;
}


bool WirelessQualityMonitor::updateThreshold( const __u32 ip ) {
	IwQual* q;
	q = (*qualMap)[ip];
	if (q != NULL) {
		//printf("threshold (%d): %d\n", ip, q->updateThreshold() );
		if (q->getDeltaSNR() > cUpdateThreshold)  {
			return true;
		}
	}
	//printf("threshold (%d): not found\n");
	return false;
}


void WirelessQualityMonitor::remove( const struct sockaddr_in ip ) {
	IwQual* q;
	q = (*qualMap)[ip.sin_addr.s_addr];
	qualMap->erase( ip.sin_addr.s_addr );
	macMap->erase( q->getMac() );
	delete(q);
}


void WirelessQualityMonitor::printMac() {
	MacMap::iterator it;
	IwQual* h;
	EthAddr e;

	printf( "MAC---\n" );
	for ( it = macMap->begin(); it != macMap->end(); it++ ) {
		h = it->second;
		e = it->first;
		print_sa_ip( h->getIpSockaddr() );
		printf(" - ");
		print_sa_mac( h->getMacSockaddr() );
		printf("\nSock: ");
		print_sa_mac( e.getSockaddr() );
		printf("\n");
	}
	printf( "MAC---\n" );
}


void WirelessQualityMonitor::print() {
	QualMap::iterator it;
	IwQual* h;

	//printf( "---\n" );
	for ( it = qualMap->begin(); it != qualMap->end(); it++ ) {
		h = it->second;
		if (h!=NULL) {
			print_sa_ip( h->getIpSockaddr() );
			printf(" - ");
			print_sa_mac( h->getMacSockaddr() );
			printf(" - %d/%d - %d - %d\n", h->getSignal(), h->getNoise(),
				h->getSNR(), h->getAverageSNR() );
		}
	}
	//printf( "---\n" );
}


/* Translate IP addresses to MAC addresses */
bool WirelessQualityMonitor::arp( IwQual* q )
{
	struct arpreq arp_query;

	arp_query.arp_pa = *(q->getIpSockaddrP());
	arp_query.arp_ha.sa_family = 0;
	arp_query.arp_flags = 0;
     	strncpy(arp_query.arp_dev, ifname, IFNAMSIZ);

	if((ioctl(skfd, SIOCGARP, &arp_query) < 0) || !(arp_query.arp_flags & ATF_COM)) {
		// could not resolve
		printf("did not find ");
		print_sa_ip(q->getIpSockaddr());
		printf(" in arp table\n");
		return false;
	}

	/* Store new MAC address */
	q->setMAC( arp_query.arp_ha );
	return true;
}


bool WirelessQualityMonitor::resolveRemainingMacs() {
	QualMap::iterator it;
	IwQual* h;
	int i=0;
	char cmdbuffer[64];

	for ( it = toArpMap->begin(); it != toArpMap->end(); it++ ) {
		h = it->second;
		if (h!=NULL) {
			if ( arp(h) ) {
				(*qualMap)[h->getIp()] = h;
				(*macMap)[h->getMac()] = h;
				toArpMap->erase(h->getIp());
				i++;
			} else {
				sprintf(cmdbuffer, "ping -c 1 %s &", inet_ntoa( (h->getIpSockaddr()).sin_addr ) );
				system(cmdbuffer);
			}
		}
	}
	//printf("resolved %d (%d remaining)\n", i, toArpMap->size());
	if (i>0) {
		return true;
	} else  {
		return false;
	}
}


/*
int main( int argc, char* argv )
{
	WirelessQualityMonitor* we = new WirelessQualityMonitor();

	struct sockaddr_in ip;
	ip.sin_family = AF_INET;
	ip.sin_port = 0;
	ip.sin_addr.s_addr= inet_addr("192.168.1.100");
	we->add(ip);


	struct sockaddr_in ip2;
	ip2.sin_family = AF_INET;
	ip2.sin_port = 0;
	ip2.sin_addr.s_addr= inet_addr("192.168.1.101");
	we->add(ip2);


	we->print();
	//we->printMac();

	try {
		we->setSpy();

		while(1) {
			we->getSpy();
			we->print();
			sleep(2);
		}
	}
	catch (IwSpyException) {
		printf("catched exception\n");
	}
}
*/
