/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001, 2004 NoMachine, http://www.nomachine.com.          */
/*                                                                        */
/* NXAGENT, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

/*

Copyright 1993 by Davor Matic

Permission to use, copy, modify, distribute, and sell this software
and its documentation for any purpose is hereby granted without fee,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation.  Davor Matic makes no representations about
the suitability of this software for any purpose.  It is provided "as
is" without express or implied warranty.

*/

#include "X.h"
#include "Xproto.h"
#include "miscstruct.h"
#include "dixstruct.h"
#include "fontstruct.h"
#include "gcstruct.h"
#include "scrnintstr.h"
#include "windowstr.h"
#include "pixmapstr.h"
#include "region.h"
#include "servermd.h"

#include "Agent.h"

#include "Display.h"
#include "Options.h"
#include "Screen.h"
#include "GC.h"
#include "Font.h"
#include "GCOps.h"
#include "Drawable.h"
#include "Window.h"
#include "Visual.h"
#include "Control.h"
#include "Trap.h"

#ifdef NXAGENT_MVFB
#include "../../fb/fb.h"
#endif

#include NXAGENT_NXLIB_INCLUDE
#include NXAGENT_NXPACK_INCLUDE

/*
 * Set here the required log level.
 */

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

/*
 * Preferred image compression settings.
 */

#define NXAGENT_USES_JPEG
#define NXAGENT_USES_PNG
#define NXAGENT_USES_JPEG_PNG

extern int nxagentPackMethod;
extern int nxagentPackQuality;
extern int nxagentTightThreshold;

/*
 * Never pack images having a data size smaller
 * than this threshold.
 */

#define IMAGE_PACK_THRESHOLD  768

/*
 * Used to reformat image when connecting to
 * displays having different byte order.
 */

extern void BitOrderInvert(unsigned char *, int);
extern void TwoByteSwap( unsigned char *, register int);
extern void FourByteSwap(register unsigned char *, register int);

/*
 * Used in image compression and streaming.
 */

extern long nxagentStopImgSz;
extern long nxagentLastImgSize;
extern ClientPtr nxagentLastImgClient;

/*
 * Store the last visual used to unpack the
 * images per each given client.
 */

static Visual nxagentUnpackVisual[MAX_CONNECTIONS];

/*
 * Store alpha data set fro the given client.
 */

typedef struct _UnpackAlpha
{
  int entries;
  unsigned char *data;

} UnpackAlphaRec;

typedef UnpackAlphaRec *UnpackAlphaPtr;

static UnpackAlphaRec *nxagentUnpackAlpha[MAX_CONNECTIONS];

/*
 * Predefined visual used for drawables having 
 * 32 bits depth and an associated picture.
 */

static Visual nxagentAlphaVisual;

/*
 * Store data of the last image got from the
 * real X server. Used to guess content of
 * remote window and avoid a round trip.
 */

XImage *nxagentLastGetImage = NULL;

/*
 * True if we are already waiting for a
 * former GetImage reply.
 */

int nxagentLastGetImageClient = UNDEFINED;

/*
 * These are taken from dispatcher.
 */

#define ClientOrder(client) \
    ((client)->swapped ? !ServerOrder() : ServerOrder())

int ServerOrder(void)
{
  int whichbyte = 1;

  if (*((char *) &whichbyte))
      return LSBFirst;

  return MSBFirst;
}

#define TEST

void nxagentImageReformat(char *base, int nbytes, int bpp, int order)
{
  /*
   * This is used whenever we need to swap the image data.
   * If we got an image from a X server having a different
   * endianess, we will need to redormat the image to match
   * our own image-order so that ProcGetImage can return
   * the expected format to the client.
   */

  switch (bpp)
  {
    case 1:
    {
      if (BITMAP_BIT_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Bit order invert with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        BitOrderInvert((unsigned char *) base, nbytes);
      }

      #if IMAGE_BYTE_ORDER != BITMAP_BIT_ORDER && BITMAP_SCANLINE_UNIT != 8

      nxagentImageReformat(base, nbytes, BITMAP_SCANLINE_UNIT, order);

      #endif

      break;
    }
    case 4:
    case 8:
    {
      break;
    }
    case 16:
    {
      if (IMAGE_BYTE_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Two bytes swap with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        TwoByteSwap((unsigned char *) base, nbytes);
      }

      break;
    }
    case 32:
    {
      if (IMAGE_BYTE_ORDER != order)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentReformatImage: Four bytes swap with size [%d] "
                    "bits per pixel [%d] byte order [%d].\n", nbytes, bpp, order);
        #endif

        FourByteSwap((unsigned char *) base, nbytes);
      }

      break;
    }
  }
}

#undef TEST

Visual *nxagentImageVisual(DrawablePtr pDrawable, int depth)
{
  /*
   * We always use our default visual, corresponding to
   * depth of the real display. If the drawable is 32 bits
   * and it has been associated to a Picture we can assume
   * that it is using the special RENDER visual having the
   * alpha channel. In this case we assume a fixed color-
   * mask layout. Is this assumption always true? What if
   * agent has a different endianess than the X server?
   */

  if (nxagentAlphaVisual.red_mask == 0)
  {
    nxagentAlphaVisual.red_mask = 0x00ff0000;
    nxagentAlphaVisual.green_mask = 0x0000ff00;
    nxagentAlphaVisual.blue_mask = 0x000000ff;
  }

  if (nxagentDrawablePicture(pDrawable) != NULL && depth == 32)
  {
    return &nxagentAlphaVisual;
  }
  else
  {
    return nxagentDefaultVisual(pDrawable -> pScreen);
  }
}

int nxagentImageLength(int width, int height, int format, int leftPad, int depth)
{
  int line = 0;

  if (format == XYBitmap)
  {
    line = BitmapBytePad(width + leftPad);
  }
  else if (format == XYPixmap)
  {
    line  = BitmapBytePad(width + leftPad);

    line *= depth;
  }
  else if (format == ZPixmap)
  {
    line = PixmapBytePad(width, depth);
  }

  return line * height;
}

int nxagentImagePad(int width, int format, int leftPad, int depth)
{
  int line = 0;

  if (format == XYBitmap)
  {
    line = BitmapBytePad(width + leftPad);
  }
  else if (format == XYPixmap)
  {
    line = BitmapBytePad(width + leftPad) * depth;
  }
  else if (format == ZPixmap)
  {
    line = PixmapBytePad(width, depth);
  }

  return line;
}

unsigned char *nxagentImageAlpha(XImage *ximage)
{
  unsigned char *pAlphaData, *pData;

  int dataSize, pos, i;

  /*
   * Use one byte per pixel.
   */

  dataSize = (ximage -> bytes_per_line * ximage -> height) >> 2;

  pAlphaData = xalloc(dataSize);

  if (pAlphaData == NULL)
  {
    return NULL;
  }

  if (ximage -> byte_order == MSBFirst)
  {
    pos = (ServerOrder() == LSBFirst ? 0 : 3);
  }
  else
  {
    pos = (ServerOrder() == LSBFirst ? 3 : 0);
  }

  pData = pAlphaData;

  for (i = 0; i < dataSize; i++, pData++)
  {
    *pData = ximage -> data[(i * 4) + pos];
  }

  return pAlphaData;
}

/*
GFPZR
*/

void nxagentSetUnpackAlpha(DrawablePtr pDrawable, XImage *pImage, ClientPtr pClient)
{
  int entries;

  unsigned char *pAlphaData;

  if (nxagentDrawablePicture(pDrawable) != NULL && pImage -> depth == 32)
  {
    pAlphaData = nxagentImageAlpha(pImage);

    if (!pAlphaData)
    {
      #ifdef PANIC
      fprintf(stderr, "nxagentSetUnpackAlpha: PANIC! Can't allocate data for the alpha channel.\n");
      #endif

      return;
    }

    entries = (pImage -> bytes_per_line * pImage -> height) >> 2;

    if (!nxagentUnpackAlpha[pClient -> index] ||
            nxagentUnpackAlpha[pClient -> index] -> entries != entries ||
                memcmp(nxagentUnpackAlpha[pClient -> index] -> data,
                           pAlphaData, entries) != 0)
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentSetUnpackAlpha: Sending alpha channel with width [%d] height [%d] "
                  "bytes per line [%d] entries [%d].\n", pImage -> width, pImage -> height,
                      pImage -> bytes_per_line, entries);
      #endif

      NXSetUnpackAlpha(nxagentDisplay, pClient -> index, entries, pAlphaData);

      if (nxagentUnpackAlpha[pClient -> index])
      {
        xfree(nxagentUnpackAlpha[pClient -> index] -> data);
      }
      else
      {
        nxagentUnpackAlpha[pClient -> index] = xalloc(sizeof(UnpackAlphaRec));
      }

      nxagentUnpackAlpha[pClient -> index] -> data = pAlphaData;
      nxagentUnpackAlpha[pClient -> index] -> entries = entries;
    }
    else
    {
      #ifdef DEBUG
      fprintf(stderr, "nxagentSetUnpackAlpha: Matched alpha channel with [%d] entries.\n",
                  entries);
      #endif

      xfree(pAlphaData);
    }
  }
}

void nxagentFillSpans(pDrawable, pGC, nSpans, pPoints, pWidths, fSorted)
     DrawablePtr pDrawable;
     GCPtr pGC;
     int nSpans;
     xPoint *pPoints;
     int *pWidths;
     int fSorted;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to FillSpans on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
     fbFillSpans(nxagentVirtualDrawable(pDrawable), pGC, nSpans, pPoints, pWidths, fSorted);
  }
#else
  ErrorF("nxagent warning: function nxagentFillSpans not implemented\n");
#endif
}

void nxagentSetSpans(pDrawable, pGC, pSrc, pPoints, pWidths, nSpans, fSorted)
     DrawablePtr pDrawable;
     GCPtr pGC;
     char * pSrc;
     xPoint *pPoints;
     int *pWidths;
     int nSpans;
     int fSorted;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to SetSpans on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
     fbSetSpans(nxagentVirtualDrawable(pDrawable), pGC, pSrc, pPoints, pWidths, nSpans, fSorted);
  }
#else
  ErrorF("nxagent warning: function nxagentSetSpans not implemented\n");
#endif
}

void nxagentGetSpans(pDrawable, maxWidth, pPoints, pWidths, nSpans, pBuffer)
     DrawablePtr pDrawable;
     int maxWidth;
     xPoint *pPoints;
     int *pWidths;
     int nSpans;
     char *pBuffer;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: going to GetSpans on mvfb pixmap = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbGetSpans(nxagentVirtualDrawable(pDrawable), maxWidth, pPoints, pWidths, nSpans, pBuffer);
  }
#else
  ErrorF("nxagent warning: function nxagentGetSpans not implemented\n");
#endif
}

#ifdef NXAGENT_QUERYBSIZE
void
nxagentQueryBestSize(class, pwidth, pheight, pScreen)
int class;
unsigned short *pwidth;
unsigned short *pheight;
ScreenPtr pScreen;
{
  /* cut'n'paste from mfbmisc.c */
  unsigned width, test;

  switch(class)
  {
    case CursorShape:
      if (*pwidth > pScreen->width)
        *pwidth = pScreen->width;
      if (*pheight > pScreen->height)
        *pheight = pScreen->height;
      break;
    case TileShape:
    case StippleShape:
      width = *pwidth;
      if (!width) break;
      /* Return the closes power of two not less than what they gave me */
      test = 0x80000000;
      /* Find the highest 1 bit in the width given */
      while(!(test & width))
         test >>= 1;
      /* If their number is greater than that, bump up to the next
       *  power of two */
      if((test - 1) & width)
         test <<= 1;
      *pwidth = test;
      /* We don't care what height they use */
      break;
  }
}
#else
void nxagentQueryBestSize(class, pWidth, pHeight, pScreen)
  int class;
  short *pWidth;
  short *pHeight;
  ScreenPtr pScreen;
{
  unsigned int width, height;

  width = *pWidth;
  height = *pHeight;

  XQueryBestSize(nxagentDisplay, class,
                 nxagentDefaultWindows[pScreen->myNum],
                 width, height, &width, &height);

  *pWidth = width;
  *pHeight = height;
}
#endif /* NXAGENT_QUERYBSIZE */

void nxagentPutImage(pDrawable, pGC, depth, x, y, w, h, leftPad, format, data)
  DrawablePtr pDrawable;
  GCPtr       pGC;
  int         depth, x, y, w, h;
  int         leftPad;
  int         format;
  char        *data;
{
  Visual *pVisual;

  int length;
  int bytesPerLine;
  int numOfSplit;
  int totalHeight;

  Bool needSplit = False;

  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPutImage(nxagentVirtualDrawable(pDrawable), pGC, depth,
                     x, y, w, h, leftPad, format, data);
    }

    return;
  }

  if (nxImageCacheStat < 0)
  {
    NXCacheInit(100);
  }

  #ifdef NXAGENT_MVFB

  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
    DrawablePtr pVirtual = nxagentVirtualDrawable(pDrawable);

    if (pVirtual -> bitsPerPixel == 0)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentPutImage: WARNING! Virtual pixmap at [%p] has "
                  "invalid bits per pixel.\n", (void *) pVirtual);

      fprintf(stderr, "nxagentPutImage: WARNING! While handling image for pixmap at [%p].\n",
                  (void *) pDrawable);
      #endif

      return;
    }

    #ifdef NXAGENT_MVFB_DEBUG 
    fprintf(stderr, "GCOps: going to put image GC [%lx] on mvfb "
                "pixmap = [%lx], w = [%i], h = [%i] format = [%s]\n",
                    (unsigned long) pGC, (unsigned long) nxagentVirtualDrawable(pDrawable),
                        w, h, (format == ZPixmap) ? "ZPixmap" : "XYPixmap");
    #endif

    fbPutImage(nxagentVirtualDrawable(pDrawable), pGC, depth,
                   x, y, w, h, leftPad, format, data);
  }

  #endif

  /*
   * Get visual according to drawable and depth.
   */

  pVisual = nxagentImageVisual(pDrawable, depth);

  #ifdef NXAGENT_STOPBIGREQ

  if ((nxagentStopImgSz > 0) &&
          (nxagentLastImgSize > nxagentStopImgSz))
  {
    needSplit = True;
  }

  #endif

  /*
   * Get bytes per line according to format.
   */

  bytesPerLine = nxagentImagePad(w, format, leftPad, depth);

  totalHeight = h;

  length = bytesPerLine * h;

  h = MIN(NXAGENT_MAX_REQUEST_EFF_SIZE, length) / bytesPerLine;

  numOfSplit = totalHeight / h;

  if (totalHeight % h)
  {
    numOfSplit++;
  }

  #ifdef DEBUG
  fprintf(stderr, "GCOps: going to split image, geometry (%d,%d,%d,%d), "
              "len %d, bpl %d, sections %d\n", x, y, w, h,
                  length, bytesPerLine, numOfSplit);
  #endif

  while (numOfSplit)
  {
    nxagentPutSplitImage(pDrawable, pGC, depth, x, y, w, h,
                             leftPad, format, data, pVisual, needSplit);
    y += h;

    if (--numOfSplit == 1)
    {
      h = totalHeight - y;
    }
  }
}

void nxagentPutSplitImage(pDrawable, pGC, depth, x, y, w, h, leftPad,
                              format, data, pVisual, needSplit)
  DrawablePtr pDrawable;
  GCPtr       pGC;
  int         depth, x, y, w, h;
  int         leftPad;
  int         format;
  char        *data;
  Visual      *pVisual;
  Bool        needSplit;
{
  NXPackedImage *packedImage      = NULL;
  XImage        *plainImage       = NULL;
  unsigned char *md5              = NULL;

  extern Bool nxagentUseNXTrans;
  extern int ServerOrder(void);

  unsigned int packMethod = nxagentPackMethod;

  /*
   * XCreateImage is the place where the leftPad should be passed.
   * The image data is received from our client unmodified. In
   * theory what we would need to do is just creating an appropri-
   * ate XImage structure based on the incoming data and let Xlib
   * do the rest. Probably we don't have to pass leftPad again in
   * the src_x of XPutImage otherwise the src_x would make Xlib
   * to take into account the xoffset field twice. Unfortunately
   * passing the leftPad doesn't work.
   *
   * plainImage = XCreateImage(nxagentDisplay, pVisual,
   *                           depth, format, leftPad, (char *) data,
   *                           w, h, BitmapPad(nxagentDisplay),
   *                           nxagentImagePad(w, format, leftPad, depth));
   */

   if (!(plainImage = XCreateImage(nxagentDisplay, pVisual,
                                   depth, format, 0, (char *) data,
                                   w, h, BitmapPad(nxagentDisplay),
                                   nxagentImagePad(w, format, leftPad, depth))))

  {
    #ifdef WARNING
    fprintf(stderr, "nxagentPutSplitImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  #ifdef DEBUG
  fprintf(stderr, "nxagentPutSplitImage: Handling image with geometry [%d, %d] depth [%d].\n",
              w, h, depth);

  fprintf(stderr, "nxagentPutSplitImage: Default pack method is [%d] quality is [%d].\n",
              nxagentPackMethod, nxagentPackQuality);
  #endif

  /*
   * We got image data from the X client with our own
   * server endianess while the ximage was initialized
   * with the endianess of the remote display. Fix the
   * ximage structure to carry to appropriate flags.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentPutSplitImage: Image has order [%d] depth [%d] bits-per-pixel [%d].\n",
              plainImage -> byte_order, plainImage -> depth, plainImage -> bits_per_pixel);
  #endif

  nxagentSetImageFormat(plainImage);

  #ifdef TEST
  fprintf(stderr, "nxagentPutSplitImage: Image has now order [%d].\n",
              plainImage -> byte_order);
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSplitImage: Image has visual with RGB color masks [%0lx][%0lx][%0lx].\n",
              pVisual -> red_mask, pVisual -> green_mask, pVisual -> blue_mask);
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSplitImage: Server order is [%d] client order is [%d].\n",
              ServerOrder(), ClientOrder(nxagentLastImgClient));
  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentPutSplitImage: Display image order is [%d] bitmap order is [%d].\n",
              ImageByteOrder(nxagentDisplay), BitmapBitOrder(nxagentDisplay));
  #endif

  /*
   * Always clean the image to help any future
   * cache method but don't clean drawables that
   * are used in render extension and that use
   * the alpha channel.
   */

  if (nxagentDrawablePicture(pDrawable) == NULL)
  {
    NXCleanInPlaceImage(plainImage);
  }

 /*
  * Don't pack images with depth 1 or having a
  * data size less than the given threshold.
  */

  if (depth != 1 && nxagentImageLength(w, h, format, leftPad, depth) > IMAGE_PACK_THRESHOLD)
  {
    if (nxagentUseNXTrans &&
            memcmp(&nxagentUnpackVisual[nxagentLastImgClient -> index],
                       pVisual, sizeof(Visual)))
    {
      memcpy(&nxagentUnpackVisual[nxagentLastImgClient -> index], pVisual, sizeof(Visual));

      #ifdef DEBUG
      fprintf(stderr, "nxagentPutSplitImage: Sending new geometry for client [%d].\n",
                  nxagentLastImgClient -> index);
      #endif
      
      NXSetUnpackGeometry(nxagentDisplay, nxagentLastImgClient -> index,
                              (Screen *) pDrawable->pScreen, pVisual);
    }

    if (nxImageCacheStat > 0)
    {
      packedImage = NXCacheFindImage(plainImage, &packMethod, &md5);

      if (packedImage != NULL)
      {
        nxagentSetUnpackAlpha(pDrawable, plainImage, nxagentLastImgClient);

        #ifdef NXAGENT_STOPBIGREQ

        if (needSplit)
        {
          nxSleepByBigReq(nxagentLastImgClient);
        }

        #endif

        NXPutPackedImage(nxagentDisplay, nxagentLastImgClient -> index, nxagentDrawable(pDrawable),
                             nxagentGC(pGC), packedImage, packMethod, depth,
                                 0, 0, x, y, w, h);

        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSplitImage: Calling NXPutPackedImage(%s) size is [%d x %d] data size is [%d]\n",
                    (packMethod >= PACK_PNG_8_COLORS && packMethod <= PACK_PNG_16M_COLORS) ?
                        "PNG" : "JPEG", plainImage -> width, plainImage -> height, packedImage -> xoffset);
        #endif

        #ifdef NXAGENT_STOPBIGREQ

        if (needSplit)
        {
          nxEndBigReq(nxagentLastImgClient);
        }

        #endif

        XFree(plainImage);

        return;
      }
    }

    if (packMethod > 0 &&
            (packMethod > PACK_TIGHT_16M_COLORS ||
                 packMethod < PACK_TIGHT_8_COLORS))
    {
      if (packMethod >= PACK_JPEG_8_COLORS &&
              packMethod <= PACK_JPEG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSplitImage: encodingJpeg forced by command line parameter\n");
        #endif

        packMethod = PACK_JPEG_16M_COLORS;

        packedImage = NXEncodeJpeg(plainImage, packMethod, nxagentPackQuality);
      }
      else if (packMethod >= PACK_PNG_8_COLORS &&
                   packMethod <= PACK_PNG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSplitImage: encodingPng forced by command line parameter\n");
        #endif

        packMethod = PACK_PNG_16M_COLORS;

        packedImage = NXEncodePng(plainImage, packMethod , NULL);
      }
      else if (packMethod >= PACK_PNG_JPEG_8_COLORS &&
                   PACK_PNG_JPEG_16M_COLORS)
      {
        #ifdef DEBUG
        fprintf(stderr, "nxagentPutSplitImage: Trying to execute deperecated "
                    "NXDynamicSelectPackMethod(). Falling back to XPutImage.\n");
        #endif
      }
    }
  }

  if (packedImage != NULL &&
          (packMethod < PACK_TIGHT_8_COLORS ||
               packMethod > PACK_TIGHT_16M_COLORS))
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSplitImage: Calling NXPutPackedImage(%s) size is [%d x %d] data size is [%d]\n",
                (packMethod >= PACK_PNG_8_COLORS && packMethod <= PACK_PNG_16M_COLORS) ?
                     "PNG" : "JPEG", plainImage -> width, plainImage -> height, packedImage -> xoffset);
    #endif

    nxagentSetUnpackAlpha(pDrawable, plainImage, nxagentLastImgClient);

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxSleepByBigReq(nxagentLastImgClient);
    }

    #endif

    NXPutPackedImage(nxagentDisplay, nxagentLastImgClient -> index, nxagentDrawable(pDrawable),
                         nxagentGC(pGC), packedImage, packMethod, depth,
                             0, 0, x, y, w, h);

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxEndBigReq(nxagentLastImgClient);
    }

    #endif
  }
  else if (packMethod >= PACK_TIGHT_8_COLORS &&
               packMethod <= PACK_TIGHT_16M_COLORS)
  {
    NXEncodeTight(nxagentDisplay, nxagentDrawable(pDrawable), nxagentDefaultVisual(pDrawable->pScreen),
                      nxagentGC(pGC), plainImage, packMethod, 0, 0, x, y, w, h);

    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSplitImage: Calling NXPutPackedImage(TIGHT) size is [%d x %d] "
                "data size is [%d].\n", plainImage -> width, plainImage -> height,
                    packedImage -> xoffset);
    #endif

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxSleepByBigReq(nxagentLastImgClient);
    }

    #endif

    NXPutPackedImage(nxagentDisplay, nxagentLastImgClient -> index, nxagentDrawable(pDrawable),
                         nxagentGC(pGC), packedImage, packMethod, depth,
                             0, 0, x, y, w, h);

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxEndBigReq(nxagentLastImgClient);
    }

    #endif
  }
  else if (packedImage == NULL)
  {
    #ifdef DEBUG
    fprintf(stderr, "nxagentPutSplitImage: Calling XPutImage, size is [%d x %d] data size is [%d].\n",
                plainImage -> width, plainImage -> height, plainImage -> bytes_per_line *
                    plainImage -> height);
    #endif

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxSleepByBigReq(nxagentLastImgClient);
    }

    #endif

    /*
     * Passing the leftPad value in src_x doesn't work.
     *
     * XPutImage(nxagentDisplay, nxagentDrawable(pDrawable),
     *               nxagentGC(pGC), plainImage, leftPad, 0, x, y, w, h);
     */

    XPutImage(nxagentDisplay, nxagentDrawable(pDrawable),
                  nxagentGC(pGC), plainImage, 0, 0, x, y, w, h);

    #ifdef NXAGENT_STOPBIGREQ

    if (needSplit)
    {
      nxEndBigReq(nxagentLastImgClient);
    }

    #endif
   }

  if (packedImage)
  {
    if (nxImageCacheStat > 0 && md5 != NULL)
    {
      NXCacheAddImage(packedImage, packMethod, md5);
    }

    XFree(packedImage);
  }
  else
  {
    if (md5 != NULL)
    {
      XFree(md5);
    }
  }

  XFree(plainImage);
}

int nxagentGetPixelHint(DrawablePtr pDrawable, int x, int y, int w, int h,
                            unsigned int format, unsigned long planeMask,
                                unsigned long *pixel)
{
  /*
   * Given the hints, try a suitable pixel.
   */

  if (format == ZPixmap && planeMask == 0xffffffff)
  {
    if ((w == 9 && h == 7) || (w == 7 && h == 9))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice tooltip.\n");
      #endif

      *pixel = 0xe6e6e6;

      return 1;
    }
    else if ((w == 10 && h <= 10) || (h == 10 && w <= 10))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice close button.\n");
      #endif

      *pixel = 0xe6e6e6;

      return 1;
    }
    else if (w == 11 && h == 11)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice toolbar.\n");
      #endif

      *pixel = 0xe6e6e6;

      return 1;
    }
    else if (w == 16 && h == 16)
    {
      if (x <= 8)
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice menu entry.\n");
        #endif

        *pixel = 0xe6e6e6;
      }
      else
      {
        #ifdef TEST
        fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice open dialog.\n");
        #endif

        *pixel = 0xffffff;
      }

      return 1;
    }
    else if (w == 18 && h == 18)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice mouse over close button.\n");
      #endif

      *pixel = 0xb7bbd4;

      return 1;
    }
    else if ((w == 24 && h <= 24) || (h == 24 && w <= 24))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice toolbar.\n");
      #endif

      *pixel = 0xe6e6e6;

      return 1;
    }
    else if ((w == 31 && h <= 31) || (h == 31 && w <= 31))
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice mouse over toolbar.\n");
      #endif

      *pixel = 0xcccccc;

      return 1;
    }
    else if (w == 32 && h == 32)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetPixelHint: Trying background for OpenOffice icon.\n");
      #endif

      *pixel = 0xe6e6e6;

      return 1;
    }
  }

  #ifdef WARNING
  fprintf(stderr, "nxagentGetPixelHint: WARNING! No match found with drawable at [%p] geometry [%d, %d].\n",
              (void *) pDrawable, w, h);

  fprintf(stderr, "nxagentGetPixelHint: WARNING! Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  return 0;
}

void nxagentGetDefaultImage(DrawablePtr pDrawable, int x, int y, int w, int h,
                                unsigned int format, unsigned long planeMask,
                                    unsigned long *hint, char *data)
{
  unsigned long pixel;

  Visual *pVisual;
  XImage *image;

  #ifdef TEST
  fprintf(stderr, "nxagentGetDefaultImage: Called with drawable at [%p] geometry [%d, %d, %d, %d].\n",
              (void *) pDrawable, x, y, w, h);

  fprintf(stderr, "nxagentGetDefaultImage: Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  /*
   * Try to match a pixel according to known
   * sizes only if a hint is not provided.
   */

  if (hint)
  {
    pixel = *hint;
  }
  else if (!nxagentGetPixelHint(pDrawable, x, y, w, h, format, planeMask, &pixel))
  {
#if 0
GFPZR

    if (nxagentLastGetImage)
    {
      #ifdef WARNING
      fprintf(stderr, "nxagentGetDefaultImage: WARNING! Trying with data from last request.\n");
      #endif

      pixel = XGetPixel(nxagentLastGetImage, 0, 0);
    }
    else
    {
      /*
       * Trying to set pixel to the window's background
       * often results in a ugly black. Let's consider
       * preferable a solid grey.
       *
       * pixel = ((WindowPtr) pDrawable) -> background.pixel;
       */

      #ifdef WARNING
      fprintf(stderr, "nxagentGetDefaultImage: WARNING! Trying with default pixel.\n");
      #endif

      pixel = 0xe6e6e6;
    }
#endif

    pixel = 0xff0000;
  }

  pVisual = nxagentImageVisual(pDrawable, pDrawable -> depth);

  if (!(image = XCreateImage(nxagentDisplay, pVisual,
                             pDrawable -> depth, format, 0, data,
                             w, h, BitmapPad(nxagentDisplay),
                             nxagentImagePad(w, format, 0, pDrawable -> depth))))
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetDefaultImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  /*
   * Reuse x and y as indices. Note that image data
   * points to the buffer created in dispatcher.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetDefaultImage: Filling image with pixel [0x%lx].\n",
              pixel);
  #endif

  for (y = 0; y < h; y++)
  {
    for (x = 0; x < w; x++)
    {
      XPutPixel(image, x, y, pixel);
    }
  }

/*
GFPZR: Must be probably reformatted here.

  nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                           format == ZPixmap ? BitsPerPixel(image -> depth) : 1, 
                               image -> byte_order);
*/

  /*
   * Don't free the image data as buffer belongs
   * to the caller.
   */

  XFree(image);
}

#define TEST

#undef TEST

void nxagentGetImage(DrawablePtr pDrawable, int x, int y, int w, int h,
                         unsigned int format, unsigned long planeMask, char *data)
{
  unsigned long pixel;

  Visual *pVisual;
  XImage *image;

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: Called with drawable at [%p] geometry [%d, %d, %d, %d].\n",
              (void *) pDrawable, x, y, w, h);

  fprintf(stderr, "nxagentGetImage: Format is [%d] plane mask is [%lx].\n",
              format, planeMask);
  #endif

  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Going to get image from virtual pixmap at [%p].\n",
                (void *) nxagentVirtualDrawable(pDrawable));
    #endif

    fbGetImage(nxagentVirtualDrawable(pDrawable), x, y, w, h, format, planeMask, data);

    return;
  }

  /*
   * If fast get image is enabled try
   * to get a hint first.
   */

/*
GFPZR

  if (nxagentFastGetImageEnable)
*/
  {
    if (nxagentGetPixelHint(pDrawable, x, y, w, h, format, planeMask, &pixel))
    {
      nxagentGetDefaultImage(pDrawable, x, y, w, h, format, planeMask, &pixel, data);

      return;
    }
  }

#define TEST

  /*
   * If fast get image is disabled, grab the window's content
   * from the real X server. This is really clumsy. In theory
   * X clients should never do that (in fact if agent window
   * is covered it will not work). In practice a few programs
   * and libraries (like surprisingly a famous Linux office
   * automation suite) do, mainly to compose images with the
   * window's backgound. Do they really need this? Why don't
   * they combine content into a Pixmap?
   */

  if (!nxagentFastGetImageEnable || !nxagentLastGetImage)
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetImage: WARNING! Going to get image from window [%ld] at [%p].\n",
                nxagentWindow((WindowPtr) pDrawable), (void *) pDrawable);
    #endif

    /*
     * If fast get image is enabled but we don't have
     * a previous image to use as reference, we will
     * have to recur to an expensive round trip. Note
     * anyway that we will only collect one pixel.
     */

    if (nxagentFastGetImageEnable)
    {
      image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                            x, y, 1, 1, planeMask, format);
    }
    else
    {
      image = XGetImage(nxagentDisplay, nxagentDrawable(pDrawable),
                            x, y, w, h, planeMask, format);
    }

    if (!image)
    {
      #ifdef TEST
      fprintf(stderr, "nxagentGetImage: WARNING! Failed to get image. Returning undefined data.\n");
      #endif

      return;
    }

    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Got pixel [0x%lx] with geometry [%d, %d, %d, %d] "
                "format [%d] mask [%lx].\n", XGetPixel(image, 0, 0), x, y, w, h,
                    format, planeMask);
    #endif

    /*
     * Always save the image. It will be used
     * whenever the agent window is covered.
     */

    #ifdef TEST
    fprintf(stderr, "nxagentGetImage: Saving last image collected from real server.\n");
    #endif

    if (nxagentLastGetImage)
    {
      XDestroyImage(nxagentLastGetImage);
    }

    nxagentLastGetImage = image;

    if (!nxagentFastGetImageEnable)
    {
      nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                               format == ZPixmap ? BitsPerPixel(image -> depth) : 1,
                                   image -> byte_order);

      memmove(data, image -> data, image -> bytes_per_line * image -> height);

      return;
    }
  }

  /*
   * Fast get image is enabled. We either collected
   * the upper left pixel from the server or we are
   * going to use a previously collected image.
   */

  #ifdef WARNING
  fprintf(stderr, "nxagentGetImage: WARNING! Skipping get image from window [%ld] at [%p].\n",
              nxagentWindow((WindowPtr) pDrawable), (void *) pDrawable);
  #endif

  pixel = XGetPixel(nxagentLastGetImage, 0, 0);

  pVisual = nxagentImageVisual(pDrawable, pDrawable -> depth);

  if (!(image = XCreateImage(nxagentDisplay, pVisual,
                             pDrawable -> depth, format, 0, data,
                             w, h, BitmapPad(nxagentDisplay),
                             nxagentImagePad(w, format, 0, pDrawable -> depth))))
  {
    #ifdef WARNING
    fprintf(stderr, "nxagentGetImage: WARNING! Failed to create image.\n");
    #endif

    return;
  }

  /*
   * Guess the content of the new image by filling
   * data with the pixel value got from the last
   * image.
   */

  #ifdef WARNING
  fprintf(stderr, "nxagentGetImage: WARNING! Trying with data from last request.\n");
  #endif

  /*
   * Reuse x and y as indices. Note that image data
   * points to the buffer created in dispatcher.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentGetImage: Filling image with pixel [0x%lx].\n",
              pixel);
  #endif

  for (y = 0; y < h; y++)
  {
    for (x = 0; x < w; x++)
    {
      XPutPixel(image, x, y, pixel);
    }
  }

  nxagentImageReformat(image -> data, image -> bytes_per_line * image -> height, 
                           format == ZPixmap ? BitsPerPixel(image -> depth) : 1, 
                               image -> byte_order);

  /*
   * Don't free the image data as buffer belongs
   * to the caller.
   */

  XFree(image);

  /*
   * Finally initiate an aynchronous get image
   * request. A notification will be received
   * as soon as image data is available.
   */

  if (nxagentLastGetImageClient == UNDEFINED)
  {
    NXCollectImage(nxagentDisplay, requestingClient -> index,
                        nxagentDrawable(pDrawable), x, y, 1, 1,
                            planeMask, format);

    nxagentLastGetImageClient = requestingClient -> index;
  }
}

void nxagentCollectImageEvent(int resource)
{
  XImage *image;

  if (!NXGetCollectedImage(nxagentDisplay, resource, &image))
  {
    #ifdef PANIC
    fprintf(stderr, "nxagentCollectImageEvent: PANIC! Failed to get collected image for resource [%d].\n",
                resource);
    #endif

    return;
  }

  nxagentLastGetImageClient = UNDEFINED;

  if (nxagentLastGetImage)
  {
    XDestroyImage(nxagentLastGetImage);
  }

  nxagentLastGetImage = image;
}

#undef TEST

#ifndef NXAGENT_NOEXPOSEOPTIMIZE

static Bool nxagentBitBlitPredicate(display, event, args)
  Display *display;
  XEvent *event;
  char *args;
{
  return (event->type == GraphicsExpose || event->type == NoExpose);
}

#endif

RegionPtr nxagentBitBlitHelper(pGC)
  GC *pGC;
{
  #ifndef NXAGENT_NOEXPOSEOPTIMIZE

  XEvent event;
  RegionPtr pReg, pTmpReg;
  BoxRec Box;
  Bool pending, overlap;

  #endif

  #ifdef TEST
  fprintf(stderr, "nxagentBitBlitHelper: Called for GC at [%p].\n", (void *) pGC);
  #endif

  #ifdef NXAGENT_NOEXPOSEOPTIMIZE

  /*
   * Force NullRegion. We consider enough the graphics
   * expose events generated internally by the nxagent
   * server.
   */

  #ifdef TEST
  fprintf(stderr, "nxagentBitBlitHelper: WARNING! Skipping check on exposures events.\n");
  #endif

  return NullRegion;

  #else

  if (nxagentGCTrap)
  {
    return NullRegion;
  }

  if (!pGC -> graphicsExposures)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentBitBlitHelper: Returning NullRegion with GraphicExposures not selected.\n");
    #endif

    return NullRegion;
  }

  pReg = REGION_CREATE(pGC->pScreen, NULL, 1);
  pTmpReg = REGION_CREATE(pGC->pScreen, NULL, 1);

  if(!pReg || !pTmpReg) return NullRegion;

  pending = True;

  while (pending)
  {
    XIfEvent(nxagentDisplay, &event, nxagentBitBlitPredicate, NULL);

    switch (event.type)
    {
      case NoExpose:
      {
        pending = False;

        #ifdef TEST
        fprintf(stderr, "nxagentBitBlitHelper: Set pending to [%d] because of NoExpose.\n", pending);
        #endif

        break;
      }
      case GraphicsExpose:
      {
        Box.x1 = event.xgraphicsexpose.x;
        Box.y1 = event.xgraphicsexpose.y;
        Box.x2 = event.xgraphicsexpose.x + event.xgraphicsexpose.width;
        Box.y2 = event.xgraphicsexpose.y + event.xgraphicsexpose.height;

        REGION_RESET(pGC->pScreen, pTmpReg, &Box);
        REGION_APPEND(pGC->pScreen, pReg, pTmpReg);

        pending = event.xgraphicsexpose.count;

        #ifdef TEST
        fprintf(stderr, "nxagentBitBlitHelper: Set pending to [%d].\n", pending);
        #endif

        break;
      }
    }
  }

  REGION_DESTROY(pGC->pScreen, pTmpReg);
  REGION_VALIDATE(pGC->pScreen, pReg, &overlap);

  return(pReg);

  #endif
}

RegionPtr nxagentCopyArea(pSrcDrawable, pDstDrawable,
                              pGC, srcx, srcy, width, height, dstx, dsty)
  DrawablePtr pSrcDrawable;
  DrawablePtr pDstDrawable;
  GC *pGC;
  int srcx, srcy;
  int width, height;
  int dstx, dsty;
  
{
  XImage *temporary_ximage = NULL;
  PixmapPtr temporary_pixmap = NULL;
  unsigned long planeMask = 0xffffffff;
  int  leftPad = 0;
  unsigned int  format;

  #ifdef NXAGENT_FASTCOPYAREA_DEBUG
  static int counter = 1;
  #endif

  if (nxagentGCTrap)
  {
    if ((pSrcDrawable) -> type == DRAWABLE_PIXMAP &&
            (pDstDrawable) -> type == DRAWABLE_PIXMAP)
    {
      fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), nxagentVirtualDrawable(pDstDrawable),
                     pGC, srcx, srcy, width, height, dstx, dsty);
    }

    return nxagentBitBlitHelper(pGC); 
  }

  XCopyArea(nxagentDisplay, nxagentDrawable(pSrcDrawable), nxagentDrawable(pDstDrawable),
                nxagentGC(pGC), srcx, srcy, width, height, dstx, dsty);

  #ifdef TEST
  fprintf(stderr, "nxagentCopyArea: Going to copy area with GC [%lx] from [%lx] to [%lx]\n",
              (unsigned long) pGC, (unsigned long) pSrcDrawable, (unsigned long) pDstDrawable);
  #endif

  if ((pSrcDrawable)->type == DRAWABLE_PIXMAP &&
          (pDstDrawable)->type == DRAWABLE_PIXMAP)
  {
    #ifdef TEST
    fprintf(stderr, "nxagentCopyArea: Going to copy area from virtual pixmap [%lx] to [%lx]\n",
                (unsigned long) nxagentVirtualDrawable(pSrcDrawable),
                    (unsigned long) nxagentVirtualDrawable(pDstDrawable));
    #endif

    fbCopyArea(nxagentVirtualDrawable(pSrcDrawable), nxagentVirtualDrawable(pDstDrawable),
                   pGC, srcx, srcy, width, height, dstx, dsty);
  }
  else if ((pSrcDrawable) -> type != DRAWABLE_PIXMAP &&
               (((WindowPtr) (pSrcDrawable)) -> viewable) &&
                   (pDstDrawable) -> type == DRAWABLE_PIXMAP)
  {
    if (nxagentFastCopyAreaEnable == False)
    {
      if (pSrcDrawable -> depth == 1)
      {
        format = XYPixmap;
      }
      else
      {
        format = ZPixmap;
      }

      #ifdef NXAGENT_FASTCOPYAREA_DEBUG
      fprintf(stderr, "GCOps: we are before calculate dimensions, srcx='%d', srcy='%d', width='%d', height='%d'\n",
                  srcx, srcy, width, height);
      #endif

      /*
       * Get rid of the part of image out of the
       * nxagent screen.
       */

      if (srcx < 0) srcx = 0;
      if (srcy < 0) srcy = 0;

      if (srcx > nxagentOption(Width))  srcx = nxagentOption(Width);
      if (srcy > nxagentOption(Height)) srcy = nxagentOption(Height);

      if (srcx + width >  nxagentOption(Width))  width  = nxagentOption(Width)  - srcx;
      if (srcy + height > nxagentOption(Height)) height = nxagentOption(Height) - srcy;

      temporary_ximage = XGetImage(nxagentDisplay, nxagentDrawable(pSrcDrawable),
                                       srcx, srcy, width, height, planeMask, format);

      #ifdef NXAGENT_FASTCOPYAREA_DEBUG
      fprintf(stderr, "GCOps: we are after calculate dimensions, srcx='%d', srcy='%d', width='%d', height='%d'\n",
                  srcx, srcy, width, height);

      fprintf(stderr, "GCOps: we have XGetImage(), counter: '%d'\n", counter++);
      #endif

      if (temporary_ximage)
      {
        temporary_pixmap = fbCreatePixmap(pSrcDrawable->pScreen, width, height, pSrcDrawable->depth);

        fbPutImage((DrawablePtr) temporary_pixmap, pGC, pSrcDrawable->depth, 0, 0, width, height,
                       leftPad, format, temporary_ximage->data);

        fbCopyArea((DrawablePtr) temporary_pixmap, nxagentVirtualDrawable(pDstDrawable),
                       pGC, 0, 0, width, height, dstx, dsty);

        XDestroyImage(temporary_ximage);

        fbDestroyPixmap(temporary_pixmap);
      }
    }
    else
    {
      /*
       * When fast mode is enabled we ignore the window
       * content, just clean the drawable and copy it
       * to the pixmap in framebuffer.
       */

      GCPtr pGCtmp;
      xRectangle rect;
      CARD32 attributes[3];
      short int nrect = 1;

      rect.x = 0;
      rect.y = 0;
      rect.width = width;
      rect.height = height;

      pGCtmp = GetScratchGC(pSrcDrawable -> depth, pSrcDrawable -> pScreen);

      /* Attributes for GCForeground */
      attributes[0] = 0xffffff;
      /* Attributes for GCBackground */
      attributes[1] = 0xffffff;
      /* Attributes for GCFillStyle*/
      attributes[2] = FillSolid;

      ChangeGC(pGCtmp, GCForeground | GCBackground | GCFillStyle , attributes);

      temporary_pixmap = fbCreatePixmap(pSrcDrawable -> pScreen, width, height,
                                            pSrcDrawable -> depth);

      ValidateGC((DrawablePtr)pDstDrawable, pGCtmp);

      fbPolyFillRect((DrawablePtr) temporary_pixmap, pGCtmp, nrect, &rect);

      fbCopyArea((DrawablePtr) temporary_pixmap, nxagentVirtualDrawable(pDstDrawable),
                     pGC, 0, 0, width, height, dstx, dsty);

      fbDestroyPixmap(temporary_pixmap);

      FreeScratchGC(pGCtmp);
    }

    #ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: going to copy area from window [%lx] to mvfb pixmap [%lx]\n",
            (unsigned long)pSrcDrawable, (unsigned long)nxagentVirtualDrawable(pDstDrawable));
    #endif
  }

  #ifdef NXAGENT_MVFB_DEBUG
  if ((pSrcDrawable)->type != DRAWABLE_PIXMAP || (pDstDrawable)->type != DRAWABLE_PIXMAP)
  {
    fprintf(stderr, "GCOps: impossible to copy area in mfvb src is type [%s], dst is type [%s]\n",
                ((pSrcDrawable)->type == DRAWABLE_PIXMAP) ? "PIXMAP" : "WINDOW",
                    ((pDstDrawable)->type == DRAWABLE_PIXMAP) ? "PIXMAP" : "WINDOW");
  }
  #endif

  return nxagentBitBlitHelper(pGC);
}

RegionPtr nxagentCopyPlane(pSrcDrawable, pDstDrawable,
                         pGC, srcx, srcy, width, height, dstx, dsty, plane)
  DrawablePtr pSrcDrawable;
  DrawablePtr pDstDrawable;
  GC *pGC;
  int srcx, srcy;
  int width, height;
  int dstx, dsty;
  unsigned long plane;
{
  if (nxagentGCTrap)
  {
    if ((pSrcDrawable)->type == DRAWABLE_PIXMAP &&
            (pDstDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbCopyPlane(nxagentVirtualDrawable(pSrcDrawable), nxagentVirtualDrawable(pDstDrawable),
                  pGC, srcx, srcy, width, height, dstx, dsty, plane);
    }

    return nxagentBitBlitHelper(pGC);
  }

  XCopyPlane(nxagentDisplay,
             nxagentDrawable(pSrcDrawable), nxagentDrawable(pDstDrawable),
             nxagentGC(pGC), srcx, srcy, width, height, dstx, dsty, plane);

#ifdef NXAGENT_MVFB
 if ((pSrcDrawable)->type == DRAWABLE_PIXMAP && (pDstDrawable)->type == DRAWABLE_PIXMAP)
 {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: going to copy plane from mvfb pixmap = [%lx] to [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pSrcDrawable), (unsigned long)nxagentVirtualDrawable(pDstDrawable));
#endif
    fbCopyPlane(nxagentVirtualDrawable(pSrcDrawable), nxagentVirtualDrawable(pDstDrawable),
                pGC, srcx, srcy, width, height, dstx, dsty, plane);
 }
#endif

  return nxagentBitBlitHelper(pGC);
}

void nxagentPolyPoint(pDrawable, pGC, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyPoint(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    }

    return;
  }


#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to poly point on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
      fprintf(stderr, "GCOps: poly point enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
           XDrawPoints(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
              (XPoint *)pPoints, nPoints, mode);
    }
    else
    {
       XDrawPoints(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                   (XPoint *)pPoints, nPoints, mode);
    }

    fbPolyPoint(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);

    return;
  }
  else
#endif
    XDrawPoints(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              (XPoint *)pPoints, nPoints, mode);
}

void nxagentPolylines(pDrawable, pGC, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyLine(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly line on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly lines enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawLines(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
             (XPoint *)pPoints, nPoints, mode);
    }
    else
    {
      XDrawLines(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XPoint *)pPoints, nPoints, mode);
    }
    fbPolyLine(nxagentVirtualDrawable(pDrawable), pGC, mode, nPoints, pPoints);
    return;
  }
  else
#endif
    XDrawLines(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
               (XPoint *)pPoints, nPoints, mode);
}

void nxagentPolySegment(pDrawable, pGC, nSegments, pSegments)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nSegments;
  xSegment *pSegments;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolySegment(nxagentVirtualDrawable(pDrawable), pGC, nSegments, pSegments);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly segment on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly segment enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawSegments(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                (XSegment *)pSegments, nSegments);
    }
    else
    {
      XDrawSegments(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XSegment *)pSegments, nSegments);
    }
    miPolySegment(nxagentVirtualDrawable(pDrawable), pGC, nSegments, pSegments);
    return;
  }
  else
#endif
    XDrawSegments(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XSegment *)pSegments, nSegments);
}

void nxagentPolyRectangle(pDrawable, pGC, nRectangles, pRectangles)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nRectangles;
  xRectangle *pRectangles;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyRectangle(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly rectangle on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly rectangle enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
        XDrawRectangles(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                        (XRectangle *)pRectangles, nRectangles);
    }
    else
    {
      XDrawRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                      (XRectangle *)pRectangles, nRectangles);
    }
    miPolyRectangle(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    return;
  }
  else
#endif
    XDrawRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XRectangle *)pRectangles, nRectangles);
}

void nxagentPolyArc(pDrawable, pGC, nArcs, pArcs)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nArcs;
  xArc *pArcs;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly arc on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly arc enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
          XDrawArcs(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
            (XArc *)pArcs, nArcs);
    }
    else
    {
       XDrawArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XArc *)pArcs, nArcs);
    }
    fbPolyArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    return;
  }
  else
#endif
    XDrawArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              (XArc *)pArcs, nArcs);
}

void nxagentFillPolygon(pDrawable, pGC, shape, mode, nPoints, pPoints)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int shape;
  int mode;
  int nPoints;
  xPoint *pPoints;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miFillPolygon(nxagentVirtualDrawable(pDrawable), pGC, shape, mode, nPoints, pPoints);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to fill polygon on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: fill polygon enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
          XFillPolygon(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                       (XPoint *)pPoints, nPoints, shape, mode);
    }
    else
    {
      XFillPolygon(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                   (XPoint *)pPoints, nPoints, shape, mode);

    }
    miFillPolygon(nxagentVirtualDrawable(pDrawable), pGC, shape, mode, nPoints, pPoints);
    return;
  }
  else
#endif
    XFillPolygon(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                 (XPoint *)pPoints, nPoints, shape, mode);
}

void nxagentPolyFillRect(pDrawable, pGC, nRectangles, pRectangles)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nRectangles;
  xRectangle *pRectangles;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      fbPolyFillRect(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly fill rect on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentPixmapIsVirtual((PixmapPtr) pDrawable))
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly fill rect enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr) pDrawable))
      {
         XFillRectangles(nxagentDisplay, nxagentDrawable((DrawablePtr) nxagentRealPixmap((PixmapPtr) pDrawable)),
                             nxagentGC(pGC), (XRectangle *) pRectangles, nRectangles);
      }
    }
    else
    {
      XFillRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                      (XRectangle *)pRectangles, nRectangles);
    }

    fbPolyFillRect(nxagentVirtualDrawable(pDrawable), pGC, nRectangles, pRectangles);

    return;
  }
  else
#endif
    XFillRectangles(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    (XRectangle *)pRectangles, nRectangles);
}

void nxagentPolyFillArc(pDrawable, pGC, nArcs, pArcs)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int nArcs;
  xArc *pArcs;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyFillArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly fillarc on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly fill arc enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XFillArcs(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                   (XArc *)pArcs, nArcs);
    }
    else
    {
      XFillArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                (XArc *)pArcs, nArcs);
    }
    miPolyFillArc(nxagentVirtualDrawable(pDrawable), pGC, nArcs, pArcs);
    return;
  }
  else
#endif
    XFillArcs(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
            (XArc *)pArcs, nArcs);
}

int nxagentPolyText8(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  char *string;
{
  int width;

  width = XTextWidth(nxagentFontStruct(pGC->font), string, count);

  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return width + x;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly text8 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly text8 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawString(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                     x, y, string, count);
    }
    else
    {
      XDrawString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                  x, y, string, count);
    }
    miPolyText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return width + x;
  }
  else
#endif
    XDrawString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
              x, y, string, count);

  return width + x;
}

int nxagentPolyText16(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  unsigned short *string;
{
  int width;

  width = XTextWidth16(nxagentFontStruct(pGC->font), (XChar2b *)string, count);

  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miPolyText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return width + x;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to poly text16 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly text16 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawString16(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                     x, y, (XChar2b *)string, count);
    }
    else
    {
      XDrawString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                    x, y, (XChar2b *)string, count);
    }
    miPolyText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return width + x;
  }
  else
#endif
    XDrawString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                  x, y, (XChar2b *)string, count);

  return width + x;
}

void nxagentImageText8(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  char *string;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miImageText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to image text8 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {

#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly image text8 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawImageString(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                          x, y, string, count);
    }
    else
    {
      XDrawImageString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                       x, y, string, count);
    }
    miImageText8(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return;
  }
  else
#endif
    XDrawImageString(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                     x, y, string, count);
}

void nxagentImageText16(pDrawable, pGC, x, y, count, string)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int count;
  unsigned short *string;
{
  if (nxagentGCTrap)
  {
    if ((pDrawable)->type == DRAWABLE_PIXMAP)
    {
      miImageText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    }

    return;
  }

#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: GC [%lx] going to image text16 on mvfb pixmap = [%lx] s = [%s]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable), string);
#endif
    if (nxagentVirtualDrawable(pDrawable) == pDrawable)
    {
#ifdef NXAGENT_MVFB_DEBUG
       fprintf(stderr, "GCOps: poly image text16 enters with virtual pixmap = [%lx] parent is = [%lx]\n",
                       (unsigned long)nxagentVirtualDrawable(pDrawable),
                       (unsigned long)nxagentRealPixmap((PixmapPtr)pDrawable));
#endif
      if (nxagentRealPixmap((PixmapPtr)pDrawable))
         XDrawImageString16(nxagentDisplay, nxagentDrawable((DrawablePtr)nxagentRealPixmap((PixmapPtr)pDrawable)), nxagentGC(pGC),
                          x, y, (XChar2b *)string, count);
    }
    else
    {
      XDrawImageString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                         x, y, (XChar2b *)string, count);
    }
    miImageText16(nxagentVirtualDrawable(pDrawable), pGC, x, y, count, string);
    return;
  }
  else
#endif
    XDrawImageString16(nxagentDisplay, nxagentDrawable(pDrawable), nxagentGC(pGC),
                     x, y, (XChar2b *)string, count);
}

void nxagentImageGlyphBlt(pDrawable, pGC, x, y, nGlyphs, pCharInfo, pGlyphBase)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int nGlyphs;
  CharInfoPtr *pCharInfo;
  pointer pGlyphBase;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to imageGlyphBlt on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbImageGlyphBlt(nxagentVirtualDrawable(pDrawable), pGC, x, y, nGlyphs, pCharInfo, pGlyphBase);
  }
#else
  ErrorF("nxagent warning: function nxagentImageGlyphBlt not implemented\n");
#endif
}

void nxagentPolyGlyphBlt(pDrawable, pGC, x, y, nGlyphs, pCharInfo, pGlyphBase)
  DrawablePtr pDrawable;
  GCPtr pGC;
  int x, y;
  int nGlyphs;
  CharInfoPtr *pCharInfo;
  pointer pGlyphBase;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to PolyGlyphBlt on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                       (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif
    fbPolyGlyphBlt(nxagentVirtualDrawable(pDrawable), pGC, x, y, nGlyphs, pCharInfo, pGlyphBase);
  }
#else
  ErrorF("nxagent warning: function nxagentPolyGlyphBlt not implemented\n");
#endif
}

void nxagentPushPixels(pGC, pBitmap, pDrawable, width, height, x, y)
  GCPtr pGC;
  PixmapPtr pBitmap;
  DrawablePtr pDrawable;
  int width, height;
  int x, y;
{
#ifdef NXAGENT_MVFB
  if ((pDrawable)->type == DRAWABLE_PIXMAP)
  {
#ifdef NXAGENT_MVFB_DEBUG
    fprintf(stderr, "GCOps: GC [%lx] going to PushPixels on mvfb pixmap = [%lx]\n", (unsigned long)pGC,
                         (unsigned long)nxagentVirtualDrawable(pDrawable));
#endif

    fbPushPixels(pGC, nxagentVirtualPixmap(pBitmap), (DrawablePtr)nxagentVirtualDrawable(pDrawable), width, height, x, y);
  }
#else
  ErrorF("nxagent warning: function nxagentPushPixels not implemented\n");
#endif
}

/*
 * We have to reset the visual cache before
 * connecting to another display, so that a
 * new unpack geometry can be communicated
 * to the new proxy.
 */

void nxagentResetVisualCache()
{
  int i;

  for (i = 0; i < MAX_CONNECTIONS; i++)
  {
    memset(&nxagentUnpackVisual[i], 0, sizeof(Visual));
  }
}

void nxagentResetAlphaCache()
{
  int i;

  for (i = 0; i < MAX_CONNECTIONS; i++)
  {
    if (nxagentUnpackAlpha[i])
    {
      xfree(nxagentUnpackAlpha[i] -> data);

      xfree(nxagentUnpackAlpha[i]);

      nxagentUnpackAlpha[i] = NULL;
    }
  }
}


