/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <sys/types.h>
#include <sys/socket.h>

#include "GenericChannel.h"

#include "EncodeBuffer.h"
#include "DecodeBuffer.h"

#include "Compressor.h"
#include "Decompressor.h"

#include "Statistics.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  TEST
#undef  DEBUG

//
// Define this to log when a channel
// is created or destroyed.
//

#undef  REFERENCES

//
// Here are the static members.
//

#ifdef REFERENCES

int GenericChannel::references_ = 0;

#endif

GenericChannel::GenericChannel(Transport *transport, Compressor *compressor,
                               Decompressor *decompressor)

  : Channel(transport, compressor, decompressor),
        readBuffer_(transport_, this)
{
  #ifdef REFERENCES
  *logofs << "GenericChannel: Created new object at " 
          << this << " for FD#" << fd_ << " out of " 
          << ++references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

GenericChannel::~GenericChannel()
{
  #ifdef REFERENCES
  *logofs << "GenericChannel: Deleted object at " 
          << this << " for FD#" << fd_ << " out of "
          << --references_ << " allocated channels.\n"
          << logofs_flush;
  #endif
}

//
// Beginning of handleRead().
//

int GenericChannel::handleRead(EncodeBuffer& encodeBuffer)
{
  #if defined(INFO) || defined(TEST)

  if (finish_ == 1)
  {
    #ifdef PANIC
    *logofs << "handleRead: PANIC! Shouldn't be called to handle "
            << "finishing descriptor FD#" << fd_ << ".\n"
            << logofs_flush;
    #endif

     cerr << "Error" << ": Shouldn't be called to handle "
          << "finishing descriptor FD#" << fd_ << ".\n";

     return -1;
  }

  #endif

  #ifdef TEST
  *logofs << "handleRead: Called for FD#" << fd_
          << ".\n" << logofs_flush;
  #endif

  //
  // Pointer to located message and its
  // size in bytes.
  //

  const unsigned char *inputMessage;
  unsigned int inputLength;

  //
  // Tag message as generic data in compression
  // routine. Opcode is not actually transferred
  // over the network.
  //

  unsigned char inputOpcode = X_NXInternalGenericData;

  //
  // Keep handling payload bytes until
  // it's time to yield.
  //

  int count = 0;
  int yield = 0;
  int loops = 0;
  int reads = 0;

  while (yield == 0)
  {
    #ifdef TEST

    *logofs << "handleRead: Initial time spent handling messages is "
            << control -> getTimeInARow() << " Ms.\n"
            << logofs_flush;

    *logofs << "handleRead: Initial bytes to write on proxy link are "
            << control -> getBytesInARow() << " bytes.\n"
            << logofs_flush;

    #endif

    //
    // Before reading from socket handle any
    // data left in the read buffer.
    //

    if (pending_ == 0)
    {
      if (readBuffer_.readMessage() < 0)
      {
        return -1;
      }
    }

    reads++;

    #if defined(INFO) || defined(TEST)
    *logofs << "handleRead: Getting messages from FD#" << fd_ << " with "
            << readBuffer_.getLength() << " bytes in the read buffer.\n"
            << logofs_flush;
    #endif

    //
    // Divide available data in multiple
    // messages and encode them.
    //

    while (yield == 0 && (inputMessage = readBuffer_.getMessage(inputLength)) != 0)
    {
      loops++;

      count += inputLength;

      encodeBuffer.encodeValue(inputLength, 32, 14);

      if (isCompressed() == 1)
      {
        unsigned int compressedDataSize = 0;
        unsigned char *compressedData   = NULL;

        if (handleCompress(encodeBuffer, inputOpcode, inputMessage,
                               inputLength, 0, compressedData,
                                   compressedDataSize) < 0)
        {
          return -1;
        }
      }
      else
      {
        encodeBuffer.encodeMemory(inputMessage, inputLength);
      }

      int bits = encodeBuffer.getBits();

      #if defined(TEST) || defined(OPCODES)
      *logofs << "handleRead: Handled generic data for FD#" << fd_
              << ". " << inputLength << " bytes in, " << bits << " bits ("
              << ((float) bits) / 8 << " bytes) out.\n" << logofs_flush;
      #endif

      if (control -> CollectStatistics)
      {
        addProtocolBits(inputLength << 3, bits);
      }

      control -> addOutputInARow(inputLength);

      control -> addBitsInARow(bits);

      if (isPrioritized() == 1)
      {
        priority_++;
      }
      else if (isFlushable() == 1)
      {
        flush_++;
      }

      //
      // If we consumed too many ticks or produced
      // enough data, then leave remaining bytes
      // in the read buffer.
      //

      if (pending_ == 0 &&
              control -> isTimeToYield(yield_in_channel))
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleRead: Giving up because of "
                << control -> getBytesInARow() << " bytes to write after "
                << control -> getTimeInARow() << " Ms for FD#" << fd_
                << ".\n" << logofs_flush;
        #endif

        yield = 1;
      }
    }

    //
    // Set pending flag if we quit the loop before
    // consuming all messages in the read buffer.
    //

    if (yield == 1)
    {
      pending_ = readBuffer_.checkMessage();
    }
    else
    {
      //
      // Give up if we just needed to handle
      // messages left in the read buffer
      //

      if (pending_ > 0)
      {
        pending_ = 0;

        #if defined(INFO) || defined(TEST)
        *logofs << "handleRead: Giving up because of no more "
                << "pending messages for FD#" << fd_
                << ".\n" << logofs_flush;
        #endif

        yield = 1;
      }

      //
      // Check if we read enough data.
      //

      else if (count >= control -> GenericMaximumReadSize)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleRead: Giving up because of limit of "
                << control -> GenericMaximumReadSize
                << " generic data bytes read for FD#"
                << fd_ << ".\n" << logofs_flush;
        #endif

        yield = 1;
      }

      //
      // If it's not time to yield, check if
      // we can read more bytes from socket.
      //

      else if (transport_ -> readable() <= 0)
      {
        #if defined(INFO) || defined(TEST)
        *logofs << "handleRead: Giving up with no bytes "
                << "available on generic FD#" << fd_
                << ".\n" << logofs_flush;
        #endif

        yield = 1;
      }

      #if defined(INFO) || defined(TEST)

      if (yield == 1)
      {
        *logofs << "handleRead: Performed " << reads << " reads "
                << "and handled " << loops << " messages producing "
                << control -> getBytesInARow() << " bytes in "
                << control -> getTimeInARow() << " Ms for FD#"
                << fd_ << ".\n" << logofs_flush;
      }
      else
      {
        *logofs << "handleRead: Still reading messages with "
                << transport_ -> readable() << " bytes available "
                << "for FD#" << fd_ << ".\n" << logofs_flush;
      }

      #endif

    } // End of if (yield == 1) ... else ...

  } // End of while (yield == 0) ...

  //
  // Mark the end of encode buffer.
  //

  encodeBuffer.encodeValue(0, 32, 14);

  //
  // Reset the read buffer.
  //

  readBuffer_.partialReset();

  return 1;
}

//
// End of handleRead().
//

//
// Beginning of handleWrite().
//

int GenericChannel::handleWrite(const unsigned char *message, unsigned int length)
{
  #ifdef TEST
  *logofs << "handleWrite: Called for FD#" << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Create the buffer from which to 
  // decode messages.
  //

  DecodeBuffer decodeBuffer(message, length);

  #if defined(INFO) || defined(TEST)
  *logofs << "handleWrite: Cumulative length of buffer is " << length
          << " bytes.\n" << logofs_flush;
  #endif

  unsigned char *outputMessage;
  unsigned int outputLength;

  //
  // Tag message as generic data
  // in decompression.
  //

  unsigned char outputOpcode = X_NXInternalGenericData;

  for (;;)
  {
    decodeBuffer.decodeValue(outputLength, 32, 14);

    if (outputLength == 0)
    {
      break;
    }

    if (isCompressed() == 1)
    {
      if (writeBuffer_.getAvailable() < outputLength ||
              (int) outputLength >= control -> TransportWriteThreshold)
      {
        #ifdef DEBUG
        *logofs << "handleWrite: Using scratch buffer for "
                << "generic data with size " << outputLength << " and "
                << writeBuffer_.getLength() << " bytes in buffer.\n"
                << logofs_flush;
        #endif

        outputMessage = writeBuffer_.addScratchMessage(outputLength);
      }
      else
      {
        outputMessage = writeBuffer_.addMessage(outputLength);
      }

      const unsigned char *compressedData = NULL;
      unsigned int compressedDataSize = 0;

      int decompressed = handleDecompress(decodeBuffer, outputOpcode, outputMessage,
                                              outputLength, 0, compressedData,
                                                  compressedDataSize);
      if (decompressed < 0)
      {
        return -1;
      }
    }
    else
    {
      #ifdef DEBUG
      *logofs << "handleWrite: Using scratch buffer for "
              << "generic data with size " << outputLength << " and "
              << writeBuffer_.getLength() << " bytes in buffer.\n"
              << logofs_flush;
      #endif

      writeBuffer_.addScratchMessage((unsigned char *)
                       decodeBuffer.decodeMemory(outputLength), outputLength);
    }

    #if defined(TEST) || defined(OPCODES)
    *logofs << "handleWrite: Handled generic data for FD#" << fd_
            << ". "  << outputLength << " bytes out.\n"
            << logofs_flush;
    #endif

    handleFlush(flush_if_needed);
  }

  //
  // Write any remaining data to socket.
  //

  if (handleFlush(flush_if_any) < 0)
  {
    return -1;
  }

  return 1;
}

//
// End of handleWrite().
//

//
// Other members.
//

int GenericChannel::handleSplit(EncodeBuffer &encodeBuffer, MessageStore *store, 
                                    const unsigned char *buffer, const unsigned int size)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleSplit(EncodeBuffer &encodeBuffer, int packetLimit)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleUnsplit(DecodeBuffer &decodeBuffer, MessageStore *store,
                                      unsigned char *buffer, const unsigned int size)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleUnsplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleUnsplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleUnsplit(DecodeBuffer &decodeBuffer)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleUnsplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleUnsplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleAbortSplit(EncodeBuffer &encodeBuffer)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleAbortSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleAbortSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleAbortSplit(DecodeBuffer &decodeBuffer)
{
  #ifdef PANIC
  *logofs << "GenericChannel: PANIC! Function handleAbortSplit() "
          << "shouldn't be called.\n" << logofs_flush;
  #endif

  cerr << "Error" << ": Function handleAbortSplit() "
       << "shouldn't be called.\n";

  HandleCleanup();

  return -1;
}

int GenericChannel::handleConfiguration()
{
  #ifdef TEST
  *logofs << "GenericChannel: Setting new buffer parameters.\n"
          << logofs_flush;
  #endif

  readBuffer_.setSize(control -> GenericInitialReadSize,
                          control -> GenericMaximumReadSize);

  writeBuffer_.setSize(control -> TransportGenericBufferSize,
                           control -> TransportGenericBufferThreshold,
                               control -> TransportMaximumBufferSize);

  transport_ -> setSize(control -> TransportGenericBufferSize,
                            control -> TransportGenericBufferThreshold,
                                control -> TransportMaximumBufferSize);

  return 1;
}

int GenericChannel::handleFinish()
{
  #ifdef TEST
  *logofs << "GenericChannel: Finishing channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  pending_ = 0;
  finish_  = 1;

  transport_ -> fullReset();

  return 1;
}

int GenericChannel::handleReset()
{
  #ifdef TEST
  *logofs << "GenericChannel: Resetting generic channel for FD#"
          << fd_ << ".\n" << logofs_flush;
  #endif

  //
  // Need to shutdown the connection or otherwise
  // we could cause corruption of data carried by
  // the protocol. Only one end must shutdown the
  // link and I don't want to specialize generic
  // channel to have a client and server class.

  if (control -> ProxyMode == PROXY_SERVER)
  {
    shutdown(fd_, SHUT_RDWR);
  }

  return 1;
}
