/**************************************************************************/
/*                                                                        */
/* Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.           */
/*                                                                        */
/* NXPROXY, NX protocol compression and NX extensions to this software    */
/* are copyright of NoMachine. Redistribution and use of the present      */
/* software is allowed according to terms specified in the file LICENSE   */
/* which comes in the source distribution.                                */
/*                                                                        */
/* Check http://www.nomachine.com/licensing.html for applicability.       */
/*                                                                        */
/* NX and NoMachine are trademarks of Medialogic S.p.A.                   */
/*                                                                        */
/* All rights reserved.                                                   */
/*                                                                        */
/**************************************************************************/

#include <iostream.h>
#include <stdio.h>
#include <ctype.h>
#include <stdlib.h>
#include <unistd.h>

#include <errno.h>
#include <string.h>

#include "NXproto.h"

#include "MD5.h"

#include "Misc.h"

//
// Set the verbosity level.
//

#define PANIC
#define WARNING
#undef  OPCODES
#undef  TEST
#undef  DEBUG
#undef  DUMP

//
// TCP port offset applied to any NX port specification.
//

const int DEFAULT_NX_PROXY_PORT_OFFSET = 4000;

//
// Default TCP port used by client proxy to listen to  
// X clients and by server proxy to connect to remote.
//

const int DEFAULT_NX_PROXY_PORT = 8;

//
// Default X display number that client proxy imitates.
//

const int DEFAULT_NX_X_PORT = 8;

//
// Sync, samba and multimedia and embedded keyboard
// tunneling.
//

const int DEFAULT_NX_SYNC_PORT_OFFSET  = 5000;
const int DEFAULT_NX_KEYBD_PORT_OFFSET = 6000;
const int DEFAULT_NX_SAMBA_PORT_OFFSET = 7000;
const int DEFAULT_NX_MEDIA_PORT_OFFSET = 8000;

//
// Listen on these ports by default.
//

const int DEFAULT_NX_SYNC_PORT  = 8;
const int DEFAULT_NX_KEYBD_PORT = 8;
const int DEFAULT_NX_SAMBA_PORT = 8;
const int DEFAULT_NX_MEDIA_PORT = 8;

//
// Connect services to these ports by default.
// Samba could be always 139 but it's better
// to let user decide if he wants to allow
// remote server's access to his Samba setup.
//

const int DEFAULT_SYNC_SERVER_PORT  = DEFAULT_NX_SYNC_PORT_OFFSET +
                                          DEFAULT_NX_SYNC_PORT;

const int DEFAULT_KEYBD_SERVER_PORT = DEFAULT_NX_KEYBD_PORT_OFFSET +
                                          DEFAULT_NX_KEYBD_PORT;

const int DEFAULT_SAMBA_SERVER_PORT = DEFAULT_NX_SAMBA_PORT_OFFSET +
                                          DEFAULT_NX_SAMBA_PORT;

const int DEFAULT_MEDIA_SERVER_PORT = DEFAULT_NX_MEDIA_PORT_OFFSET +
                                          DEFAULT_NX_MEDIA_PORT;

//
// Footnotes.
//

static const char CopyrightInfo[] =
"\
Copyright (c) 2001,2003 NoMachine, http://www.nomachine.com.\n\
\n\
NXPROXY, NX protocol compression and NX extensions to this software\n\
are copyright of NoMachine. Redistribution and use of the present\n\
software is allowed according to terms specified in the file LICENSE\n\
which comes in the source distribution.\n\
\n\
Check http://www.nomachine.com/licensing.html for applicability.\n\
\n\
NX and NoMachine are trademarks of Medialogic S.p.A.\n\
\n\
All rights reserved.\n\
\n\
";

const char *GetCopyrightInfo()
{
  return CopyrightInfo;
}

static const char OtherCopyrightInfo[] =
"\
NX protocol compression is derived from DXPC project.\n\
\n\
Copyright (c) 1995,1996 Brian Pane\n\
Copyright (c) 1996,1997 Zachary Vonler and Brian Pane\n\
Copyright (c) 1999 Kevin Vigor and Brian Pane\n\
Copyright (c) 2000,2003 Gian Filippo Pinzari and Brian Pane\n\
\n\
All rights reserved.\n\
\n\
";

const char *GetOtherCopyrightInfo()
{
  return OtherCopyrightInfo;
}

int _hostBigEndian  = 0;
int _storeBigEndian = 0;

unsigned int GetUINT(unsigned const char *buffer, int bigEndian)
{
  //
  // TODO: It doesn't work on SPARCs if buffer
  // is not aligned to word boundary. We should
  // check the CPU, not the OS as this probably
  // applies to other processors.
  //

  #ifndef __sun

  if (_hostBigEndian == bigEndian)
  {
    return *((unsigned short *) buffer);
  }

  #else

  if (_hostBigEndian == bigEndian && ((unsigned int) buffer) & 0x1 == 0)
  {
    return *((unsigned short *) buffer);
  }

  #endif

  unsigned int result;

  if (bigEndian)
  {
    result = *buffer;

    result <<= 8;

    result += buffer[1];
  }
  else
  {
    result = buffer[1];

    result <<= 8;

    result += *buffer;
  }

  return result;
}

unsigned int GetULONG(unsigned const char *buffer, int bigEndian)
{
  //
  // TODO: It doesn't work on SPARCs if buffer
  // is not aligned to word boundary. We should
  // check the CPU, not the OS as this probably
  // applies to other processors.
  //

  #ifndef __sun

  if (_hostBigEndian == bigEndian)
  {
    return *((unsigned int *) buffer);
  }

  #else

  if (_hostBigEndian == bigEndian && ((unsigned int) buffer) & 0x3 == 0)
  {
    return *((unsigned int *) buffer);
  }

  #endif

  const unsigned char *next = (bigEndian ? buffer : buffer + 3);

  unsigned int result = 0;

  for (int i = 0; i < 4; i++)
  {
    result <<= 8;

    result += *next;

    if (bigEndian)
    {
      next++;
    }
    else
    {
      next--;
    }
  }

  return result;
}

void PutUINT(unsigned int value, unsigned char *buffer, int bigEndian)
{
  if (_hostBigEndian == bigEndian)
  {
    *((unsigned short *) buffer) = value;

    return;
  }

  if (bigEndian)
  {
    buffer[1] = (unsigned char) (value & 0xff);

    value >>= 8;

    *buffer = (unsigned char) value;
  }
  else
  {
    *buffer = (unsigned char) (value & 0xff);

    value >>= 8;

    buffer[1] = (unsigned char) value;
  }
}

void PutULONG(unsigned int value, unsigned char *buffer, int bigEndian)
{
  if (_hostBigEndian == bigEndian)
  {
    *((unsigned int *) buffer) = value;

    return;
  }

  if (bigEndian)
  {
    buffer += 3;

    for (int i = 4; i; i--)
    {
      *buffer-- = (unsigned char) (value & 0xff);

      value >>= 8;
    }
  }
  else
  {
    for (int i = 4; i; i--)
    {
      *buffer++ = (unsigned char) (value & 0xff);

      value >>= 8;
    }
  }
}

int PutData(ostream *fs, unsigned char *buffer, int size)
{
  fs -> write((char *) buffer, size);

  #ifdef DEBUG
  *logofs << "PutData: Written " << size << " bytes with eof "
          << fs -> eof() << " fail " << fs -> fail() << " and bad "
          << fs -> bad() << ".\n" << logofs_flush;
  #endif

  if (fs -> fail())
  {
    return -1;
  }

  return size;
}

int GetData(istream *fs, unsigned char *buffer, int size)
{
  fs -> read((char *) buffer, size);

  #ifdef DEBUG
  *logofs << "GetData: Read " << size << " bytes with eof "
          << fs -> eof() << " fail " << fs -> fail()
          << " and bad " << fs -> bad() << ".\n"
          << logofs_flush;
  #endif

  #ifdef __APPLE__

  if (fs -> bad())
  {
    return -1;
  }

  #else

  if (fs -> fail())
  {
    return -1;
  }

  #endif

  return size;
}

unsigned int RoundUp4(unsigned int x)
{
  unsigned int y = x / 4;

  y *= 4;

  if (y != x)
  {
    y += 4;
  }

  return y;
}

//
// Always include this in code as it is generally
// needed to test channels and split store.
//

const char *DumpChecksum(const void *checksum)
{
  static char string[MD5_LENGTH * 2 + 1];

  if (checksum != NULL)
  {
    for (unsigned int i = 0; i < MD5_LENGTH; i++)
    {
      sprintf(string + (i * 2), "%02X", ((unsigned char *) checksum)[i]);
    }
  }
  else
  {
    strcpy(string, "null");
  }

  return string;
}

//
// Define OPCODES here and in in client or
// server channels if you want to log the
// opcodes' literal.
//

#if defined(DUMP) || defined(OPCODES)

const char *DumpOpcode(const int &opcode)
{
  switch (opcode)
  {
    case X_CreateWindow:
    {
      return "X_CreateWindow";
    }
    case X_ChangeWindowAttributes:
    {
      return "X_ChangeWindowAttributes";
    }
    case X_GetWindowAttributes:
    {
      return "X_GetWindowAttributes";
    }
    case X_DestroyWindow:
    {
      return "X_DestroyWindow";
    }
    case X_DestroySubwindows:
    {
      return "X_DestroySubwindows";
    }
    case X_ChangeSaveSet:
    {
      return "X_ChangeSaveSet";
    }
    case X_ReparentWindow:
    {
      return "X_ReparentWindow";
    }
    case X_MapWindow:
    {
      return "X_MapWindow";
    }
    case X_MapSubwindows:
    {
      return "X_MapSubwindows";
    }
    case X_UnmapWindow:
    {
      return "X_UnmapWindow";
    }
    case X_UnmapSubwindows:
    {
      return "X_UnmapSubwindows";
    }
    case X_ConfigureWindow:
    {
      return "X_ConfigureWindow";
    }
    case X_CirculateWindow:
    {
      return "X_CirculateWindow";
    }
    case X_GetGeometry:
    {
      return "X_GetGeometry";
    }
    case X_QueryTree:
    {
      return "X_QueryTree";
    }
    case X_InternAtom:
    {
      return "X_InternAtom";
    }
    case X_GetAtomName:
    {
      return "X_GetAtomName";
    }
    case X_ChangeProperty:
    {
      return "X_ChangeProperty";
    }
    case X_DeleteProperty:
    {
      return "X_DeleteProperty";
    }
    case X_GetProperty:
    {
      return "X_GetProperty";
    }
    case X_ListProperties:
    {
      return "X_ListProperties";
    }
    case X_SetSelectionOwner:
    {
      return "X_SetSelectionOwner";
    }
    case X_GetSelectionOwner:
    {
      return "X_GetSelectionOwner";
    }
    case X_ConvertSelection:
    {
      return "X_ConvertSelection";
    }
    case X_SendEvent:
    {
      return "X_SendEvent";
    }
    case X_GrabPointer:
    {
      return "X_GrabPointer";
    }
    case X_UngrabPointer:
    {
      return "X_UngrabPointer";
    }
    case X_GrabButton:
    {
      return "X_GrabButton";
    }
    case X_UngrabButton:
    {
      return "X_UngrabButton";
    }
    case X_ChangeActivePointerGrab:
    {
      return "X_ChangeActivePointerGrab";
    }
    case X_GrabKeyboard:
    {
      return "X_GrabKeyboard";
    }
    case X_UngrabKeyboard:
    {
      return "X_UngrabKeyboard";
    }
    case X_GrabKey:
    {
      return "X_GrabKey";
    }
    case X_UngrabKey:
    {
      return "X_UngrabKey";
    }
    case X_AllowEvents:
    {
      return "X_AllowEvents";
    }
    case X_GrabServer:
    {
      return "X_GrabServer";
    }
    case X_UngrabServer:
    {
      return "X_UngrabServer";
    }
    case X_QueryPointer:
    {
      return "X_QueryPointer";
    }
    case X_GetMotionEvents:
    {
      return "X_GetMotionEvents";
    }
    case X_TranslateCoords:
    {
      return "X_TranslateCoords";
    }
    case X_WarpPointer:
    {
      return "X_WarpPointer";
    }
    case X_SetInputFocus:
    {
      return "X_SetInputFocus";
    }
    case X_GetInputFocus:
    {
      return "X_GetInputFocus";
    }
    case X_QueryKeymap:
    {
      return "X_QueryKeymap";
    }
    case X_OpenFont:
    {
      return "X_OpenFont";
    }
    case X_CloseFont:
    {
      return "X_CloseFont";
    }
    case X_QueryFont:
    {
      return "X_QueryFont";
    }
    case X_QueryTextExtents:
    {
      return "X_QueryTextExtents";
    }
    case X_ListFonts:
    {
      return "X_ListFonts";
    }
    case X_ListFontsWithInfo:
    {
      return "X_ListFontsWithInfo";
    }
    case X_SetFontPath:
    {
      return "X_SetFontPath";
    }
    case X_GetFontPath:
    {
      return "X_GetFontPath";
    }
    case X_CreatePixmap:
    {
      return "X_CreatePixmap";
    }
    case X_FreePixmap:
    {
      return "X_FreePixmap";
    }
    case X_CreateGC:
    {
      return "X_CreateGC";
    }
    case X_ChangeGC:
    {
      return "X_ChangeGC";
    }
    case X_CopyGC:
    {
      return "X_CopyGC";
    }
    case X_SetDashes:
    {
      return "X_SetDashes";
    }
    case X_SetClipRectangles:
    {
      return "X_SetClipRectangles";
    }
    case X_FreeGC:
    {
      return "X_FreeGC";
    }
    case X_ClearArea:
    {
      return "X_ClearArea";
    }
    case X_CopyArea:
    {
      return "X_CopyArea";
    }
    case X_CopyPlane:
    {
      return "X_CopyPlane";
    }
    case X_PolyPoint:
    {
      return "X_PolyPoint";
    }
    case X_PolyLine:
    {
      return "X_PolyLine";
    }
    case X_PolySegment:
    {
      return "X_PolySegment";
    }
    case X_PolyRectangle:
    {
      return "X_PolyRectangle";
    }
    case X_PolyArc:
    {
      return "X_PolyArc";
    }
    case X_FillPoly:
    {
      return "X_FillPoly";
    }
    case X_PolyFillRectangle:
    {
      return "X_PolyFillRectangle";
    }
    case X_PolyFillArc:
    {
      return "X_PolyFillArc";
    }
    case X_PutImage:
    {
      return "X_PutImage";
    }
    case X_GetImage:
    {
      return "X_GetImage";
    }
    case X_PolyText8:
    {
      return "X_PolyText8";
    }
    case X_PolyText16:
    {
      return "X_PolyText16";
    }
    case X_ImageText8:
    {
      return "X_ImageText8";
    }
    case X_ImageText16:
    {
      return "X_ImageText16";
    }
    case X_CreateColormap:
    {
      return "X_CreateColormap";
    }
    case X_FreeColormap:
    {
      return "X_FreeColormap";
    }
    case X_CopyColormapAndFree:
    {
      return "X_CopyColormapAndFree";
    }
    case X_InstallColormap:
    {
      return "X_InstallColormap";
    }
    case X_UninstallColormap:
    {
      return "X_UninstallColormap";
    }
    case X_ListInstalledColormaps:
    {
      return "X_ListInstalledColormaps";
    }
    case X_AllocColor:
    {
      return "X_AllocColor";
    }
    case X_AllocNamedColor:
    {
      return "X_AllocNamedColor";
    }
    case X_AllocColorCells:
    {
      return "X_AllocColorCells";
    }
    case X_AllocColorPlanes:
    {
      return "X_AllocColorPlanes";
    }
    case X_FreeColors:
    {
      return "X_FreeColors";
    }
    case X_StoreColors:
    {
      return "X_StoreColors";
    }
    case X_StoreNamedColor:
    {
      return "X_StoreNamedColor";
    }
    case X_QueryColors:
    {
      return "X_QueryColors";
    }
    case X_LookupColor:
    {
      return "X_LookupColor";
    }
    case X_CreateCursor:
    {
      return "X_CreateCursor";
    }
    case X_CreateGlyphCursor:
    {
      return "X_CreateGlyphCursor";
    }
    case X_FreeCursor:
    {
      return "X_FreeCursor";
    }
    case X_RecolorCursor:
    {
      return "X_RecolorCursor";
    }
    case X_QueryBestSize:
    {
      return "X_QueryBestSize";
    }
    case X_QueryExtension:
    {
      return "X_QueryExtension";
    }
    case X_ListExtensions:
    {
      return "X_ListExtensions";
    }
    case X_ChangeKeyboardMapping:
    {
      return "X_ChangeKeyboardMapping";
    }
    case X_GetKeyboardMapping:
    {
      return "X_GetKeyboardMapping";
    }
    case X_ChangeKeyboardControl:
    {
      return "X_ChangeKeyboardControl";
    }
    case X_GetKeyboardControl:
    {
      return "X_GetKeyboardControl";
    }
    case X_Bell:
    {
      return "X_Bell";
    }
    case X_ChangePointerControl:
    {
      return "X_ChangePointerControl";
    }
    case X_GetPointerControl:
    {
      return "X_GetPointerControl";
    }
    case X_SetScreenSaver:
    {
      return "X_SetScreenSaver";
    }
    case X_GetScreenSaver:
    {
      return "X_GetScreenSaver";
    }
    case X_ChangeHosts:
    {
      return "X_ChangeHosts";
    }
    case X_ListHosts:
    {
      return "X_ListHosts";
    }
    case X_SetAccessControl:
    {
      return "X_SetAccessControl";
    }
    case X_SetCloseDownMode:
    {
      return "X_SetCloseDownMode";
    }
    case X_KillClient:
    {
      return "X_KillClient";
    }
    case X_RotateProperties:
    {
      return "X_RotateProperties";
    }
    case X_ForceScreenSaver:
    {
      return "X_ForceScreenSaver";
    }
    case X_SetPointerMapping:
    {
      return "X_SetPointerMapping";
    }
    case X_GetPointerMapping:
    {
      return "X_GetPointerMapping";
    }
    case X_SetModifierMapping:
    {
      return "X_SetModifierMapping";
    }
    case X_GetModifierMapping:
    {
      return "X_GetModifierMapping";
    }
    case X_NoOperation:
    {
      return "X_NoOperation";
    }
    case X_NXInternalGenericData:
    {
      return "X_NXInternalGenericData";
    }
    //
    // case X_NXInternalGenericReply:
    // {
    //   return "X_NXInternalGenericReply";
    // }
    //
    case X_NXInternalGenericRequest:
    {
      return "X_NXInternalGenericRequest";
    }
    case X_NXInternalShapeExtension:
    {
      return "X_NXInternalShapeExtension";
    }
    case X_NXGetControlParameters:
    {
      return "X_NXGetControlParameters";
    }
    case X_NXGetCleanupParameters:
    {
      return "X_NXGetCleanupParameters";
    }
    case X_NXGetImageParameters:
    {
      return "X_NXGetImageParameters";
    }
    case X_NXGetUnpackParameters:
    {
      return "X_NXGetUnpackParameters";
    }
    case X_NXGetShmemParameters:
    {
      return "X_NXGetShmemParameters";
    }
    case X_NXStartSplit:
    {
      return "X_NXStartSplit";
    }
    case X_NXEndSplit:
    {
      return "X_NXEndSplit";
    }
    case X_NXSplit:
    {
      return "X_NXSplit";
    }
    case X_NXCommitSplit:
    {
      return "X_NXCommitSplit";
    }
    case X_NXSync:
    {
      return "X_NXSync";
    }
    case X_NXKarma:
    {
      return "X_NXKarma";
    }
    case X_NXSetExposeEvents:
    {
      return "X_NXSetExposeEvents";
    }
    case X_NXSetUnpackGeometry:
    {
      return "X_NXSetUnpackGeometry";
    }
    case X_NXSetUnpackColormap:
    {
      return "X_NXSetUnpackColormap";
    }
    case X_NXSetUnpackAlpha:
    {
      return "X_NXSetUnpackAlpha";
    }
    case X_NXPutPackedImage:
    {
      return "X_NXPutPackedImage";
    }
    case X_NXAbortSplit:
    {
      return "X_NXAbortSplit";
    }
    default:
    {
      if (opcode > 127)
      {
        return "Extension";
      }
      else
      {
        return "?";
      }
    }
  }
}

#else /* #if defined(DUMP) || defined(OPCODES) */

const char *DumpOpcode(const int &opcode)
{
  return "?";
}

#endif /* #if defined(DUMP) || defined(OPCODES) */

//
// Save space by compiling these functions
// only if DUMP explicitly defined.
//

#ifdef DUMP

void DumpData(const unsigned char *buffer, unsigned int size)
{
  if (buffer != NULL)
  {
    unsigned int i = 0;

    while (i < size)
    {
      *logofs << "[" << i << "]\t";

      for (unsigned int ii = 0; i < size && ii < 8; i++, ii++)
      {
        *logofs << (unsigned int) (buffer[i]) << "\t";
      }

      *logofs << "\n" << logofs_flush;
    }
  }
}

void DumpChecksum(const unsigned char *buffer, unsigned int size)
{
  if (buffer != NULL)
  {
    md5_byte_t md5_digest[MD5_LENGTH];

    md5_state_t md5_state;

    md5_init(&md5_state);

    md5_append(&md5_state, buffer, size);

    md5_finish(&md5_state, md5_digest);

    char md5_string[MD5_LENGTH * 2 + 1];

    for (unsigned int i = 0; i < MD5_LENGTH; i++)
    {
      sprintf(md5_string + (i * 2), "%02X", md5_digest[i]);
    }

    *logofs <<  "[" << md5_string << "]" << logofs_flush;
  }
}

void DumpBlockChecksums(const unsigned char *buffer,
                            unsigned int size, unsigned int block)
{
  for (unsigned int i = 0; i < (size / block); i++)
  {
    *logofs << "[" << i * block << "]";

    DumpChecksum(buffer + (i * block), block);

    *logofs << "\n";
  }

  if (size % block > 0)
  {
    *logofs << "[" << size / block * block << "]";

    DumpChecksum(buffer + (size / block * block), size % block);

    *logofs << "\n";
  }
}

void DumpHexData(const unsigned char *buffer, unsigned int size)
{
  char message [65536];
  char ascii   [17];

  unsigned int index = 0;
  unsigned int linescan = 0;
  unsigned int index_ascii = 0;

  sprintf  (message,"\n####  Start Dump Buffer of [%.5d] Bytes ####\n\n",size);

  *logofs << message << logofs_flush;

  // 
  // "Index    0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f  Ascii           "
  // "-----   -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --  ----------------"
  // "00000 : 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00 00  ................"
  // 

  sprintf (message,"Index   0  1  2  3  4  5  6  7  8  9  a  b  c  d  e  f  Ascii           \n");
  *logofs << message << logofs_flush;
  sprintf (message,"-----  -- -- -- -- -- -- -- -- -- -- -- -- -- -- -- --  ----------------\n");
  *logofs << message << logofs_flush;

  index = 0;

  while (index < size)
  {
    memset (ascii, ' ', sizeof(ascii));

    ascii[16] = '\0';

    sprintf  (message,"%.5d  ", index);

    for (index_ascii = 0, linescan = index;
             ((index < (linescan + 16)) && (index < size)); 
                  index++, index_ascii++)
    {
      if (isprint(buffer [index]))
      {
        ascii[index_ascii] = buffer [index];
      }
      else
      {
        ascii[index_ascii] = '.';
      }

      sprintf  (&message [strlen (message)],"%.2x ", (unsigned char) buffer [index]);
    }

    for (linescan = index_ascii; linescan < 16; linescan++)
    {
      strcat (&message [strlen (message)], "   ");
    }

    sprintf  (&message [strlen (message)]," %s\n", ascii);

    *logofs << message << logofs_flush;
  } 

  sprintf  (message,"\n####  End Dump Buffer ####\n\n");

  *logofs << message << logofs_flush;
}

#endif /* #ifdef DUMP */
