/*
 *	cfg.c - configuration file stuff
 *	Copyright (C) 2000-2003 Fred Barnes <frmb2@ukc.ac.uk>
 *
 *	This program is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version.
 *
 *	This program is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this program; if not, write to the Free Software
 *	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#define __CFG_C

/*{{{  includes*/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <unistd.h>
#include <X11/Xlib.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <sys/fcntl.h>
#include <sys/mman.h>
#include <sys/stat.h>
#include <netdb.h>
#include <errno.h>

#include "support.h"
#include "cfg.h"
#include "interface.h"
#include "error.h"
/*}}}*/
/*{{{  statics, externs */
#define NUM_CONFIG_FILES (2)
static char *config_files[] = {
	SYSCONFDIR "/xdmchoose.conf",
	"./xdmchoose.conf"
};
extern char *cmdline_conffile;			/* in chooser.c */
extern int grabkybd;				/* in chooser.c */

static char *config_file = NULL;
static int config_fd = -1;
static char *config_mmap = NULL;
static int config_mmap_size = 0;
static char *data_start = NULL;
static char **config_lines = NULL;
static int config_nlines = 0;
static int data_line;
/*          %SERVER%      %SERVER.IP% %XDMHOST%      %XDMHOST.IP% */
static char *server_name, *server_ip, *xdmhost_name, *xdmhost_ip;
/*}}}*/
/*{{{  forward decls*/
static int scanto (char *ident, char *setting);
static int extract_int (char *ident, char *setting, int *dest);
static int extract_mstring (char *ident, char *setting, char **dest);
static int extract_lstrings (char *ident, char *setting, char ***dest);
/*}}}*/
/*{{{  void show_config_filenames (FILE *stream)*/
/*
 *	void show_config_filenames (FILE *stream)
 *	shows the various compiled-in config filenames, one per line, tab indented.
 */
void show_config_filenames (FILE *stream)
{
	int i;

	for (i=0; i<NUM_CONFIG_FILES; i++) {
		fprintf (stream, "\t%s\n", config_files[i]);
	}
	return;
}
/*}}}*/
/*{{{  int init_config_file (GlobalConfigData *data)*/
/*
 *	int init_config_file (GlobalConfigData *data)
 *	Initialises the configuration stuff, placing global settings in 'data'
 *	Returns 0 on success, -1 on failure
 */
int init_config_file (GlobalConfigData *data)
{
	int i;
	struct hostent *hp;
	char name_buffer[256], *disp_env, *ch, *dh;
	struct stat st_buf;
	int n_lines;
	int in_string;

	disp_env = getenv ("DISPLAY");
	if (!disp_env) {
		APP_MSG ("DISPLAY environment not set!");
		server_name = string_dup ("unknown");
		server_ip = string_dup ("0.0.0.0");
	} else {
		strncpy (name_buffer, disp_env, sizeof(name_buffer)-1);
		for (ch = name_buffer; (*ch != ':') && (*ch != '\0'); ch++);
		*ch = '\0';
		if (!strlen (name_buffer)) {
			/* probably was :0.0 or similar */
			*ch = ':';
			server_name = string_dup (ch);
			server_ip = string_dup ("127.0.0.1");
		} else {
			hp = gethostbyname (name_buffer);
			if (!hp) {
				server_name = string_dup (name_buffer);
				server_ip = string_dup ("0.0.0.0");
			} else {
				server_name = string_dup (hp->h_name);
				*ch = ':';
				server_name = string_cat (server_name, ch);
				server_ip = string_dup (inet_ntoa (*(struct in_addr *)(hp->h_addr_list[0])));
			}
		}
	}
	if (gethostname (name_buffer, sizeof(name_buffer)-1)) {
		APP_MSG ("unable to gethostname ()!");
		xdmhost_name = string_dup ("unknown");
		xdmhost_ip = string_dup ("0.0.0.0");
	} else {
		hp = gethostbyname (name_buffer);
		if (!hp) {
			xdmhost_name = string_dup (name_buffer);
			xdmhost_ip = string_dup ("0.0.0.0");
		} else {
			xdmhost_name = string_dup (hp->h_name);
			xdmhost_ip = string_dup (inet_ntoa (*(struct in_addr *)(hp->h_addr_list[0])));
		}
	}
	if (cmdline_conffile) {
		if (access (cmdline_conffile, R_OK)) {
			APP_MSG ("unable to read config file %s", cmdline_conffile);
			return -1;
		}
		config_file = cmdline_conffile;
	} else {
		for (i=0; i<NUM_CONFIG_FILES; i++) {
			if (!access (config_files[i], R_OK)) {
				break;
			}
		}
		if (i == NUM_CONFIG_FILES) {
			APP_MSG ("unable to find valid configuration file");
			return -1;
		}
		config_file = config_files[i];
	}
	if (stat (config_file, &st_buf)) {
		APP_MSG ("unable to stat configuration file %s", config_file);
		return -1;
	}

	/* mmap the file and create line-array */
	config_fd = open (config_file, O_RDONLY);
	if (config_fd < 0) {
		APP_MSG ("unable to open configuration file %s: %s", config_file, strerror (errno));
		return -1;
	}
	config_mmap_size = st_buf.st_size;
	config_mmap = (char *)mmap ((void *)0, config_mmap_size, PROT_READ|PROT_WRITE, MAP_PRIVATE, config_fd, 0);
	if (config_mmap == (char *)-1) {
		close (config_fd);
		APP_MSG ("unable to mmap() configuration file %s: %s", config_file, strerror (errno));
		return -1;
	}

	/* create line offsets */
	n_lines = 0;
	for (ch = config_mmap; ch < (config_mmap + config_mmap_size); ch++) {
		switch (*ch) {
		case '\n':
			n_lines++;
			break;
		}
	}
	config_lines = (char **)xmalloc (n_lines * sizeof (char *));
	memset (config_lines, 0, n_lines * sizeof (char *));
	dh = config_mmap;		/* ptr to line-start */
	in_string = 0;
	i = 0;
	ch = dh;
	while (ch < (config_mmap + config_mmap_size)) {
		/* find start of line */
		for (ch = dh; ((*ch == ' ') || (*ch == '\t')) && (ch < (config_mmap + config_mmap_size)); ch++);
		if ((*ch == '\r') || (*ch == '\n')) {
			/* blank line */
			for (dh = ch; (*dh == '\n') || (*dh == '\r'); dh++);
		} else if (*ch == '#') {
			/* comment line, scan to start of next line */
			for (dh = ch; (*dh != '\n') && (*dh != '\r') && (dh < (config_mmap + config_mmap_size)); dh++);
			while (((*dh == '\n') || (*dh == '\r')) && (dh < (config_mmap + config_mmap_size))) dh++;
		} else {
			/* something here */
			config_lines[i++] = ch;
			while ((*ch != '\n') && (*ch != '\r') && (ch < (config_mmap + config_mmap_size))) {
				if (*ch == '\"') {
					in_string = (in_string ? 0 : 1);
				} else if ((*ch == '#') && !in_string) {
					/* comment start */
					*ch = '\0';
					for (ch++; (*ch != '\n') && (*ch != '\r') && (ch < (config_mmap + config_mmap_size)); ch++);
					break;
				}
				ch++;
			}
			if (ch < (config_mmap + config_mmap_size)) {
				while (((*ch == '\r') || (*ch == '\n')) && (ch < (config_mmap + config_mmap_size))) {
					*ch = '\0';
					ch++;
				}
			}
			dh = ch;
		}
	}
	config_nlines = i;

	/* Read global config */
	data->hidequery = 0;
	data->allowbroadcast = 0;
	data->startscript = NULL;
	data->extratext = 0;
	data->allowctrlc = 1;
	data->ctrlcexitcode = 255;
	data->stripdomain = 0;
	data->startscriptkill = NULL;
	data->allowtype = 1;
	data->hostlist = NULL;
	data->grabkybd = grabkybd;
	extract_int ("app", "hidequery", &data->hidequery);
	extract_int ("app", "allowbroadcast", &data->allowbroadcast);
	extract_mstring ("app", "startscript", &data->startscript);
	extract_int ("app", "extratext", &data->extratext);
	extract_int ("app", "allowctrlc", &data->allowctrlc);
	extract_int ("app", "ctrlcexitcode", &data->ctrlcexitcode);
	extract_int ("app", "stripdomain", &data->stripdomain);
	extract_mstring ("app", "startscriptkill", &data->startscriptkill);
	extract_int ("app", "allowtype", &data->allowtype);
	extract_lstrings ("app", "hostlist", &data->hostlist);
	extract_int ("app", "grabkeyboard", &data->grabkybd);
	if (data->stripdomain) {
		/* remove .XYZ.foo.bar from server_name and xdmhost_name */
		char *ch;

		for (ch=server_name; (*ch != '.') && (*ch != '\0'); ch++);
		if (*ch == '.') {
			*ch = '\0';
		}
		for (ch=xdmhost_name; (*ch != '.') && (*ch != '\0'); ch++);
		if (*ch == '.') {
			*ch = '\0';
		}
	}
	return 0;
}
/*}}}*/
/*{{{  void close_config_file (void)*/
/*
 *	void close_config_file (void)
 *	closes the config file
 */
void close_config_file (void)
{
	if (config_lines) {
		free (config_lines);
		config_lines = NULL;
		config_nlines = 0;
	}
	if (config_mmap) {
		munmap (config_mmap, config_mmap_size);
		config_mmap = NULL;
	}
	if (config_fd >= 0) {
		close (config_fd);
		config_fd = -1;
	}
	return;
}
/*}}}*/
/*{{{  int read_config_entry (char *item, GadgetConfigData *data, int fields)*/
/*
 *	int read_config_entry (char *item, GadgetConfigData *data, int fields)
 *	fills up 'data' with config information about 'item'
 *	Returns 0 on success, -1 on failure
 */
int read_config_entry (char *item, GadgetConfigData *data, int fields)
{
	int errored;

	errored = 0;
	if (fields & FLD_TEXT) {
		if (extract_mstring (item, "text", &data->text)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "text");
			errored = -1;
		}
	} else {
		data->text = NULL;
	}
	if (fields & FLD_FOREGROUND) {
		if (extract_mstring (item, "foreground", &data->foreground)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "foreground");
			errored = -1;
		}
	} else {
		data->foreground = NULL;
	}
	if (fields & FLD_BACKGROUND) {
		if (extract_mstring (item, "background", &data->background)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "background");
			errored = -1;
		}
	} else {
		data->background = NULL;
	}
	if (fields & FLD_HIGHLIGHT) {
		if (extract_mstring (item, "highlight", &data->highlight)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "highlight");
			errored = -1;
		}
	} else {
		data->highlight = NULL;
	}
	if (fields & FLD_SHADOW) {
		if (extract_mstring (item, "shadow", &data->shadow)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "shadow");
			errored = -1;
		}
	} else {
		data->shadow = NULL;
	}
	if (fields & FLD_FONT) {
		if (extract_mstring (item, "font", &data->font)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "font");
			errored = -1;
		}
	} else {
		data->font = NULL;
	}
	if (fields & FLD_TENTRY) {
		if (extract_mstring (item, "tentry", &data->tentry)) {
			APP_MSG ("configuration line for %s.%s missing.", item, "tentry");
			errored = -1;
		}
	} else {	
		data->tentry = NULL;
	}
	return errored;
}
/*}}}*/
/*{{{  int have_config_entry (char *item, int fields)*/
/*
 *	checks to see if particular entries are here
 *	return 0 on success, number of non-matches on failure
 */
int have_config_entry (char *item, int fields)
{
	int errored = 0;

	if ((fields & FLD_TEXT) && scanto (item, "text")) {
		errored++;
	}
	if ((fields & FLD_FOREGROUND) && scanto (item, "foreground")) {
		errored++;
	}
	if ((fields & FLD_BACKGROUND) && scanto (item, "background")) {
		errored++;
	}
	if ((fields & FLD_HIGHLIGHT) && scanto (item, "highlight")) {
		errored++;
	}
	if ((fields & FLD_SHADOW) && scanto (item, "shadow")) {
		errored++;
	}
	if ((fields & FLD_FONT) && scanto (item, "font")) {
		errored++;
	}
	if ((fields & FLD_TENTRY) && scanto (item, "tentry")) {
		errored++;
	}
	return errored;
}
/*}}}*/
/*{{{  static int scanto (char *ident, char *setting)*/
/*
 *	int scanto (char *ident, char *setting)
 *	scans to a particular line in the config file
 *	Returns 0 on success, -1 on failure
 */
static int scanto (char *ident, char *setting)
{
	int i, ilen, slen;

	if (!config_mmap || !config_lines) {
		return -1;
	}
	ilen = strlen (ident);
	slen = strlen (setting);
	for (i=0; i<config_nlines; i++) {
		if (strlen (config_lines[i]) < (ilen + slen + 1)) {
			continue;
		}
		if (memcmp (config_lines[i], ident, ilen)) {
			continue;
		}
		if (config_lines[i][ilen] != '.') {
			continue;
		}
		if (memcmp (config_lines[i] + (ilen + 1), setting, slen)) {
			continue;
		}
		data_start = config_lines[i] + (slen + ilen + 1);
		while ((*data_start == ' ') || (*data_start == '\t')) {
			data_start++;
		}
		break;
	}
	if (i == config_nlines) {
		return -1;
	}
	return 0;
}
/*}}}*/
/*{{{  static int extract_int (char *ident, char *setting, int *dest)*/
/*
 *	int extract_int (char *ident, char *setting, int *dest)
 *	Extracts an integer value of the specified name
 *	Returns 0 on success, -1 on failure
 */
static int extract_int (char *ident, char *setting, int *dest)
{
	int tmp;

	if (scanto (ident, setting)) {
		if (scanto ("unknown", setting)) {
			return -1;
		}
	}
	if (!sscanf (data_start, "%d", &tmp)) {
		APP_MSG ("badly formed integer for %s.%s in %s(%d).", ident, setting, config_file, data_line);
		return -1;
	}
	*dest = tmp;
	return 0;
}
/*}}}*/
/*{{{  static int extract_mstring (char *ident, char *setting, char **dest)*/
/*
 *	int extract_mstring (char *ident, char *setting, char **dest)
 *	Extracts a string, but allocates memory for the return
 *	Returns 0 on success, -1 on failure
 */
static int extract_mstring (char *ident, char *setting, char **dest)
{
	char *ch;
	char *sublist[5] = { "%SERVER%", "%SERVER.IP%", "%XDMHOST%", "%XDMHOST.IP%", "%VERSION%" };
	static char *verstr = VERSION;
	char **withlist[5] = { &server_name, &server_ip, &xdmhost_name, &xdmhost_ip, &verstr };
	int num_items = 5;
	int i, x, y, z, newlen, offs;
	static char databuf[256];

	if (scanto (ident, setting)) {
		if (scanto ("unknown", setting)) {
			return -1;
		}
	}
	if (strlen (data_start) > 255) {
		return -1;
	}
	strcpy (databuf, data_start);
	data_start = databuf;
	if (*data_start == '\"') {
		ch = strchr (data_start + 1, '\"');
		if (!ch) {
			APP_MSG ("missing quote for %s.%s in %s(%d).", ident, setting, config_file, data_line);
			return -1;
		}
		*ch = '\0';
	}
	data_start++;
	*dest = string_dup (data_start);
	/* do the subsitutions thing */
	for (i=0; i<num_items; i++) {
		while ((ch = strstr (*dest, sublist[i]))) {
			offs = (ch - *dest);
			x = strlen (*dest) + 1;
			y = strlen (*(withlist[i]));
			z = strlen (sublist[i]);
			newlen = (x - z) + y;
			if (newlen < x) {
				/* hello world! */
			} else {
				*dest = (char *)xrealloc ((void *)*dest, x, (x - z) + y);
			}
			ch = *dest + offs;
			if (*(ch + z) == '\0') {
				strcpy (*dest + offs, *(withlist[i]));
			} else {
				memmove (ch + y, ch + z, strlen (ch + z)+1);
				memmove (ch, *(withlist[i]), y);
			}
		}
	}
	return 0;
}
/*}}}*/
/*{{{  static int extract_lstrings (char *ident, char *setting, char ***dest)*/
/*
 *	static int extract_lstrings (char *ident, char *setting, char ***dest)
 *	Extracts a list of strings, no subsitutions here
 */
static int extract_lstrings (char *ident, char *setting, char ***dest)
{
	static char databuf[1024];
	char **tmp;
	int n_tmp, m_tmp;

	if (scanto (ident, setting)) {
		if (scanto ("unknown", setting)) {
			return -1;
		}
	}
	if (strlen (data_start) > 1023) {
		return -1;
	}
	strcpy (databuf, data_start);
	data_start = databuf;
	n_tmp = m_tmp = 0;
	tmp = NULL;
	while (*data_start == '"') {
		char *ch;

		for (ch = data_start + 1; (*ch != '\"') && (*ch != '\0'); ch++);
		if (*ch == '\0') {
			APP_MSG ("missing quote in config file for %s.%s in %s(%d).", ident, setting, config_file, data_line);
			return -1;
		}
		*ch = '\0';
		if (n_tmp == m_tmp) {
			tmp = (char **)xrealloc (tmp, m_tmp * sizeof(char *), (m_tmp + 10) * sizeof(char *));
			m_tmp += 10;
		}
		tmp[n_tmp++] = string_dup (data_start + 1);
		for (ch++; (*ch == ' ') || (*ch == '\t'); ch++);
		if (*ch == ',') {
			for (ch++; (*ch == ' ') || (*ch == '\t'); ch++);
		}
		data_start = ch;
	}
	if (*data_start != '\0') {
		APP_MSG ("garbage in config file for %s.%s in %s(%d).", ident, setting, config_file, data_line);
		return -1;
	}
	data_start++;
	/* sort out target */
	if (n_tmp) {
		*dest = (char **)xmalloc ((n_tmp + 1) * sizeof(char *));
		memcpy (*dest, tmp, n_tmp * sizeof(char *));
		(*dest)[n_tmp] = NULL;
		free (tmp);
	}
	return 0;
}
/*}}}*/

