/*****************************************************************************
 * channel.c: channel for Bt848 frame grabber driver
 *****************************************************************************
 * $Id: channel.c,v 1.102 2004/12/08 10:04:09 alainjj Exp $
 *****************************************************************************
 * Copyright (C) 2001 Keuleu
 * Sun 09 Nov 2003 07:20:47 PM EET Lucian Langa
 *	Added eventmap rc file config
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111, USA.
 *****************************************************************************
 *
 *  Original code:
 *  channel for Bt848 frame grabber driver
 *
 *  Copyright (C) 1996,97 Marcus Metzler (mocm@thp.uni-koeln.de)
 *
 *  many changes by Gerd Knorr <kraxel@cs.tu-berlin.de>
 *      [ hmm, think by now nearly nothing left from the original code ... ]
 *
 *****************************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <ctype.h>
#include <math.h>

#include <X11/Intrinsic.h>
#ifndef NO_X11
# include <X11/StringDefs.h>
# include <X11/Xaw/XawInit.h>
# include <X11/Xaw/Command.h>
#endif

#include "config.h"
#include "strtab.h"
#include "grab.h"
#include "x11.h"
#include "channels.h"
#include "channel.h"
#include "mixer.h"
#include "audio.h"
#include "memcpy.h"
#include "deinterlace.h"
#include "divx_ui.h"
#include "subtitles_ui.h"
#include "grab_ui.h"
#include "divx.h"
#include "alevtparams_ui.h"
#include "xosdparams_ui.h"
#include "x11params_ui.h"
#include "blackborder.h"
#include "commands.h"
#include "vop.h"

#include "event.h"

#ifdef HAVE_FFMPEG
# include <ffmpeg/avcodec.h>
#endif

/* ----------------------------------------------------------------------- */
/* misc common stuff, not only channel related                             */

struct CHANNEL defaults = {
  "defaults",        // name
  NULL,              // key
  "33",              // cname
  0,                 // channel
  0,                 // fine
  0,                 // freq
  CAPTURE_OVERLAY,   // capture
  0,                 // source
  2,                 // norm
  19000,             // color
  34000,             // bright
  32768,             // hue
  28000,             // contrast
  0,                 // pixmap
  0,                 // button
  0,                 // ckey
  DEINTERLACE_NONE,  // deinterlace
  "888",               // subpage        
};

struct CHANNEL **channels = NULL;

int count = 0;

int cur_sender = -1, cur_channel = 5, cur_fine = 0;
int cur_norm = -1, cur_input = -1, cur_freq = -1;
char *cur_subpage;

int cur_maxwidth = 768, cur_maxheight = 576, cur_secam = 0;
int cur_maxfpsnum = 25, cur_maxfpsden = 1;
int chan_tab = 4;

extern char user_dir[255];
extern int have_config;
extern struct GRABBER *grabbers[];
extern int grabber;
extern int fs_width, fs_height;
extern int fs_xoff, fs_yoff;
extern int pix_width, pix_height;
extern int owidth, oheight;
extern int wwidth, wheight;
extern int width_capture_max, height_capture_max;
extern struct STRTAB deinterlace_list[];
extern struct STRTAB subpageno_list[];
extern int deinterlace_method;
extern void set_deinterlace(int j);
extern int xawpopup;
void set_blackborder (int j);
extern int message_timer;
extern int decoration; 
#ifdef HAVE_XV
extern int cmdline_xv_img_port;
#endif

//#ifdef USE_LIBXOSD <= xosd inside now
extern char *font, *font_vtx, *colour, *outline_colour;
extern int shadow_offset, outline_offset;
extern char main_font[50], main_size[50];
extern char vtx_font[50], vtx_size[50];
extern int libxosd;
//#endif <= xosd inside now
#ifndef NO_X11
extern Display *dpy;
void button_cb (Widget widget, XtPointer clientdata, XtPointer call_data);

extern Widget app_shell, chan_box, chan_viewport, tv, opt_paned, c_stayontop, c_decoration;
#endif

extern divx_t divx;
extern char *pplayer;
extern int stay;
extern int subactive;
extern int oldswitch;
extern char *filepath;
#ifdef HAVE_LIBJPEG
extern int jpegpcent;
#endif

extern int memcpy_method;
struct STRTAB memcpy_list[] = {
  {MEMCPY_PROBE,      "probe"},
  {MEMCPY_GLIBC,      "glibc"},
  {MEMCPY_KERNEL,     "kernel"},
  {MEMCPY_FPU_512,    "FPU_512"},
  {MEMCPY_FPU_1K,     "FPU_1K"},
  {MEMCPY_FPU_2K,     "FPU_2K"},
  {MEMCPY_FPU_4K,     "FPU_4K"},
  {MEMCPY_FPU_8K,     "FPU_8K"},
  {MEMCPY_FPU_16K,    "FPU_16K"},
  {MEMCPY_FPU_32K,    "FPU_32K"},
  {MEMCPY_MMX,        "mmx"},
  {MEMCPY_MMX_512,    "mmx_512"},
  {MEMCPY_MMX_1K,     "mmx_1K"},
  {MEMCPY_MMX_2K,     "mmx_2K"},
  {MEMCPY_MMX_4K,     "mmx_4K"},
  {MEMCPY_MMX_8K,     "mmx_8K"},
  {MEMCPY_MMX_16K,    "mmx_16K"},
  {MEMCPY_MMX_32K,    "mmx_32K"},
  {MEMCPY_MMXEXT,     "mmxext"},
  {MEMCPY_MMXEXT_1K,  "mmxext_1K"},
  {MEMCPY_MMXEXT_2K,  "mmxext_2K"},
  {MEMCPY_MMXEXT_4K,  "mmxext_4K"},
  {MEMCPY_MMXEXT_8K,  "mmxext_8K"},
  {MEMCPY_MMXEXT_16K, "mmxext_16K"},
  {MEMCPY_MMXEXT_32K, "mmxext_32K"},
  {MEMCPY_MMXEXT_64K, "mmxext_64K"},
  {MEMCPY_SSE,        "sse"},
  {MEMCPY_SSE_1K,     "sse_1K"},
  {MEMCPY_SSE_2K,     "sse_2K"},
  {MEMCPY_SSE_4K,     "sse_4K"},
  {MEMCPY_SSE_8K,     "sse_8K"},
  {MEMCPY_SSE_16K,    "sse_16K"},
  {MEMCPY_SSE_32K,    "sse_32K"},
  {MEMCPY_SSE_64K,    "sse_64K"},
  {-1, NULL}
};

/* ----------------------------------------------------------------------- */
void save_config (void);
/* ----------------------------------------------------------------------- */

int
lookup_channel (char *channel)
{
  int i, nr1, nr2;
  char tag1[5], tag2[5];

  if (isdigit (channel[0]))
    {
      tag1[0] = 0;
      nr1 = atoi (channel);
    }
  else
    {
      sscanf (channel, "%4[A-Za-z]%d", tag1, &nr1);
    }

  for (i = 0; i < CHAN_ENTRIES; i++)
    {
      if (isdigit (tvtuner[i].name[0]))
        {
          tag2[0] = 0;
          nr2 = atoi (tvtuner[i].name);
        }
      else
        {
          sscanf (tvtuner[i].name, "%4[A-Za-z]%d", tag2, &nr2);
        }

      /* First simply test for name equality */
      if (strcmp(channel, tvtuner[i].name) == 0)
        break;

      /* If it doesn't match, try to make a guess */
      if (tag1[0] && tag2[0])
        if (nr1 == nr2 && 0 == strcmp (tag1, tag2))
          break;
      if (!tag1[0] && !tag2[0])
        if (nr1 == nr2)
          break;
    }
  if (i == CHAN_ENTRIES)
    return -1;

  return i;
}

int
get_freq (int i)
{
  if (!tvtuner[i].freq[chan_tab])
    return -1;
  return tvtuner[i].freq[chan_tab] * 16 / 1000;
}

int
cf2freq (char *name, int fine)
{
  int i;

  if (-1 == (i = lookup_channel (name)))
    return -1;
  return get_freq (i) + fine;
}

void
get_font_infos (char *font, char *font_type, char *font_size)
{
char *tok;
int i = 0;
char *cmd;
cmd = malloc(strlen(font)+1);
strcpy(cmd, font);
tok = strtok(cmd,"-"); 
  while(tok != NULL) 
  {
    i = i + 1;
    if (i == 2) { strcpy(font_type,tok); }
    if (i == 8) { strcpy(font_size,tok); }
    tok = strtok(NULL,"-");
  }
}


/* ----------------------------------------------------------------------- */

struct STRTAB captab[] = {
  {CAPTURE_OFF, "off"},
  {CAPTURE_OFF, "no"},
  {CAPTURE_OFF, "false"},
  {CAPTURE_OVERLAY, "on"},
  {CAPTURE_OVERLAY, "yes"},
  {CAPTURE_OVERLAY, "true"},
  {CAPTURE_OVERLAY, "overlay"},
  {CAPTURE_GRABDISPLAY, "grab"},
  {CAPTURE_GRABDISPLAY, "grabdisplay"},
  {-1, NULL}
};

/* just malloc memory for a new channel ... */
struct CHANNEL *
add_channel (char *name)
{
  static int alloc_count = 0;
  struct CHANNEL *channel;

  if (alloc_count == count)
    {
      alloc_count += 16;
      if (alloc_count == 16)
        channels = malloc (sizeof (struct CHANNEL *) * alloc_count);
      else
        channels =
          realloc (channels, sizeof (struct CHANNEL *) * alloc_count);
    }
  channel = channels[count++] = malloc (sizeof (struct CHANNEL));
  memcpy (channel, &defaults, sizeof (struct CHANNEL));
  channel->name = strdup (name);
  return channel;
}

#ifndef NO_X11
void
hotkey_channel (struct CHANNEL *channel)
{
  char str[100], key[32], ctrl[16];

  if (NULL == channel->key)
    return;
  if (2 == sscanf (channel->key, "%15[A-Za-z0-9_]+%31[A-Za-z0-9_]",
                   ctrl, key))
    sprintf (str, "%s<Key>%s: SetChannel(\"%s\")", ctrl, key, channel->name);
  else
    sprintf (str, "<Key>%s: SetChannel(\"%s\")", channel->key, channel->name);
  XtOverrideTranslations (tv, XtParseTranslationTable (str));
  XtOverrideTranslations (opt_paned, XtParseTranslationTable (str));
  XtOverrideTranslations (chan_viewport, XtParseTranslationTable (str));
}
#endif

/* ... and initalize later */
void
configure_channel (struct CHANNEL *channel)
{
  channel->freq = get_freq (channel->channel) + channel->fine;
#ifndef NO_X11
  channel->button =
    XtVaCreateManagedWidget (channel->name,
                             commandWidgetClass, chan_box,
                             XtNwidth, pix_width,
                             XtNheight, pix_height, NULL);
  XtAddCallback (channel->button, XtNcallback, button_cb,
                 (XtPointer *) channel);
  hotkey_channel (channel);
#endif
}

/* delete channel */
void
del_channel (int i)
{
#ifndef NO_X11
  XtDestroyWidget (channels[i]->button);
#endif
  free (channels[i]->name);
  if (channels[i]->key)
    free (channels[i]->key);
  free (channels[i]);
  count--;
  if (i < count)
    memmove (channels + i, channels + i + 1,
             (count - i) * sizeof (struct CHANNEL *));
}


void
update_channel(struct CHANNEL *dest, struct CHANNEL *src)
{
  Pixmap p;
  Widget b;
  int ck;

  p = dest->pixmap;
  b = dest->button;
  ck = dest->ckey;
  memcpy(dest, src, sizeof(struct CHANNEL));
  dest->pixmap = p;
  dest->button = b;
  dest->ckey = ck;
}

void
debug_channel(struct CHANNEL *chan)
{
  printf("\n");
  printf("channel.name        = %s\n", chan->name);
  printf("channel.key         = %s\n", chan->key);
  printf("channel.cname       = %s\n", chan->cname);
  printf("channel.channel     = %d\n", chan->channel);
  printf("channel.fine        = %d\n", chan->fine);
  printf("channel.freq        = %d\n", chan->freq);
  printf("channel.capture     = %s\n", int_to_str(chan->capture, captab));
  printf("channel.source      = %s\n", int_to_str(chan->source, grabbers[grabber]->inputs));
  printf("channel.norm        = %s\n", int_to_str(chan->norm, grabbers[grabber]->norms));
  printf("channel.color       = %d\n", chan->color);
  printf("channel.bright      = %d\n", chan->bright);
  printf("channel.hue         = %d\n", chan->hue);
  printf("channel.contrast    = %d\n", chan->contrast);
  printf("channel.pixmap      = %ld\n", chan->pixmap);
  printf("channel.button      = %p\n", chan->button);
  printf("channel.ckey        = %d\n", chan->ckey);                
  printf("channel.deinterlace = %d\n", chan->deinterlace);                
  printf("channel.subpage     = %s\n", chan->subpage);
  printf("\n");
}

/* ----------------------------------------------------------------------- */
int
read_channel(char *filename, char *line, int nr, struct CHANNEL **pt_current, char *tag, char *val, int flag)
{
  char *h;
  int i;
  struct CHANNEL *current = *pt_current;

  if (1 == sscanf (line, "[%99[^]]]", val))
    {
      if (flag == True)
        {
          current = add_channel (val);
          *pt_current = current;
        }
      else
        current->name = strdup(val);

      //continue;
      return 0;
    }

  if (2 != sscanf (line, " %31[^= ] = %99[^\n]", tag, val))
    {
      fprintf (stderr, "%s:%d: parse error\n", filename, nr);
      //continue;
      return 0;
    }

  for (h = val + strlen (val); h > val && *h == ' '; h--)
    *h = '\0';

  if (0 == strcmp (tag, "key"))
    {
      current->key = strdup (val);
    }
  else if (0 == strcmp (tag, "capture"))
    {
      if (-1 != (i = str_to_int (val, captab)))
        current->capture = i;
      else
        fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                 filename, nr, tag, val);
    }
  else if (0 == strcmp (tag, "source"))
    {
      if (-1 != (i = str_to_int (val, grabbers[grabber]->inputs)))
        current->source = i;
      else
        fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                 filename, nr, tag, val);
    }
  else if (0 == strcmp (tag, "norm"))
    {
      if (-1 != (i = str_to_int (val, grabbers[grabber]->norms)))
        current->norm = i;
      else
        fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                 filename, nr, tag, val);
    }
  else if (0 == strcmp (tag, "channel"))
    {
      current->cname = strdup (val);
      current->channel = lookup_channel (current->cname);
      if (-1 == current->channel)
        fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                 filename, nr, tag, val);
    }
  // Patch provided by Henk
  //This makes finetuning of freq. configured channels possible
  else if (0 == strcmp(tag, "freq"))
    {
      int n;
      unsigned d;

      current->freq = (int)(atof(val) * 1000);
      n = 0;
      d = (unsigned)-1;
      for (i = 0; i < CHAN_ENTRIES; i++)
        {
          if (abs(tvtuner[i].freq[chan_tab] - current->freq) < d)
            {
              d = abs(tvtuner[i].freq[chan_tab] - current->freq);
              n = i;
            }
        }
      current->cname = strdup(tvtuner[n].name);
      current->channel = lookup_channel(current->cname);
      current->fine = (current->freq - tvtuner[n].freq[chan_tab]) * 16 / 1000;
      current->freq = tvtuner[n].freq[chan_tab] * 16 / 1000;
      // debug_channel(current);
    }
  // End of Patch provided by Henk
  else if (0 == strcmp (tag, "fine"))
    {
      current->fine = atoi (val);
    }
  else if (0 == strcmp (tag, "color"))
    {
        current->color = atoi (val);
    }
  else if (0 == strcmp (tag, "bright"))
    {
      current->bright = atoi (val);
    }
  else if (0 == strcmp (tag, "hue"))
    {
      current->hue = atoi (val);
    }
  else if (0 == strcmp (tag, "contrast"))
    {
      current->contrast = atoi (val);
    }
  else if (0 == strcmp (tag, "deinterlace"))
    {
      if (-1 != (i = str_to_int (val, deinterlace_list)))
        current->deinterlace = i;
      else
        fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                 filename, nr, tag, val);
    }
  else if (0 == strcmp (tag, "subpage"))
    {
      current->subpage = strdup (val);
    }
  else
    {
      // nothing has been find
      return -1;
    }

  // something relevant has been find and hopefully interpreted
  return 0;
}

// this function read special parameters like decoration
// that must be extract before several inits
void
read_first(void)
{
  FILE *fp;
  char filename[100], line[100], tag[32], val[100];
  int nr = 0;
  
  sprintf (filename, "%s/%s", user_dir, "xdtvrc");
  fprintf (stderr, "filename = %s\n",filename);  
  fp = fopen (filename, "r");
  if (NULL == fp)
    return;
  
    while (NULL != fgets (line, 99, fp))
    {
      nr++;
      sscanf (line, " %31[^= ] = %99[^\n]", tag, val);
      if (line[0] == '\n' || line[0] == '#' || line[0] == '%')
        continue;
      if (0 == count && 0 == strcmp (tag, "decoration"))
        {
	  if ( 0 == strcmp (val, "off") )
	   decoration = 0;
	 else if ( 0 == strcmp (val, "on") )
	   decoration = 1;
	 else
	   fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		    filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "stayontop"))
        {
	  if ( 0 == strcmp (val, "off") )
	   stay = 0;
	 else if ( 0 == strcmp (val, "on") )
	   stay = 1;
	 else
	   fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		    filename, nr, tag, val);
        }
//#ifdef USE_LIBXOSD <= xosd inside now
      else if (0 == count && 0 == strcmp (tag, "xosd"))
        {
	  if ( 0 == strcmp (val, "off") )
	   libxosd = 0;
	 else if ( 0 == strcmp (val, "on") )
	   libxosd = 1;
	 else
	   fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		    filename, nr, tag, val);
        }
//#endif <= xosd inside now
      else if (0 == count && 0 == strcmp (tag, "subtitles"))
        {
	  if ( 0 == strcmp (val, "off") )
	   subactive = 0;
	 else if ( 0 == strcmp (val, "on") )
	   subactive = 1;
	 else
	   fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		    filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "fullscreen_mode"))
        {
	  if ( 0 == strcmp (val, "new") )
	   oldswitch = 0;
	 else if ( 0 == strcmp (val, "old") )
	   oldswitch = 1;
	 else
	   fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		    filename, nr, tag, val);
        }		
#ifdef HAVE_XV
      else if (0 == count && 0 == strcmp (tag, "xvport"))
	cmdline_xv_img_port = atoi(val);
#endif
      else if (0 == count && 0 == strcmp (tag, "mixer_dev"))
	{
	  mixer_dev = strdup (val);
	}
      else if (0 == count && 0 == strcmp (tag, "mixer_tvchan"))
	{
	  mixer_tvchan = strdup (val);
	}
      else if (0 == count && 0 == strcmp (tag, "mixer_pcmchan"))
	{
	  mixer_pcmchan = strdup (val);
	}
      else if (0 == count && 0 == strcmp (tag, "audio_dev"))
	{
	  audio_dev = strdup (val);
	}
      else if (0 == count && 0 == strcmp (tag, "restoresnd"))
	{
	  if ( 0 == strcmp (val, "off") )
	    restoresnd = 0;
	  else if ( 0 == strcmp (val, "on") )
	    restoresnd = 1;
	  else
	    fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		     filename, nr, tag, val);
	}
      else if (0 == count && 0 == strcmp (tag, "vop_autograb"))
	{
	  if ( 0 == strcmp (val, "off") )
	    vop_autograb = 0;
	  else if ( 0 == strcmp (val, "on") )
	    vop_autograb = 1;
	  else
	    fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		     filename, nr, tag, val);
	}
    }
  fclose (fp);
}

void
read_config (void)
{
  FILE *fp;
  char filename[100], line[100], tag[32], val[100], mytag[32];
  int i, j, nr = 0;
  char map[100];
  int rate = 0;
  float k = 0.0;
  struct CHANNEL *current = &defaults;

  if(int_to_str(current->norm,grabbers[grabber]->norms)==NULL)
    current->norm = 0;
  sprintf (filename, "%s/%s", user_dir, "xdtvrc");
  fp = fopen (filename, "r");
  if (NULL == fp)
    {
      fprintf (stderr, "Can't open config file %s:", filename);
      fprintf (stderr, "writing a default one.\n");
      fprintf (stderr, "Please edit %s to suit your needs\n", filename);
      /* Write a default config file */
      save_config ();

      /* And reread it ! */
      if ((fp = fopen (filename, "r")) == (FILE *) NULL)
        return;
    }
  have_config = 1;

  while (NULL != fgets (line, 99, fp))
    {
      nr++;
      if (line[0] == '\n' || line[0] == '#' || line[0] == '%')
        continue;

      if (read_channel(filename, line, nr, &current, tag, val, True) == 0)
        {
          continue;
        }
      else if (0 == count && 0 == strcmp (tag, "freqtab"))
        {
          if (-1 != (i = str_to_int (val, chan_names)))
            chan_tab = i;
          else {
	    /* temporary compatibilities with old names */
	    struct STRTAB freqtab_compat[] = {
	      {0, "ntsc-bcast"},
	      {1, "ntsc-cable"},
	      {2, "ntsc-bcast-jp"},
	      {3, "ntsc-cable-jp"},
	      {6, "pal-europe"},
	      {-1, NULL}};
	    if(-1 != (i = str_to_int (val, freqtab_compat))) {
	      fprintf(stderr, 
		      "freqtab=%s should be replaced by freqtab=%s\n",
		      val,int_to_str(i,chan_names));
	      chan_tab = i;
	    }
	  }
	  if(i==-1)
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "fullscreen"))
        {
          if (2 != sscanf (val, "%d x %d", &fs_width, &fs_height))
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
              fs_width = fs_height = 0;
            }
        }
      else if (0 == count && 0 == strcmp (tag, "windowsize"))
        {
          if (2 == sscanf (val, "%d x %d", &i, &j))
            {
              video_set_size (i, j);
            }
          else
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
            }
        }
      else if (0 == count && 0 == strcmp (tag, "pixsize"))
        {
	  if (2 == sscanf (val, "%d x %d", &pix_width, &pix_height))
	  {
            pix_height = pix_width * 3 / 4;
	  }
          else
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
              pix_width = 128;
              pix_height = 96;
            }
        }
      else if (0 == count && 0 == strcmp (tag, "wm-off-by"))
        {
          if (2 != sscanf (val, "%d %d", &fs_xoff, &fs_yoff))
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
              fs_xoff = fs_yoff = 0;
            }
        }
      else if (0 == count && 0 == strcmp (tag, "colorkey"))
        {
          if (1 == sscanf (val, "%d", &i))
            {
              set_colorkey(dpy, i);
            }
          else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }

      else if (0 == count && 0 == strcmp (tag, "capture_size"))
        {
          if (2 != sscanf (val, "%d x %d", &width_capture_max, &height_capture_max))
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
              width_capture_max = 768;
              height_capture_max = 576;
            }
        }

      else if (0 == count && 0 == strcmp (tag, "blackborder"))
        {
          if (1 == sscanf (val, "%d", &i))
            {
              set_ybar_ratio(i);
            }
	  else
	    fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
		     filename, nr, tag, val);
        }

      else if (0 == count && 0 == strcmp (tag, "codec"))
        {
          if (0 == strcmp(val,"UnCompressed"))
            {
              divx.codec = UNCOMPRESSED;
            }
#ifdef HAVE_DIVX4LINUX
          else if (0 == strcmp(val,"DivX"))
            {
              divx.codec = DIVX4LINUX;
            }
#endif
#ifdef HAVE_FFMPEG
          else if (0 == strcmp(val,"FFMPEG - Mpeg4"))
            {
              divx.codec = FFMPEG_MPEG4;
            }
          else if (0 == strcmp(val,"FFMPEG - Mpeg1Video"))
            {
              divx.codec = FFMPEG_MPEG1;
            }
#if LIBAVCODEC_BUILD >= 4734
          else if (0 == strcmp(val,"FFMPEG - Ffvhuff"))
            {
              divx.codec = FFMPEG_FFVHUFF;
            }
#endif
          else if (0 == strcmp(val,"FFMPEG - Huffyuv"))
            {
              divx.codec = FFMPEG_HUFFYUV;
            }
#endif
#ifdef HAVE_XVID
          else if (0 == strcmp(val,"XviD"))
            {
              divx.codec = XVID;
            }
#endif
          else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "width"))
        {
          if (1 != sscanf(val,"%d",&divx.width))
            {
	      fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "height"))
        {
          if (1 != sscanf(val,"%d",&divx.height))
            {
	      fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "bitrate"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
	      fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    divx.bitrate = i*1000;
	  }
        }
     else if (0 == count && 0 == strcmp (tag, "quality"))
       {
         switch(divx.codec)
	   {
	 #ifdef HAVE_DIVX4LINUX
	   case DIVX4LINUX:
	     if ( 0 == strcmp (val, "Fastest"))
	       divx.quality = 1;
	     else if ( 0 == strcmp (val, "Fast"))
	       divx.quality = 2;
	     else if ( 0 == strcmp (val, "Medium"))
	       divx.quality = 3;
	     else if ( 0 == strcmp (val, "Slow"))
	       divx.quality = 4;
	     else if ( 0 == strcmp (val, "Slowest"))
	       divx.quality = 5;
	     else
               fprintf (stderr, "%s:%d: invalid value for %s: %s\n", filename, nr, tag, val);
	     break;
	 #endif
	 #ifdef HAVE_FFMPEG
	   case FFMPEG_MPEG4:
	   case FFMPEG_MPEG1:
	     if ( 0 == strcmp (val, "none"))
	       divx.quality = 0;
	     else if ( 0 == strcmp (val, "full"))
	       divx.quality = 1;
	     else if ( 0 == strcmp (val, "log"))
	       divx.quality = 2;
	     else if ( 0 == strcmp (val, "phods"))
	       divx.quality = 3;
	     else if ( 0 == strcmp (val, "EPZS"))
	       divx.quality = 4;
	     else if ( 0 == strcmp (val, "X1"))
	       divx.quality = 5;
	     else
               fprintf (stderr, "%s:%d: invalid value for %s: %s\n", filename, nr, tag, val);
	     break;
	 #endif
	 #ifdef HAVE_XVID
	   case XVID:
	     if ( 0 == strcmp (val, "H623, No ME"))
	       divx.quality = 0;
	     else if ( 0 == strcmp (val, "MPEG, Low"))
	       divx.quality = 1;
	     else if ( 0 == strcmp (val, "H623, Low"))
	       divx.quality = 2;
	     else if ( 0 == strcmp (val, "H623, Medium"))
	       divx.quality = 3;
	     else if ( 0 == strcmp (val, "H623, High"))
	       divx.quality = 4;
	     else if ( 0 == strcmp (val, "Very High"))
	       divx.quality = 5;
	     else if ( 0 == strcmp (val, "Ultra High"))
	       divx.quality = 6;
	     else
               fprintf (stderr, "%s:%d: invalid value for %s: %s\n", filename, nr, tag, val);
	     break;
	 #endif
           }
	}
      else if (0 == count && 0 == strcmp (tag, "stereo_mode"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.stereo = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.stereo = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "compress_audio"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.compress_audio = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.compress_audio = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "mp3_bitrate"))
        {
          if (1 != sscanf(val,"%d",&divx.mp3_bitrate))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "mp3_quality"))
        {
          if (1 != sscanf(val,"%d",&divx.mp3_quality))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "mp3_vbr_mode"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.mp3_vbr = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.mp3_vbr = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "mp3_vbr_quality"))
        {
          if (1 != sscanf(val,"%d",&divx.mp3_vbr_quality))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "fps"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    divx.fps = i*1000;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "min_quantizer"))
        {
          if (1 != sscanf(val,"%d",&divx.min_quantizer))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "max_quantizer"))
        {
          if (1 != sscanf(val,"%d",&divx.max_quantizer))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "audio_buffer_size"))
        {
          if (1 != sscanf(val,"%d",&divx.audio_buffer_size))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "max_gap"))
        {
          if (1 != sscanf(val,"%f",&k))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    divx.maxgap = k/1000.0;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "record_delay"))
        {
          if (1 != sscanf(val,"%f",&k))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    divx.delay = k/1000.0;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "audio_fragments"))
        {
          if (1 != sscanf(val,"%d",&divx.audio_fragments))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "audio_sizefragment"))
        {
          if (1 != sscanf(val,"%d",&divx.audio_sizefragment))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
        }
      else if (0 == count && 0 == strcmp (tag, "display_frame"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.display_frames = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.display_frames = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "record_sub"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.sub = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.sub = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "record_chg"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.chg = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.chg = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "streaming_mode"))
        {
         if ( 0 == strcmp (val, "off") )
	   divx.stream = 0;
	 else if ( 0 == strcmp (val, "on") )
	   divx.stream = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
      else if (0 == count && 0 == strcmp (tag, "xawpopup"))
        {
         if ( 0 == strcmp (val, "off") )
	   xawpopup = 0;
	 else if ( 0 == strcmp (val, "on"))
	   xawpopup = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }	
      else if (0 == count && 0 == strcmp (tag, "streaming_http_port"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    divx.http_port = i;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "preview_player"))
        {
		pplayer = strdup(val);
        }	
      /* read divx file path into xdtvrc and give it to DivX GUI */
      else if (0 == count && 0 == strcmp (tag, "divx_path"))
	 divx.filename = strdup(val);
      else if (0 == count && 0 == strcmp (tag, "message_timer"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    message_timer = i;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "grab_filepath"))
        {
		filepath = strdup(val);
        }
#ifdef HAVE_LIBJPEG
      else if (0 == count && 0 == strcmp (tag, "jpeg_compression"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    if (i <= 100 && i >= 0)
	      jpegpcent = i;
	    else
	      jpegpcent = 100;
	  }
        }
#endif	
//#ifdef USE_LIBXOSD <= xosd inside now
      else if (0 == count && 0 == strcmp (tag, "xosd_main_color"))
        {
		colour = strdup(val);
        }
      else if (0 == count && 0 == strcmp (tag, "xosd_outline_color"))
        {
		outline_colour = strdup(val);
        }
      else if (0 == count && 0 == strcmp (tag, "xosd_outline_offset"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    outline_offset = i;
	  }
        }
      else if (0 == count && 0 == strcmp (tag, "xosd_shadow_offset"))
        {
          if (1 != sscanf(val,"%d",&i))
            {
!            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
	    }
	  else
	  {
	    shadow_offset = i;
	  }
        }	
      else if (0 == count && 0 == strcmp (tag, "xosd_main_font"))
        {
		font = strdup(val);
		if (debug)
		  fprintf(stderr,"Main font = %s\n",font);		
		get_font_infos (font, main_font, main_size);
		if (debug) {
		  fprintf(stderr,"Main font type = %s\n",main_font);
                  fprintf(stderr,"Main font size = %s\n",main_size); 
		}
        }
       else if (0 == count && 0 == strcmp (tag, "xosd_vtx_font"))
        {
		font_vtx = strdup(val);
		if (debug)		
		  fprintf(stderr,"Teletext font = %s\n",font_vtx);		
		get_font_infos (font_vtx, vtx_font, vtx_size);
		if (debug) {		
		  fprintf(stderr,"Teletext font type = %s\n",vtx_font);
		  fprintf(stderr,"Teletext font size = %s\n",vtx_size); 
		}
        }
//#endif <= xosd inside now

      /* search for eventmap stuff in the config file */
      else if (0 == count && 0 == strncmp (tag, "lirc-key-", 9))
        {
	    fprintf(stderr, "ev config event:%s\n", tag);
	    sscanf(tag, "%s", mytag);
	    strcpy(map, val);
	    if (debug)
		fprintf(stderr, "map:%s\n", map);
	    event_register(mytag, map, 0);
	}
      else if (0 == count && 0 == strcmp (tag, "repeat"))
    	{
	    sscanf(val, "%d", &rate);
	    if (debug)
		fprintf(stderr, "repeat_rate:%d\n", rate);
	    change_repeat_last_registered(rate);
        }

	/* search for alevt parameters into the config file */

      else if (0 == count && 0 == strcmp (tag, "alevt_defaultpage"))
        {
	 alevt_defaultpage_str = strdup(val);
        }
      else if (0 == count && 0 == strcmp (tag, "alevt_finetune_mode"))
        {
         if ( 0 == strcmp (val, "none") )
	   alevt_finetune_mode = 0;
	 else if ( 0 == strcmp (val, "auto") )
	   alevt_finetune_mode = 1;
	 else if ( 0 == strcmp (val, "manual") )
	   alevt_finetune_mode = 2;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }

      else if (0 == count && 0 == strcmp (tag, "alevt_finetune"))
        {
	 alevt_finetune_str = strdup(val);
        }
      else if (0 == count && 0 == strcmp (tag, "alevt_error_reduction"))
        {
         if ( 0 == strcmp (val, "yes") )
	   alevt_error_reduction = 0;
	 else if ( 0 == strcmp (val, "no") )
	   alevt_error_reduction = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }

      else if (0 == count && 0 == strcmp (tag, "alevt_error_bell"))
        {
         if ( 0 == strcmp (val, "yes") )
	   alevt_error_bell = 0;
	 else if ( 0 == strcmp (val, "no") )
	   alevt_error_bell = 1;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }

      else if (0 == count && 0 == strcmp (tag, "alevt_charset"))
        {
         if ( 0 == strcmp (val, "latin-1") )
	   alevt_charset = 0;
	 else if ( 0 == strcmp (val, "latin-2") )
	   alevt_charset = 1;
	 else if ( 0 == strcmp (val, "koi8-r") )
	   alevt_charset = 2;
	 else if ( 0 == strcmp (val, "iso8859-7") )
	   alevt_charset = 3;
	 else
            fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                     filename, nr, tag, val);
        }
    }
  fclose (fp);

  /* calculate channel frequencies */
  defaults.channel = lookup_channel (defaults.cname);
  defaults.freq = get_freq (defaults.channel) + defaults.fine;
  for (i = 0; i < count; i++)
    configure_channel (channels[i]);
}

/* ----------------------------------------------------------------------- */

void
save_channel(FILE *fp, int i)
{
  fprintf (fp, "[%s]\n", channels[i]->name);

  fprintf (fp, "channel = %s\n", tvtuner[channels[i]->channel].name);

  if (defaults.fine != channels[i]->fine)
    fprintf (fp, "fine = %d\n", channels[i]->fine);

  if (defaults.norm != channels[i]->norm)
    fprintf (fp, "norm = %s\n",
             int_to_str (channels[i]->norm, grabbers[grabber]->norms));

  if (channels[i]->key != NULL)
    fprintf (fp, "key = %s\n", channels[i]->key);

  if (defaults.capture != channels[i]->capture)
    fprintf (fp, "capture = %s\n",
             int_to_str (channels[i]->capture, captab));

  if (defaults.source != channels[i]->source)
    fprintf (fp, "source = %s\n",
             int_to_str (channels[i]->source, grabbers[grabber]->inputs));

  if (defaults.color != channels[i]->color)
    fprintf (fp, "color = %d\n", channels[i]->color);

  if (defaults.bright != channels[i]->bright)
    fprintf (fp, "bright = %d\n", channels[i]->bright);

  if (defaults.hue != channels[i]->hue)
    fprintf (fp, "hue = %d\n", channels[i]->hue);

  if (defaults.contrast != channels[i]->contrast)
    fprintf (fp, "contrast = %d\n", channels[i]->contrast);
  
  if (defaults.deinterlace != channels[i]->deinterlace)
    fprintf (fp, "deinterlace = %s\n", int_to_str(channels[i]->deinterlace, deinterlace_list));

  if (strcmp(defaults.subpage, channels[i]->subpage))
    fprintf (fp, "subpage = %s\n", channels[i]->subpage); // page 888 par defaut
   
  fprintf (fp, "\n");
}

void
save_alevt(FILE *fp)
{
  fprintf(fp,"\n");
  fprintf (fp, "#\n# alevt options\n#\n");
  update_defaultpage();
  fprintf (fp,"alevt_defaultpage = %s\n",alevt_defaultpage_str);

  switch(alevt_finetune_mode)
  {
    case 0:
      fprintf (fp,"alevt_finetune_mode = none\n");
      break;
    case 1:
      fprintf (fp,"alevt_finetune_mode = auto\n");
      break;
    case 2:
      fprintf (fp,"alevt_finetune_mode = manual\n");
      break;
  }
  update_finetune();
  fprintf (fp,"alevt_finetune = %s\n",alevt_finetune_str);
    switch(alevt_error_reduction)
  {
    case 0:
      fprintf (fp,"alevt_error_reduction = yes\n");
      break;
    case 1:
      fprintf (fp,"alevt_error_reduction = no\n");
      break;
  }
    switch(alevt_error_bell)
  {
    case 0:
      fprintf (fp,"alevt_error_bell = yes\n");
      break;
    case 1:
      fprintf (fp,"alevt_error_bell = no\n");
      break;
  }
  switch(alevt_charset)
  {
    case 0:
      fprintf (fp,"alevt_charset = latin-1\n");
      break;
    case 1:
      fprintf (fp,"alevt_charset = latin-2\n");
      break;
    case 2:
      fprintf (fp,"alevt_charset = koi8-r\n");
      break;
    case 3:
      fprintf (fp,"alevt_charset = iso8859-7\n");
      break;
  }
}

void
save_record(FILE *fp)
{
  fprintf (fp, "#\n# record options\n#\n");
  switch(divx.codec)
  {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      fprintf (fp, "codec = DivX\n");
      break;
#endif

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
      fprintf (fp, "codec = FFMPEG - Mpeg4\n");
      break;
    case FFMPEG_MPEG1:
      fprintf (fp, "codec = FFMPEG - Mpeg1Video\n");
      break;
#if LIBAVCODEC_BUILD >= 4734
    case FFMPEG_FFVHUFF:
      fprintf (fp, "codec = FFMPEG - Ffvhuff\n");
      break;
#else
    case FFMPEG_HUFFYUV:
      fprintf (fp, "codec = FFMPEG - Huffyuv\n");
      break;
#endif
#endif

#ifdef HAVE_XVID
    case XVID:
      fprintf (fp, "codec = XviD\n");
      break;
#endif
  case UNCOMPRESSED:
      fprintf (fp, "codec = UnCompressed\n");
      break;
   }
   fprintf (fp,"width = %d\n",divx.width);
   fprintf (fp,"height = %d\n",divx.height);
   fprintf (fp,"bitrate = %ld\n",divx.bitrate/1000);
  switch(divx.codec)
  {
#ifdef HAVE_DIVX4LINUX
    case DIVX4LINUX:
      switch(divx.quality)
      {
      	case 1:
      	 fprintf (fp,"quality = Fastest\n");
      	 break;
      	case 2:
      	 fprintf (fp,"quality = Fast\n");
      	 break;
      	case 3:
      	 fprintf (fp,"quality = Medium\n");
      	 break;
      	case 4:
      	 fprintf (fp,"quality = Slow\n");
      	 break;
      	case 5:
      	 fprintf (fp,"quality = Slowest\n");
      	 break;
      }
      break;
#endif

#ifdef HAVE_FFMPEG
    case FFMPEG_MPEG4:
    case FFMPEG_MPEG1:
      switch(divx.quality)
      {
      	case 0:
      	 fprintf (fp,"quality = none\n");
      	 break;
      	case 1:
      	 fprintf (fp,"quality = full\n");
      	 break;
      	case 2:
      	 fprintf (fp,"quality = log\n");
      	 break;
      	case 3:
      	 fprintf (fp,"quality = phods\n");
      	 break;
      	case 4:
      	 fprintf (fp,"quality = EPZS\n");
      	 break;
      	case 5:
      	 fprintf (fp,"quality = X1\n");
      	 break;
      }
      break;
#endif

#ifdef HAVE_XVID
    case XVID:
      switch(divx.quality)
      {
      	case 0:
      	 fprintf (fp,"quality = H623, No ME\n");
      	 break;
      	case 1:
      	 fprintf (fp,"quality = MPEG, Low\n");
      	 break;
      	case 2:
      	 fprintf (fp,"quality = H623, Low\n");
      	 break;
      	case 3:
      	 fprintf (fp,"quality = H623, Medium\n");
      	 break;
      	case 4:
      	 fprintf (fp,"quality = H623, High\n");
      	 break;
      	case 5:
      	 fprintf (fp,"quality = Very High\n");
      	 break;
      	case 6:
      	 fprintf (fp,"quality = Ultra High\n");
      	 break;
      }
      break;
#endif
   }
   switch(divx.stereo)
   {
     case 0:
       fprintf (fp,"stereo_mode = off\n");
       break;
     case 1:
       fprintf (fp,"stereo_mode = on\n");
       break;
   }
   switch(divx.compress_audio)
   {
     case 0:
       fprintf (fp,"compress_audio = off\n");
       break;
     case 1:
       fprintf (fp,"compress_audio = on\n");
       break;
   }
   fprintf (fp,"mp3_bitrate = %d\n",divx.mp3_bitrate);
   fprintf (fp,"mp3_quality = %d\n",divx.mp3_quality);
   switch(divx.mp3_vbr)
   {
     case 0:
       fprintf (fp,"mp3_vbr_mode = off\n");
       break;
     case 1:
       fprintf (fp,"mp3_vbr_mode = on\n");
       break;
   }
   fprintf (fp,"mp3_vbr_quality = %d\n",divx.mp3_vbr_quality);
   fprintf (fp,"fps = %d\n",divx.fps/1000);
   fprintf (fp,"max_gap = %f\n",divx.maxgap*1000.0);
   fprintf (fp,"audio_fragments = %d\n",divx.audio_fragments);
   fprintf (fp,"audio_sizefragment = %d\n",divx.audio_sizefragment);
   fprintf (fp,"min_quantizer = %d\n",divx.min_quantizer);
   fprintf (fp,"max_quantizer = %d\n",divx.max_quantizer);
   fprintf (fp,"audio_buffer_size = %d\n",divx.audio_buffer_size);

/* write divx file path into xdtvrc */
   update_divxfilename();
   if(divx.filename)
     fprintf (fp,"divx_path = %s\n",divx.filename);

   switch(divx.display_frames)
   {
     case 0:
       fprintf (fp,"display_frame = off\n");
       break;
     case 1:
       fprintf (fp,"display_frame = on\n");
       break;
   }
   switch(divx.sub)
   {
     case 0:
       fprintf (fp,"record_sub = off\n");
       break;
     case 1:
       fprintf (fp,"record_sub = on\n");
       break;
   }
   switch(divx.chg)
   {
     case 0:
       fprintf (fp,"record_chg = off\n");
       break;
     case 1:
       fprintf (fp,"record_chg = on\n");
       break;
   }
   fprintf (fp,"record_delay = %f\n",divx.delay*1000.0);

   switch(divx.stream)
   {
     case 0:
       fprintf (fp,"streaming_mode = off\n");
       break;
     case 1:
       fprintf (fp,"streaming_mode = on\n");
       break;
   }
   update_http_port();
   fprintf (fp,"streaming_http_port = %d\n",divx.http_port);
   
   if (pplayer)
     fprintf (fp, "preview_player = %s\n", pplayer);   
   
   fprintf(fp,"\n");
}

void
save_config (void)
{
  char filename1[100], filename2[100];
  FILE *fp;
  int i;
  Dimension w = 0;
  Dimension h = 0;

  sprintf (filename1, "%s/%s", user_dir, "xdtvrc");
  sprintf (filename2, "%s/%s", user_dir, "xdtvrc.bak");

  /* delete old backup */
  unlink (filename2);

  /* current becomes backup */
  if (0 == link (filename1, filename2))
    unlink (filename1);

  /* write new one... */
  fp = fopen (filename1, "w");
  if (NULL == fp)
    {
      fprintf (stderr, "can't open config file %s\n", filename1);
      return;
    }

  /* write defaults */
  fprintf(fp, "#\n# Global options\n#\n");
  if(int_to_str (defaults.norm, grabbers[grabber]->norms)==NULL)
    defaults.norm = 0;
  fprintf (fp, "norm = %s\n",
           int_to_str (defaults.norm, grabbers[grabber]->norms));

  fprintf (fp, "capture = %s\n", int_to_str (defaults.capture, captab));

  fprintf (fp, "source = %s\n",
           int_to_str (defaults.source, grabbers[grabber]->inputs));

  if ((defaults.color != 19000) && (defaults.color != 0))
    fprintf (fp, "color = %d\n", defaults.color);

  if ((defaults.bright != 34000) && (defaults.bright != 0))
    fprintf (fp, "bright = %d\n", defaults.bright);

  if ((defaults.hue != 32768) && (defaults.hue != 0))
    fprintf (fp, "hue = %d\n", defaults.hue);

  if ((defaults.contrast != 28000) && (defaults.contrast != 0))
    fprintf (fp, "contrast = %d\n", defaults.contrast);

  fprintf (fp, "subpage = %s\n", defaults.subpage);

  fprintf (fp, "freqtab = %s\n", int_to_str (chan_tab, chan_names));  
  
  fprintf (fp, "\n");   
  
  if (mixer_dev)
    fprintf (fp, "mixer_dev = %s\n", mixer_dev);
  if (mixer_tvchan)
    fprintf (fp, "mixer_tvchan = %s\n", mixer_tvchan);
  if (mixer_pcmchan)
    fprintf (fp, "mixer_pcmchan = %s\n", mixer_pcmchan);
  if (audio_dev)
    fprintf (fp, "audio_dev = %s\n", audio_dev);
  
  fprintf (fp, "restoresnd = %s\n", restoresnd ? "on" : "off");  

  fprintf (fp, "vop_autograb = %s\n", vop_autograb ? "on" : "off");  

  fprintf (fp, "\n");     
      
  if (xawpopup)
    fprintf (fp, "xawpopup = on\n");
  else
    fprintf (fp, "xawpopup = off\n");

  if (message_timer)
    fprintf (fp, "message_timer = %d\n", message_timer);     
    
  if (decoration)
    fprintf (fp, "decoration = on\n");    
  else
    fprintf (fp, "decoration = off\n");
    
  if (stay)
    fprintf (fp, "stayontop = on\n");    
  else
    fprintf (fp, "stayontop = off\n");    
         
  if (subactive)
    fprintf (fp, "subtitles = on\n");    
  else
    fprintf (fp, "subtitles = off\n");        

#ifdef HAVE_XV
  if(cmdline_xv_img_port)
    fprintf(fp, "xvport = %d\n", cmdline_xv_img_port);
#endif

  fprintf (fp, "\n");  
  fprintf(fp, "#\n# Fullscreen options\n#\n");    
  
  if (fs_width && fs_height)
    fprintf (fp, "fullscreen = %d x %d\n", fs_width, fs_height);
  else
    fprintf (fp, "fullscreen = 768 x 576\n");
    
  if (oldswitch)
    fprintf (fp, "fullscreen_mode = old\n");    
  else
    fprintf (fp, "fullscreen_mode = new\n");    

  update_wmoffby();
  if (fs_xoff || fs_yoff)
    fprintf (fp, "wm-off-by = %+d%+d\n", fs_xoff, fs_yoff);

  update_pixmaps_size();
  fprintf (fp, "pixsize = %d x %d\n", pix_width, pix_height);

  if (fs_width && fs_height)
  {
    XtVaGetValues (tv, XtNwidth, &w, XtNheight, &h, NULL);
    fprintf (fp, "windowsize = %d x %d\n", w, h);
  }
  else fprintf (fp, "windowsize = 384 x 288\n");
  
  update_colorkey();
  if ((i = get_colorkey(dpy)) >= 0)
    fprintf (fp, "colorkey = %d\n", i);
  else
    fprintf (fp, "colorkey = 123456\n");    

  if (width_capture_max && height_capture_max)
    fprintf (fp, "capture_size = %d x %d\n", width_capture_max, height_capture_max);
  else
    /*maybe use CAPT_WIDTH_MAX, CAPT_HEIGHT_MAX instead*/
    fprintf (fp, "capture_size = 768 x 576\n");

  if (get_ybar_ratio())
    fprintf (fp, "blackborder = %d\n", get_ybar_ratio());

  fprintf (fp, "\n");
  fprintf(fp, "#\n# Grab options\n#\n");
  
  update_grabfilename();
  if (filepath)
    fprintf (fp, "grab_filepath = %s\n", filepath);
  else
    fprintf (fp, "grab_filepath = %s\n", getcwd(NULL,0));
#ifdef HAVE_LIBJPEG    
  if (jpegpcent)
    fprintf (fp, "jpeg_compression = %d\n", jpegpcent);  
#endif  
//#ifdef USE_LIBXOSD <= xosd inside now

  fprintf (fp, "\n");
  fprintf(fp, "#\n# XOSD options\n#\n");  

  if (libxosd)
    fprintf (fp, "xosd = on\n");    
  else
    fprintf (fp, "xosd = off\n");  
  
  if (colour)
    fprintf (fp, "xosd_main_color = %s\n", colour);

  if (outline_colour)
    fprintf (fp, "xosd_outline_color = %s\n", outline_colour);    

  if (outline_offset)
    fprintf (fp, "xosd_outline_offset = %d\n", outline_offset);     

  if (shadow_offset)
    fprintf (fp, "xosd_shadow_offset = %d\n", shadow_offset);       
            
  if (font)
    fprintf (fp, "xosd_main_font = %s\n", font);

  if (font_vtx)
    fprintf (fp, "xosd_vtx_font = %s\n", font_vtx);
//#endif <= xosd inside now

  fprintf (fp, "\n");

  save_record(fp);

  save_user_events(fp);

  save_alevt(fp);

  fprintf (fp, "\n");

  /* write channels */
  if (count > 0)
    {
      fprintf(fp, "#\n# Channel options\n#\n");
      for (i = 0; i < count; i++)
        {
          save_channel(fp, i);
        }
    }
  else
    {
      fprintf (fp,
               "# For each channel define at least following information\n");
      fprintf (fp, "# [Channel Name]\n");
      fprintf (fp, "# channel = Channel Number\n");
      fprintf (fp, "# norm = PAL|NTSC|SECAM|AUTO\n");
      fprintf (fp,
               "# key = key to press to switch the channel on (KP_End, KP_Next,...)\n");
      fprintf (fp, "# capture = off|grabdisplay|overlay\n");
      fprintf (fp, "\n");
    }



  fclose (fp);
}

int
load_last_channel(struct CHANNEL *last_channel)
{
  FILE *fp;
  char filename[100], line[100], tag[32], val[100];
  int nr = 0;
  int r, status = 0;
  int w, h;

  sprintf (filename, "%s/%s", user_dir, "last_channel");
  fp = fopen (filename, "r");
  if (NULL == fp)
    {
      fprintf(stderr, "load_last_channel: can't open %s\n", filename);
      return -1;
    }

  memcpy(last_channel, &defaults, sizeof(struct CHANNEL));

  while (NULL != fgets (line, 99, fp))
    {
      nr++;
      if (line[0] == '\n' || line[0] == '#' || line[0] == '%')
        continue;


      if ((r=read_channel(filename, line, nr, &last_channel, tag, val, False)) == 0)
        {
          continue;
        }
      else if (0 == strcmp (tag, "windowsize"))
        {
          if (2 == sscanf (val, "%d x %d", &w, &h))
            {
              video_set_size (w, h);
            }
          else
            {
              fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                       filename, nr, tag, val);
            }
        }
      else
        {
          // Nothing was found;
          status += r;
        }
    }
  fclose(fp);

  if (nr == 0)
    return -1; // file was empty !!!
  else if (status == 0)
    {
      /* calculate channel frequencies */
      last_channel->freq = get_freq (last_channel->channel) + last_channel->fine;
   //   hotkey_channel (last_channel);
      return 0; // everything went fine
    }
  else
    return -1;
}

void
save_last_channel(int last_channel)
{
  FILE *fp;
  char filename[100];
  //Dimension w, h;
  
  sprintf (filename, "%s/%s", user_dir, "last_channel");
  fp = fopen (filename, "w");
  if (NULL == fp)
    {
      fprintf(stderr, "save_last_channel: can't open %s\n", filename);
      return;
    }

  // Causes more troubles than real added value
  //XtVaGetValues (tv, XtNwidth, &w, XtNheight, &h, NULL);
  //fprintf (fp, "windowsize = %d x %d\n", w, h);

  save_channel(fp, last_channel);

  fclose(fp);
}


void
save_memcpy_method(int method)
{
  FILE *fp;
  char filename[100];

  sprintf (filename, "%s/%s", user_dir, "memcpy_method");
  fp = fopen (filename, "w");

  fprintf(fp, "# memcpy_method: probe | glibc | kernel | FPU_xx | mmx | mmx_xx | mmxext | mmxext_xx | sse | sse_xx\n"
              "version = %s\n"
              "memcpy_method = %s\n", VERSION, int_to_str(method, memcpy_list));

  fclose(fp);
}


void
load_memcpy_method(void)
{
  FILE *fp;
  char filename[100];
  char line[128];
  int nr;
  char *h;
  char tag[32], val[128];
  int i;
  // Version found in file vs current version
  int version=0; // 0: not found, 1: same, -1: different

  sprintf (filename, "%s/%s", user_dir, "memcpy_method");
  fp = fopen (filename, "r");
  if (NULL == fp)
    {
      fprintf(stderr, "load_memcpy_method: can't open %s\n", filename);
      return;
    }

  nr = 0;
  while (NULL != fgets (line, 127, fp))
    {
      nr++;
      if (line[0] == '\n' || line[0] == '#' || line[0] == '%')
        continue;

      if (2 != sscanf (line, " %31[^= ] = %127[^\n]", tag, val))
        {
          fprintf (stderr, "%s:%d: parse error\n", filename, nr);
          continue;
        }

      for (h = val + strlen (val); h > val && *h == ' '; h--)
        *h = '\0';
      

      if (0 == strcmp (tag, "memcpy_method"))
        { // Found a method
          if (version != -1)
          { // version "not found" or "same"
              if (-1 != (i = str_to_int (val, memcpy_list)))
                { // Found a licit method
                  memcpy_method = i;
                  fprintf (stderr, "Method %s\n", int_to_str (i, memcpy_list)) ;
                }
              else
                { // Found an illicit method
                  memcpy_method = MEMCPY_PROBE;
                  fprintf (stderr, "%s:%d: invalid value for %s: %s\n",
                           filename, nr, tag, val);
                }
          }
        }

       if (0 == strcmp (tag, "version"))
        { // Found a version
          if (strcmp(val, VERSION))
            { // Not the same version, have to probe
              memcpy_method = MEMCPY_PROBE;
              version = -1; // version set to "different"
              fprintf (stderr, "Changing version from %s to %s: probing memcpy methods\n",
                       val, VERSION);
            }
          else
            {
                version = 1; // version set to "same"
            }
        }
    }

    if (!version)
    { // No version found after all, have to probe
     memcpy_method = MEMCPY_PROBE;
     fprintf (stderr, "New memcpy methods available: probing methods\n");
    }
  fclose(fp);
}
