/*
 * xmms-mad - mp3 plugin for xmms
 * Copyright (C) 2001-2002 Sam Clegg - See COPYING
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 * $Id: xmms-mad.c,v 1.18 2006/01/05 10:38:59 sam Exp $
 */
#include "config.h"
#include "xmms-mad.h"
#include "input.h"

#include <math.h> // pow

#include <pthread.h>
#include <gtk/gtk.h>
#include <xmms/util.h>
#include <xmms/configfile.h>
#include <stdarg.h>
#include <fcntl.h>

/*#define DEBUG*/

/*
 * Global variables
 */
InputPlugin mad_plugin;   /**< the plugin itself */
struct xmmsmad_config_t xmmsmad_config;   /**< global configuration */

/*
 * static variables
 */
static pthread_t *decode_thread; /**< the single decoder thread */
static struct mad_info_t info;   /**< info for current track */

#ifndef NOGUI
static GtkWidget *about_window = 0;
static GtkWidget *error_dialog = 0;
#endif /* !NOGUI */

void xmmsmad_config_compute(struct xmmsmad_config_t * config) {
  /* set some config parameters by parsing text fields 
     (RG default gain, etc..)
  */
  const gchar * text;
  gdouble x;

  text = config->pregain_db;
  x = g_strtod(text, NULL);
  config->pregain_scale = (x!=0) ? pow(10.0, x/20) : 1;
#ifdef DEBUG
  g_message("pregain=[%s] -> %g  -> %g", text, x, config->pregain_scale);
#endif
  text = config->replaygain.default_db;
  x = g_strtod(text, NULL);
  config->replaygain.default_scale = (x!=0) ? pow(10.0, x/20) : 1;
#ifdef DEBUG
  g_message("RG.default=[%s] -> %g  -> %g", text, x, config->replaygain.default_scale);
#endif
}


static void
xmmsmad_init ()
{
  ConfigFile *cfg;
  xmmsmad_config.http_buffer_size = HTTP_BUFFER_SIZE;
  xmmsmad_config.fast_play_time_calc = TRUE;
  xmmsmad_config.use_xing = TRUE;
  xmmsmad_config.dither = TRUE;
  xmmsmad_config.pregain_db = "+0.00";
  xmmsmad_config.replaygain.enable = TRUE;
  xmmsmad_config.replaygain.track_mode = FALSE;
  xmmsmad_config.hard_limit = FALSE;
  xmmsmad_config.replaygain.default_db = "-9.00";

  cfg = xmms_cfg_open_default_file();
  if (cfg)
    {
      xmms_cfg_read_int(cfg, "MAD", "http_buffer_size", &xmmsmad_config.http_buffer_size);   
      xmms_cfg_read_boolean(cfg, "MAD", "fast_play_time_calc", &xmmsmad_config.fast_play_time_calc);
      xmms_cfg_read_boolean(cfg, "MAD", "use_xing", &xmmsmad_config.use_xing);
      xmms_cfg_read_boolean(cfg, "MAD", "dither", &xmmsmad_config.dither);
      xmms_cfg_read_boolean(cfg, "MAD", "hard_limit", &xmmsmad_config.hard_limit);
      xmms_cfg_read_string(cfg, "MAD", "pregain_db", &xmmsmad_config.pregain_db);
      xmms_cfg_read_boolean(cfg, "MAD", "RG.enable", &xmmsmad_config.replaygain.enable);
      xmms_cfg_read_boolean(cfg, "MAD", "RG.track_mode", &xmmsmad_config.replaygain.track_mode);
      xmms_cfg_read_string(cfg, "MAD", "RG.default_db", &xmmsmad_config.replaygain.default_db);
    }
  xmmsmad_config_compute(&xmmsmad_config);
}

static void
xmmsmad_cleanup ()
{
}

static void
xmmsmad_set_eq (int on, float preamp, float *bands)
{
  //printf("set_eq\n");
}

static int mpg123_head_check(unsigned long head)
{
    /*
     * First two bytes must be a sync header (11 bits all 1)
     * http://www.mp3-tech.org/programmer/frame_header.html
     */
    if ((head & 0xffe00000) != 0xffe00000)
      return FALSE;
    if (!((head >> 17) & 3))
      return FALSE;
    if (((head >> 12) & 0xf) == 0xf)
      return FALSE;
    if (!((head >> 12) & 0xf))
      return FALSE;
    if (((head >> 10) & 0x3) == 0x3)
      return FALSE;
    if (((head >> 19) & 1) == 1 &&
        ((head >> 17) & 3) == 3 &&
        ((head >> 16) & 1) == 1)
      return FALSE;
    if ((head & 0xffff0000) == 0xfffe0000)
      return FALSE;
    return TRUE;
}

static int
xmmsmad_is_our_file (char *filename)
{
  int rtn = 0;
  guchar check [4];

  if (strncasecmp ("http://", filename, 7) == 0)
    {
      if (strcasecmp ("mp3", filename + strlen (filename) - 3) == 0)
        {
          /* [FIXME] do some checks here to make sure we have an mp3 stream */
          rtn = 1;
        }
    }
  else
    {
      /* I've seen some flac files beginning with id3 frames..
         so let's exclude known non-mp3 filename extensions */
      if (strcasecmp ("flac", filename + strlen (filename) - 4) == 0 ||
          strcasecmp ("mpc", filename + strlen (filename) - 3) == 0 )
        {
          rtn = 0;
        }
      int fin = open (filename, O_RDONLY);
      if (fin >= 0 && read (fin, check, 4) == 4)
	{
          /*
	   * or three bytes are "ID3"
	   */
	  if (mpg123_head_check(*(unsigned long*)check))
	    {
	      rtn = 1;
	    }
          else if (memcmp (check, "ID3", 3) == 0)
            {
	      rtn = 1;
            }
	  else if (memcmp (check, "RIFF", 4) == 0) /* RIFF data */
	    {
	      lseek (fin, 4, SEEK_CUR);
	      read (fin, check, 4);

	      if (memcmp (check, "RMP3", 4) == 0)
		{
		  rtn = 1;
		}
#if 0
	      /* FIXME: make this test endian safe */
	      else if (memcmp (check, "WAVE"))
	        {
		  /* seek to codec ID and see if it's MP3 */
		  lseek (fin, 20, SEEK_SET);
		  check = 0L;
		  read (fin, &check, 2);
		  if (check == 0x55)
		    {
		      rtn = 1;
		    }
		}
#endif 
	    }
	}
      close (fin);
    }

  return rtn;
}

static void
xmmsmad_stop ()
{
#ifdef DEBUG
  g_message ("f: xmmsmad_stop");
#endif /* DEBUG */
  if (decode_thread)
    {
      info.stop = 1;
#ifdef DEBUG
      g_message ("waiting for thread");
#endif /* DEBUG */
      pthread_join (*decode_thread, 0);
#ifdef DEBUG
      g_message ("thread done");
#endif /* DEBUG */
      g_free (decode_thread);
      input_term (&info);
      decode_thread = 0;
      info.stop = 0;
    }
}

static void
xmmsmad_play_file (char *url)
{
  gboolean rtn;
  xmmsmad_stop ();

#ifdef DEBUG
  g_message ("playing %s", url);
#endif /* DEBUG */

  if (input_init (&info, url) == FALSE)
    {
      g_message ("error initialising input");
      return;
    }

  rtn = input_get_info (&info, xmmsmad_config.fast_play_time_calc);

  if (rtn == FALSE)
    {
      g_message ("error reading input info");
      return;
    }

  decode_thread = g_malloc (sizeof (pthread_t));
  if (!decode_thread)
    {
      g_message ("error creating decode thread");
      return;
    }

  if (pthread_create (decode_thread, 0, decode, (void *) &info))
    {
      /* malloc succeded but pthread_create failed */
      g_message ("error creating decode thread");
      g_free (decode_thread);
      decode_thread = 0;
      return;
    }
}

static void
xmmsmad_pause (short paused)
{
  mad_plugin.output->pause (paused);
}

static void
xmmsmad_seek (int time)
{
  /* xmms gives us the desired seek time in seconds */
  info.seek = time;
}

static int
xmmsmad_get_time ()
{
  if (info.stop)
    {
#ifdef DEBUG
      g_message ("returning -1 from get_time");
#endif
      return -1;
    }
  return mad_plugin.output->output_time ();
}

/**
 * Scan the given file or URL.
 * Fills in the title string and the track length in milliseconds.
 */
static void
xmmsmad_get_song_info (char *url, char **title, int *length)
{
  struct mad_info_t myinfo;
#ifdef DEBUG
  g_message ("f: xmmsmad_get_song_info: %s", url);
#endif /* DEBUG */
  input_init (&myinfo, url);
  if (input_get_info (&myinfo, TRUE) == TRUE)
    {
      *title  = strdup (myinfo.title);
      *length = mad_timer_count (myinfo.duration, MAD_UNITS_MILLISECONDS);
    }
  else
    {
      *title = strdup (url);
      *length = -1;
    }
  input_term (&myinfo);
#ifdef DEBUG
  g_message ("e: xmmsmad_get_song_info");
#endif /* DEBUG */
}

/**
 * Pop-up GTK about box.
 * Taken from mpg123 plugin.
 */
static void
xmmsmad_about ()
{
#ifndef NOGUI
  GtkWidget *dialog_vbox1;
  GtkWidget *hbox1;
  GtkWidget *label1;
  GtkWidget *dialog_action_area1;
  GtkWidget *about_exit;

  if (!about_window)
    {
      about_window = gtk_dialog_new ();
      gtk_object_set_data (GTK_OBJECT (about_window), "about_window", about_window);
      gtk_window_set_title (GTK_WINDOW (about_window), "About MPEG Layer 1/2/3 plugin (support 3DNow!)");
      gtk_window_set_policy (GTK_WINDOW (about_window), FALSE, FALSE, FALSE);
      gtk_window_set_position (GTK_WINDOW (about_window), GTK_WIN_POS_MOUSE);
      gtk_signal_connect (GTK_OBJECT (about_window), "destroy", GTK_SIGNAL_FUNC (gtk_widget_destroyed), &about_window);
      gtk_container_border_width (GTK_CONTAINER (about_window), 10);

      dialog_vbox1 = GTK_DIALOG (about_window)->vbox;
      gtk_object_set_data (GTK_OBJECT (about_window), "dialog_vbox1", dialog_vbox1);
      gtk_widget_show (dialog_vbox1);
      gtk_container_border_width (GTK_CONTAINER (dialog_vbox1), 5);

      hbox1 = gtk_hbox_new (FALSE, 0);
      gtk_object_set_data (GTK_OBJECT(about_window), "hbox1", hbox1);
      gtk_widget_show (hbox1);
      gtk_box_pack_start (GTK_BOX(dialog_vbox1), hbox1, TRUE, TRUE, 0);
      gtk_container_border_width (GTK_CONTAINER(hbox1), 5);
      gtk_widget_realize (about_window);

      label1 = gtk_label_new (DESCRIPTION " v" VERSION " by Sam Clegg\nhttp://www.superduper.net/xmms-mad/\nWith SKR mods (v1.20)\nhttp://www.crans.org/~krempp/xmms-mad/\nBased on libmad v" MAD_VERSION " by Rob Leslie\nhttp://mad.sourceforge.net");
      gtk_object_set_data (GTK_OBJECT (about_window), "label1", label1);
      gtk_widget_show (label1);
      gtk_box_pack_start (GTK_BOX (hbox1), label1, TRUE, TRUE, 0);

      dialog_action_area1 = GTK_DIALOG (about_window)->action_area;
      gtk_object_set_data (GTK_OBJECT (about_window), "dialog_action_area1", dialog_action_area1);
      gtk_widget_show (dialog_action_area1);
      gtk_container_border_width (GTK_CONTAINER (dialog_action_area1), 10);

      about_exit = gtk_button_new_with_label ("Ok");
      gtk_signal_connect_object (GTK_OBJECT (about_exit), "clicked",
                                 GTK_SIGNAL_FUNC (gtk_widget_destroy),
                                 GTK_OBJECT (about_window));

      gtk_object_set_data (GTK_OBJECT (about_window), "about_exit", about_exit);
      gtk_widget_show (about_exit);
      gtk_box_pack_start (GTK_BOX (dialog_action_area1), about_exit, TRUE, TRUE, 0);

      gtk_widget_show (about_window);
    }
  else
    {
      gdk_window_raise (about_window->window);
    }
#endif /* !NOGUI */
}

/**
 * Display a GTK box containing the given error message.
 * Taken from mpg123 plugin.
 */
void
xmmsmad_error (char *error, ...)
{
#ifndef NOGUI
  if (!error_dialog)
    {
      va_list args;
      char string[256];
      va_start (args, error);
      vsnprintf (string, 256, error, args);
      va_end (args);
      GDK_THREADS_ENTER ();
      error_dialog = xmms_show_message ("Error", string, "Ok", FALSE, 0, 0);
      gtk_signal_connect (GTK_OBJECT (error_dialog),
                          "destroy",
                          GTK_SIGNAL_FUNC (gtk_widget_destroyed),
                          &error_dialog);
      GDK_THREADS_LEAVE ();
    }
#endif /* !NOGUI */
}

extern void xmmsmad_get_file_info (char *filename);
extern void xmmsmad_configure ();

/**
 * XMMS input plugin hook.
 */
InputPlugin *
get_iplugin_info (void)
{
  memset (&mad_plugin, 0, sizeof (InputPlugin));
  mad_plugin.description   = DESCRIPTION " " VERSION;
  mad_plugin.about         = xmmsmad_about;
  mad_plugin.init          = xmmsmad_init;
  mad_plugin.set_eq        = xmmsmad_set_eq;
  mad_plugin.configure     = xmmsmad_configure;
  mad_plugin.is_our_file   = xmmsmad_is_our_file;
  mad_plugin.play_file     = xmmsmad_play_file;
  mad_plugin.stop          = xmmsmad_stop;
  mad_plugin.pause         = xmmsmad_pause;
  mad_plugin.seek          = xmmsmad_seek;
  mad_plugin.get_time      = xmmsmad_get_time;
  mad_plugin.get_song_info = xmmsmad_get_song_info;
  mad_plugin.file_info_box = xmmsmad_get_file_info;
  mad_plugin.cleanup       = xmmsmad_cleanup;

  return &mad_plugin;
}
