/*
   SHARP CE-AG06 library based on OPIE camera sources
   Copyright 2005 Alexander Chukov <sash@pdaXrom.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <unistd.h>
#include <math.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "zcamera.h"

#define SHARPZDC "/dev/sharp_zdc"

#ifndef TRUE
#define TRUE	1
#endif

#ifndef FALSE
#define FALSE	0
#endif

static int _driver;
static char _status[4];
static int _pressed = FALSE;
static int _height = 0;
static int _width = 0;
static int _zoom = 0;
static int _flip = -1;
static int  _rot = 0;
static int _readlen = 0;

int cam_open(void)
{
    _driver = open( SHARPZDC, O_RDWR );
    if ( _driver == -1 )
        fprintf(stderr, "Can't open camera driver.\n");
    else
        cam_init();

    return _driver > 0;
}

int cam_close(void)
{
    if ( _driver != -1 ) {
        cam_setReadMode( 0 );
        close( _driver );
    }
    return 0;
}

int cam_read( char* buf, int len )
{
    int rlen = read( _driver, buf, len );
#ifdef DEBUG
    if ( rlen )
	fprintf(stderr, "read %d (%c%c%c%c) from driver.\n", rlen, buf[0], buf[1], buf[2], buf[3]);
    else
        fprintf(stderr, "read nothing from driver.\n");
#endif
    return rlen == len;
}

int cam_write( char* buf, int len )
{
    if ( !len )
        len = strlen( buf );
#ifdef DEBUG
    fprintf(stderr, "writing '%s' to driver.\n", buf);
#endif
    return write( _driver, buf, len ) == len;
}

void cam_init(void)
{
    cam_setReadMode( STATUS );
}

int cam_keyOn(void)
{
    int tmp;
    tmp = _pressed;
    if (tmp) {
        cam_clearShutterLatch();
        _status[0] = 's';
        _pressed = FALSE;
    }
    return tmp;
}

int cam_isShutterPressed(void)
{
    if ( _status[0] == 'S' )
    {
//        if ( !_pressed )    // wasn't pressed before, but is now!
//        {
            _pressed = TRUE;
            return TRUE;
//        } else {
//            cam_clearShutterLatch();
//            _status[0] = 's';
//            _pressed = FALSE;
//        }
    }
    
    return FALSE;
}

void cam_clearShutterLatch(void)
{
    cam_write( "B", 1 );
}

int cam_isFinderReversed(void)
{
    return _status[1] == 'M';
}


int cam_isCapturing(void)
{
    return _status[2] == 'C';
}


int cam_isAvailable(void)
{
    return _status[3] == 'A';
}

int cam_setCaptureFrame( int width, int height, int zoom, int rot )
{
    char b[100];
#ifdef DEBUG
    fprintf(stderr, "setCaptureFrame( %d, %d, %d, %d )\n", width, height, zoom, rot);
#endif
    sprintf( b, "%c=%d,%d,%d,%d", rot ? 'R':'S', width, height, zoom, width*2 );
    if ( cam_write( b , 0 ) )
    {
        _width = width;
        _height = height;
        _zoom = zoom;
        _rot = rot;
        _readlen = 2 * _width * _height; // camera is fixed @ 16 bits per pixel
        return TRUE;
    }
    fprintf(stderr, "couldn't write to driver.\n");
    return FALSE;
}

int cam_setZoom( int zoom )
{
    return cam_setCaptureFrame( _width, _height, zoom, _rot );
}

void cam_setReadMode( int mode )
{
    char b[10];
    sprintf( b, "M=%d", mode );
    cam_write( b, mode <= 9 ? 3 : 4 );
    if ( mode & STATUS ) // STATUS bit is set
    {
        cam_read( _status, 4 );
        if ( cam_isShutterPressed() ) {
        }
    }
}

void cam_setFlip( int flip )
{
    _flip = flip;
}

int cam_snapshot( unsigned char* buf )
{
    int readmode;

    cam_setReadMode( STATUS );
#ifdef DEBUG
    fprintf(stderr, "finder reversed = %d\n", cam_isFinderReversed());
    fprintf(stderr, "rotation = %d\n", _rot);
#endif
    if ( _flip == -1 ) // AUTO
    {
        if ( _rot ) // Portrait
        {
            readmode = IMAGE | cam_isFinderReversed() ? XFLIP | YFLIP : 0;
        }
        else // Landscape
        {
            readmode = IMAGE | XFLIP | YFLIP;
        }
    }
    else // OVERRIDE
    {
        readmode = IMAGE | _flip;
    }

    cam_setReadMode( readmode );
    return cam_read( (char*) buf, _readlen );
}

void cam_captureFrame( int w, int h, int zoom, unsigned char* buf )
{
    //FIXME: this is too slow
    int pw = _width;
    int ph = _height;
    cam_setCaptureFrame( w, h, zoom, _rot );
    cam_snapshot( buf );
    cam_setCaptureFrame( pw, ph, _zoom, _rot );
}
