/* PDB2PHNE converts the HP 200LX PDB database (phone book) in conjunction
 * with GDBIO so that the output can be transferred to an ERICSSON
 * mobile phone with WWW/LX, Robot/LX and the robot script "phone.scr".
 *
 * (c) 2002 by Daniel Hertrich http://www.daniel-hertrich.de/pdb2x
 *
 * needed: GDBIO (http://www.palmtop.net/super.html)
 *         Post/LX from D&A Software (http://www.dasoft.com)
 *
 * usage:  prepare phone book file with
 *
 *           GDBIO C:\_DAT\PHONE.PDB > PHONE.CDF
 *
 *         Then do a
 *
 *           GDBIO C:\_DAT\PHONE.PDB /S | more
 *
 *         This will show you the structure of your phone database.
 *         Determine the field numbers corresponding to the phone
 *         number field(s) you want to import into the phone.
 *         You can use up to 5 fields, they are copied into the Ericsson
 *         phone as the related phone book entry categpry.
 *         Also you need to know which field number
 *         contains the name for each entry. It will be 1 in most cases.
 *
 *         Use PHONE.CDF as input file for PDB2PHNE and give the
 *         field numbers you want to be scanned for name a numbers
 *         as arguments.
 *         If a scanned entry doesn't contain any number in one of
 *         the fields, the entry is skipped and PDB2PHN continues with
 *         the next one
 *
 *         So your output will only contain entries to which a number
 *         exists in one or more of the scanned fields.
 *
 *         PDB2PHNE needs 3 to 7 command line arguments:
 *         1. input file name              (mandatory)
 *         2. name column                  (mandatory)
 *         3. "Home" phone number column   (mandatory)
 *         4. "Work" phone number column   (optional)
 *         5. "Other" phone number column  (optional)
 *         6. "Mobile" phone number column (optional)
 *         7. "Fax" phone number column    (optional)
 *
 *         For each number field in which a phone number is found,
 *         a seperate line in the output file is created, this is what
 *         the Ericsson phones need, because one name with two assigned
 *         numbers needs indeed two entries in the phone book of the phone.
 *
 *         So, call for example
 *
 *         PDB2PHN PHONE.CDF 1 4 3 0 6 0 > phone.phn
 *
 *         Say, one PDB entry has the following fields:
 *
 *         Field  Name    Content
 *
 *         1      Name         Hertrich, Daniel
 *         2      email        daniel@somewhere.intheuniverse
 *         3      Work phone   +49 12 3456789
 *         4      Home phone   +49 98 7654321
 *         5      Fax          +49 11 1111111
 *         6      Mobile phone +49 77 7777777
 *         6      address      .....
 *
 *         PDB2PHNE would create the following Ericsson phone compatible
 *         output lines:
 *
 *         Hertrich, Daniel|+49987654321/H
 *         Hertrich, Daniel|+49123456789/W
 *         Hertrich, Daniel|+49777777777/M
 *         6      Mobile phone +49 77 7777777
 *         6      address      .....
 *
 *         PDB2PHNE would create the following Ericsson phone compatible
 *         output lines:
 *
 *         Hertrich, Daniel|+49987654321/H
 *         Hertrich, Daniel|+49123456789/W
 *         Hertrich, Daniel|+49777777777/M
 *
 *         Fax number would not be copied, because fax column numner
 *         is set to 0. The parameters after the phone mumber stand for:
 *         /H Home, /W Work, /O Other, /M Mobile, /F Fax.
 *         This is according to the standard Ericsson use for their phones.
 *
 *         Since you redirect the output into another file, you will end
 *         up with a file containing entries which can directly be sent
 *         to the phone using the robot/lx script PHONE.SCR, which comes
 *         with the Robot/LX package.
 *
 *         The PDB entry "Hertrich, Daniel" would take three entries in
 *         the Ericsson phone's phone book.
 *
 *         So take care the file containing the output lines doesn't get
 *         too large. For example, the Ericsson T39M can hold 500 phone
 *         book entries, so if your PDB has 200 entries, and each entry
 *         contains 3 phone mumbers, you would end up with a too large
 *         file (200 * 3 = 600 lines). The consider which number field is
 *         least important to have on the mobile phone, and leave it away.
 *         Then you will have a 400 lines long phone book, which fits into
 *         the T39m.
 *
 *
 */

#define VERSION "2.2"
#define YEAR "2003"
#define TOKLENGTH 80    // maximal length for a token, such as email, phone no, name of phone.cdf
#define LINELENGTH 1024 // maximal length of a line in cdf file
#define NAMELENGTH 17   // maximal allowed string length for the name in the
                        // phone book of the telephone (AT+CPBW=? (last number))

#include <stdlib.h>
#include <stdio.h>
#include <string.h>


FILE *fpi;
int namecol, prio1, prio2, prio3, prio4, prio5;
int linecounter=0;
int i,j;
char testchar;
char line[LINELENGTH];
char name[TOKLENGTH];
char num_home[TOKLENGTH];
char num_work[TOKLENGTH];
char num_other[TOKLENGTH];
char num_mobile[TOKLENGTH];
char num_fax[TOKLENGTH];
char shortnum[TOKLENGTH];
char *dup1;
char *dup2;
char *dup3;
char *dup4;
char *dup5;

void hello(char *filename, int nc, int p1, int p2, int p3, int p4, int p5);
void usage (void);
char *extr_cdf_col(char *cdfline, int c);
void process_file(int namecol, int p1, int p2, int p3, int p4, int p5);

/* MAIN ============================================== */

int main(int argc, char *argv[])
{
  if (argc < 4) usage();

  namecol = atoi(argv[2]);

  prio1 = atoi(argv[3]);

  prio2 = prio3 = prio4 = prio5 = 0;

  if (argc > 4) prio2 = atoi(argv[4]);
  if (argc > 5) prio3 = atoi(argv[5]);
  if (argc > 6) prio4 = atoi(argv[6]);
  if (argc > 7) prio5 = atoi(argv[7]);

  hello(argv[1], namecol, prio1, prio2, prio3, prio4, prio5);

  if ((fpi=fopen(argv[1],"r"))!=NULL)
  {

    process_file(namecol, prio1, prio2, prio3, prio4, prio5);

    if ((namecol == 0) && (prio1 == 0) && (prio2 == 0) && (prio3 == 0) && (prio4 == 0) && (prio5 == 0))
    {
      fprintf(stderr,"PDB2PHNE:\a\nNothing to do, no valid column for name or phone number given!\n\n");
    }

    fclose(fpi);
  }
  else
  {
    fprintf(stderr,"\n\aError opening input file \"%s\"!\nAbnormal program termination.\n\n",argv[1]);
    return(1);
  }
  return(0);
}



/* HELLO ============================================= */

void hello(char *filename, int nc, int p1, int p2, int p3, int p4, int p5)
{
  if (filename == NULL)
  {
    filename="NO INPUT FILE NAME GIVEN!";
  }
  fprintf(stderr,"\n*** This is PDB2PHNE, Version %s, (c) 2002 Daniel Hertrich\n",VERSION);
  fprintf(stderr,"*** Input CDF file: %s\n",filename);
  fprintf(stderr,"*** Field numbers: Name: %i, Home: %i, Work: %i, Other: %i, Mobile: %i, Fax: %i\n",nc,p1,p2,p3,p4,p5);
  fprintf(stderr,"*** Please wait while processing input file.....\n\n");
}

/* USAGE =========================================== */

void usage (void)
{
  printf("\n*** This is PDB2PHNE, Version %s, (c) %s Daniel Hertrich\n",VERSION,YEAR);
  printf("\a\n\nError!\nCall of PDB2PHNE: \nPDB2PHN <input CDF file> <name field> <H [W [O [M [F] ] ] ] > > <output PHN file>\n");
  printf("See pdb2phne.c source file for details\n\n");
  exit(1);
}


/* EXTR_CDF_COL ===================================== */
/* give parameters:
      1. copy of a line of a cdf file (made by gdbio)
      2. number of column you want to extract
   get back:
      pointer to string containing string of given column
*/

char *extr_cdf_col(char *cdfline, int c)
{
  static char token[TOKLENGTH];
  char *loc;
  char *temptok = malloc(LINELENGTH*sizeof(char));
  int i = 0;

  temptok[0]=0;

  for (i=0; i<c; i++)
  {
    if (i>0) cdfline=loc+3;
    loc=strstr(cdfline,"\",\"");
    if (loc != NULL)
    {
      strncpy(temptok,cdfline,loc-cdfline);
      temptok[loc-cdfline]='\0';
    }
    else
    {
      fprintf(stderr,"ERROR! Contact Author! loc is NULL!, c is %i\nCDFLINE is: %s\nLine number: %i\n\n\a\a",c,cdfline,linecounter);
      exit(1);
    }
  }
  strcpy(token,temptok);
  free(temptok);
  return(token);
}


/* PROCESS_FILE ========================================= */

void process_file(int nc, int p1, int p2, int p3, int p4, int p5)
{
  int outlines = 0;

  while((testchar=fgetc(fpi))!= EOF)
  {
    linecounter++;
    fgets(line, LINELENGTH, fpi);
    dup1=strdup(line);
    dup2=strdup(line);
    dup3=strdup(line);
    dup4=strdup(line);
    dup5=strdup(line);
    strncpy(name,extr_cdf_col(line,nc),NAMELENGTH);


    if (p1 != 0)
    {
      strcpy(num_home,extr_cdf_col(dup1,p1));
      if (strlen(num_home) > 0)  /* wenn Nummer vorhanden */
      {
        j = 0;
        strcpy(shortnum,"");
        for (i=0;i<strlen(num_home);i++)
        {
          if ((num_home[i]!=' ') && (num_home[i]!='\t') && (num_home[i]!='-') && (num_home[i]!='/') && (num_home[i]!='.'))
  /* ersetzen durch Audruck "wenn nicht nummer oder +" */
          {
            shortnum[j] = num_home[i];
            j++;
            shortnum[j] = '\0';
          }
        }
        printf("%s|%s/H\n",shortnum,name);
        outlines++;
      }
    }
    free(dup1);


    if (p2 != 0)
    {
      strcpy(num_work,extr_cdf_col(dup2,p2));
      if (strlen(num_work) > 0)  /* wenn Nummer vorhanden */
      {
        j = 0;
        strcpy(shortnum,"");
        for (i=0;i<strlen(num_work);i++)
        {
          if ((num_work[i]!=' ') && (num_work[i]!='\t') && (num_work[i]!='-') && (num_work[i]!='/') && (num_work[i]!='.'))
  /* ersetzen durch Audruck "wenn nicht nummer oder +" */
          {
            shortnum[j] = num_work[i];
            j++;
            shortnum[j] = '\0';
          }
        }
        printf("%s|%s/W\n",shortnum,name);
        outlines++;
      }
    }
    free(dup2);


    if (p3 != 0)
    {
      strcpy(num_other,extr_cdf_col(dup3,p3));
      if (strlen(num_other) > 0)  /* wenn Nummer vorhanden */
      {
        j = 0;
        strcpy(shortnum,"");
        for (i=0;i<strlen(num_other);i++)
        {
          if ((num_other[i]!=' ') && (num_other[i]!='\t') && (num_other[i]!='-') && (num_other[i]!='/') && (num_other[i]!='.'))
  /* ersetzen durch Audruck "wenn nicht nummer oder +" */
          {
            shortnum[j] = num_other[i];
            j++;
            shortnum[j] = '\0';
          }
        }
        printf("%s|%s/O\n",shortnum,name);
        outlines++;
      }
    }
    free(dup3);


    if (p4 != 0)
    {
      strcpy(num_mobile,extr_cdf_col(dup4,p4));
      if (strlen(num_mobile) > 0)  /* wenn Nummer vorhanden */
      {
        j = 0;
        strcpy(shortnum,"");
        for (i=0;i<strlen(num_mobile);i++)
        {
          if ((num_mobile[i]!=' ') && (num_mobile[i]!='\t') && (num_mobile[i]!='-') && (num_mobile[i]!='/') && (num_mobile[i]!='.'))
  /* ersetzen durch Audruck "wenn nicht nummer oder +" */
          {
            shortnum[j] = num_mobile[i];
            j++;
            shortnum[j] = '\0';
          }
        }
        printf("%s|%s/M\n",shortnum,name);
        outlines++;
      }
    }
    free(dup4);


    if (p5 != 0)
    {
      strcpy(num_fax,extr_cdf_col(dup5,p5));
      if (strlen(num_fax) > 0)  /* wenn Nummer vorhanden */
      {
        j = 0;
        strcpy(shortnum,"");
        for (i=0;i<strlen(num_fax);i++)
        {
          if ((num_fax[i]!=' ') && (num_fax[i]!='\t') && (num_fax[i]!='-') && (num_fax[i]!='/') && (num_fax[i]!='.'))
  /* ersetzen durch Audruck "wenn nicht nummer oder +" */
          {
            shortnum[j] = num_fax[i];
            j++;
            shortnum[j] = '\0';
          }
        }
        printf("%s|%s/F\n",shortnum,name);
        outlines++;
      }
    }
    free(dup5);
  }
  fprintf(stderr,"Your mobile phone's phone book needs a capacity of\n%i entries to hold all the numbers!\n\n",outlines);
}



